/* set ts=4 */
#include <stdlib.h>
#include "_uexec.h"

/* Micro-Executive
 * Copyright (c) 1998 ImageCraft and Richard F. Man
 * http://www.imagecraft.com
 */
static TaskControlBlock free_list[NUM_TASKS], *free_list_ptr;
static TaskControlBlock *current_task;
unsigned char *uexc_current_sp;
void (*uexc_current_func)(void);
void (*UEXC_InterruptDriver)(void);

static int next_tid;
static unsigned char null_task_stack[UEXC_MIN_STACK_SIZE];
static void InitSystem(void);
static void NullTask(void);
static void KillTask(int tid);

int uexc_ticks;

/* User Callable  Functions
 * ========================
 */

/* Start the multi-tasking system
 */
int UEXC_StartScheduler(void)
	{
	if (current_task == 0)
		return -1;
	UEXC_StartTimer();
	INTR_ON();
	UEXC_Schedule();
	return 0;	/* never execute */
	}

/* Create a new process and link it to the task list
 */
int UEXC_CreateTask(void (*func)(void), 
                    unsigned char *stack_start,
                    unsigned stack_size,
                    unsigned ticks)
	{
	TaskControlBlock *p, *next;

	if (free_list_ptr == 0)
		InitSystem();
	INTR_OFF();
	p = free_list_ptr;
	free_list_ptr = free_list_ptr->next;
	p->func = func;
	p->state = T_CREATED;
	p->tid = next_tid++;
	p->ticks = (ticks == 0) ? DEFAULT_TICKS : ticks;

	/* stack grows from high address to low address
	 */
	p->stack_start = stack_start;
	p->stack_end = stack_start+stack_size-1;

	/* align stack here, if needed
	 */
	p->sp = p->stack_end;

	/* create a circular linked list
	 */
	if (current_task == 0)
		p->next = p, current_task = p;
	else
		next = current_task->next, current_task->next = p, p->next = next;
	INTR_ON();
	return p->tid;
	}

/* Give up timeslice
 */
void UEXC_Defer(void)
	{
	INTR_OFF();
	uexc_current_sp = (unsigned char *)&current_task->sp;
	UEXC_SavregsAndResched();
	}

/* Hog the processor for more cycle time
 */
void UEXC_HogProcessor(void)
	{
	INTR_OFF();
	current_task->current_ticks = current_task->ticks;
	INTR_ON();
	}

/* kill a process
 */
void UEXC_KillTask(int tid)
	{
	TaskControlBlock **qb, *p;

	/* tid == 0 -> null task, never kill that 
	 */
	if (tid == 0)
		return;

	INTR_OFF();
	/* search the circular list until we find a match or gone through
	 * the whole list
	 */
	for (qb = &(current_task)->next; (*qb)->tid != tid && *qb != current_task;
	     qb = &(*qb)->next)
		;
	/* if found, then delete the process
	 */
	if ((*qb)->tid == tid)
		{
		/* If this is the head of the task list, reassign the
		 * head. This has the effect of skipping the next task
		 * for one round of scheduling but should not matter
		 * much.
		 */
		if ((*qb) == current_task)
			current_task = current_task->next;

		/* delink and free the memory
		 */
		p = *qb;
		*qb = (*qb)->next;
		p->next = free_list_ptr;
		free_list_ptr = p;
		}
	INTR_ON();
	}

/* Internal Functions
 * ==================
 */

/* Initialize the system.
 */
static void InitSystem(void)
	{
	int i;

	/* initialize the free list
	 */
	for (i = 0; i < NUM_TASKS-1; i++)
		free_list[i].next = &free_list[i+1];
	free_list_ptr = &free_list[0];

	/* null task has tid of 0 
	 */
	UEXC_CreateTask(NullTask, null_task_stack, sizeof (null_task_stack), 0);
	}

/* This is the "return point" of a new task. Reclaim storage
 * and reschdule
 */
void UEXC_KillSelf(void)
	{
	UEXC_KillTask(current_task->tid);
	UEXC_Schedule();
	}

/* Always runnable task. This has the tid of zero
 */
static void NullTask(void)
	{
	while (1)
		UEXC_Defer();
	}

/* called by the timer interrupt. See if we need to reschedule
 */
void UEXC_CheckTask(void)
	{
	uexc_ticks++;

	/* call any user installed interrupt routine
	 */
	if (UEXC_InterruptDriver)
		(*UEXC_InterruptDriver)();

	UEXC_RestartTimer();
	/* Kill the task if the stack pointer is bad.
	 * There ought to be a way to inform the user...
	 */
	if (uexc_current_sp > current_task->stack_end ||
	    uexc_current_sp < current_task->stack_start)
		UEXC_KillSelf();

	/* decrement the tick value and reschedule if needed
	 */
	if (--current_task->current_ticks == 0)
		{
		/* memorize the state
		 */
		current_task->sp = uexc_current_sp;
		UEXC_Schedule();
		}
	}

/* Run a different task. Set the current task as the next one in the (circular)
 * list, then set the global variables and call the appropriate asm routines
 * to do the job.
 */
void UEXC_Schedule(void)
	{
	/* pick the next task in the list
	 */
	current_task = current_task->next;
	current_task->current_ticks = current_task->ticks;

	/* set the global variables and go
	 */
	uexc_current_sp = current_task->sp;
	if (current_task->state == T_READY)
		UEXC_Resume();
	else /* if (task->state == T_CREATED) */
		{
		current_task->state = T_READY;
		uexc_current_func = current_task->func;
		UEXC_StartNewTask();
		}
	}
