;E128dem2.ASM
**********************************************************************************
*Description:		Demo program for Adapt9S12E128 demonstrating DAC operation
*Use:			Connect to a terminal program at 9600 baud; monitor voltage on DAC pins (H1-14,15)
*Author:		Carl Barnes
*Target:		Adapt9S12E128
*Assembly syntax:	MiniIDE  (available from www.mgtek.com/miniide)
*
*REVISION HISTORY:
*
*DATE                   REV. NO.        DESCRIPTION
*====			========	===========
*May 22, 2014		1.00		Initial release
*
***********************************************************************************

#include "S12E128Regs.INC"	;original E128Regs.inc has errors with DAC register definitions!

*****************************************************************************************
*Operational Parameters
*****************************************************************************************
RAM		equ	$2000		;9S12E128 internal RAM
STACK		equ	$4000		;Stackpointer points one past end of RAM
FLASH		equ	$4000		;Flash start address
FIXEDPAGE	equ	$E000		;addresses in last page of flash are absolute, so ISRs go there

;System Clock
OscFreq		equ	8000		;Oscillator frequency
initSYNR	equ	$02		; mult by synr + 1 = 3 (24MHz)
initREFDV	equ	$00		;
PLLSEL		equ	%10000000	;PLL select bit
LOCK		equ	%00001000	;lock status bit
PLLON		equ	%01000000	;phase locked loop on bit

;ATD Variables
admask2         equ     %11000000       ;AFFC,ADPU=1 - Enable Analog to Digital
admask3         equ     %00000000       ;FRZ1,FRZ0=0
admask4         equ     %10000001       ;SMP1,SMP0 = 0; S10BM,PRS0=1 - Select Sample time adn Bit mode
admask5         equ     %01110000       ;S8CM = 1, SCAN = 1, MULT = 1
SCFflag         equ     %10000000       ;SCF - Sequence Complete flag

;RTI Variables
clrmask         equ     %11000000       ;mask for clearing timer flags
RTIRate		equ     %01110000       ;
RTIF		equ	%10000000
RTIE		equ	%10000000

;SCI Variables
scimask         equ     %00101100       ;RIE - SCI Interrupt enable
                                        ;RE - Receiver Enable
RDRFflag        equ     %00100000       ;RDRF - Receive Data Register Full flag
TDREflag        equ     %10000000       ;TDRE - Transmit Data Register Empty flag

;Baud rate definitions
BusFreq		equ	((OscFreq/(initREFDV+1))*(initSYNR+1))
baud115200	equ	(BusFreq/16)*10/1152  			;sets baud rate to 115,200
baud9600	equ	(BusFreq/16)*10/96  			;sets baud rate to 009,600
initSCI0CR2	equ	$0C					;SCI0 Control Register 2

;Port Masks

PUPAE		equ	%00000001	;masks for Port Pullup Enable registers
PUPBE		equ	%00000010
PUPEE		equ	%00010000
PUPKE		equ	%10000000

LED		equ	%00000001	;Port P bit 0 connected to LED D1

*****************************************************************************************
* Operational Constants
*****************************************************************************************

TRUE            equ     $FF
FALSE           equ     $00
CR              equ     $D
LF              equ     $A
SPACE           equ     $20


*****************************************************************************************
* System Variables
*****************************************************************************************
		org	RAM


COMMAND		ds	1		;used by ProcessCommand
COMMAND_PENDING	ds	1		;used by ProcessCommand
CONTINUOUS	ds	1		;flag for real-time a/d screen update

state		ds	2
state_timer	ds	2

H		ds	1		;used in binary to decimal conversion
TO		ds	1

ad_flag		ds	1
ad_val		ds	2
DBUFR		ds	5
d_flag		ds	1

temp		ds	1

;atd_var

ad0		ds	2
ad1		ds	2
ad2		ds	2
ad3		ds	2
ad4		ds	2
ad5		ds	2
ad6		ds	2
ad7		ds	2
adflag		ds	1

TIMER		ds	2       ;software timers using RTI
TIMER1		ds	2
COUNTER		ds	1
delay_timer	ds	2


***************************** Program *******************************

		org	Flash

ResetFunc				;This is where the RESET vector points to
	sei				;mask interrupts

;Initialize Stack
	lds	#STACK			;initialize stack pointer
	
; Initialize clock generator and PLL for 24MHz operation
	bclr	CLKSEL,PLLSEL      	;disengage PLL from system
	bset	PLLCTL,PLLON       	;activate PLL

	movb	#initSYNR,SYNR     	;set PLL multiplier 
	movb	#initREFDV,REFDV   	;set PLL divider
	nop
	nop
	brclr	CRGFLG,LOCK,*+0    	;while (!(crg.crgflg.bit.lock==1))
	bset	CLKSEL,PLLSEL      	;once locked, engage PLL to system

;setup the ports and hardware subsystems that we'll be using
	bset	PUCR,PUPAE+PUPBE+PUPEE+PUPKE	;Enable all pull up's
	movb	#$FF,PERT
	movb	#$FF,PERS
	movb	#$FF,PERM
	movb	#$FF,PERP
	movb	#$FF,PERQ
	movb	#$FF,PERU
;	movb	#$FF,PERADH		;optional pullups for A-to-D port pins
;	movb	#$FF,PERADL

	bset	DDRP,#$01		;make PP0 an output to drive LED D1
	bset	DDRM,#$03		;make DAC channels outputs

;Initialize DACs
	movb	#$89,DAC0C0		;enable DACs for right-justified data
	movb	#$89,DAC1C0
	clr	DAC0DR			;ensure DAC right-justified data registers are 0
	clr	DAC1DR			; for 0 Volts output

;Initialize Real-time Interrupt
	movb	#RTIRate,RTICTL		;initialize RTI rate (RTI used for real-time analog display)
	bset	CRGFLG,RTIF		;clear flag
	bset	CRGINT,RTIE		;Enable RTI

;Initialize Analog To Digital Converter Subsystem
	movb	#$80,ATDCTL2	        ;enable ATD
	movb	#$40,ATDCTL3	        ;8 Channels of ATD results
	movb	#$60,ATDCTL4		;Select Sample rate
	movb	#$B0,ATDCTL5		;Select 8 channel mode, Continuous scan
	clr	adflag

;Initialize Main Serial Communication Interface
	movw	#baud9600,SCI0BDH	;Set baud rate to 9600
	movb	#scimask,SCI0CR2	;enable RX and TX
	ldab	SCI0SR1  		;read register to clear RDRF
	ldab	SCI0DRL			;dummy read to flush the receive buffer

	clr	COMMAND_PENDING
	clr	CONTINUOUS

	cli				;unmask interrupts

	jsr	ShowMenu		;generate user menu display via serial port

	bset	PORTP,#LED		;flash the LED once
	jsr	delay
	bclr	PORTP,#LED
	jsr	delay




;------------------------------------------------------------------------------
main					;Main Loop
	jsr	ProcessCommand		;this is the program loop

	ldaa	CONTINUOUS		;if in continuous Real-time analog update mode then show values
	beq	NotCont
	jsr	ShowAnalog

NotCont
	bra	main

*******************************************************************************
*  Subroutine Definitions
*******************************************************************************

;-----------------------------
;Commands are received via SCI interrupt routine, setting COMMAND_PENDING flag

ProcessCommand
	ldaa	COMMAND_PENDING		;if there is a command pending, process it
	lbeq	PCX

	clr	COMMAND_PENDING		;first, reset the flags
	clr	CONTINUOUS

	ldaa	COMMAND		;save a copy of the command
	staa	temp

	cmpa	#'+'
	lbeq	IncDACs
	cmpa	#'-'
	lbeq	DecDACs

	anda	#$df            ;simple convert to upper case (only works for alpha char)

	cmpa	#'A'		;display PortA
	beq	PCA

	cmpa	#'B'		;display PortB
	beq	PCB

	cmpa	#'C'		;clear output ports
	lbeq	PCC

	cmpa	#'D'		;enable DACs to mirror ADCs (o and 1)
	lbeq	PCD

	cmpa	#'F'		;flash LED on PP0
	lbeq	PCF

	cmpa	#'M'		;display PortM
	lbeq	PCM

	cmpa	#'Q'		;display PortQ
	beq	PCQ

	cmpa	#'R'		;display Real-time Analog port values continuously
	beq	PXR

	cmpa	#'T'		;display PortT
	beq	PCT

	cmpa	#'U'		;display PortU
	beq	PCU

	ldaa	temp		;toggle the corresponding PortP bit
	cmpa	#'0'
	lblo	PPCMenu
	cmpa	#'5'
	lbhi	PPCMenu

	jsr	TogglePLine
	rts

;----------------------------
;X register points to routine
;----------------------------

PCA
        ldx     #PPCA
        jsr     0,x             ;go execute routine
        bra     PCX
PCB
        ldx     #PPCB
        jsr     0,x             ;go execute routine
        bra     PCX

PCC
        ldx     #PPCC
        jsr     0,x             ;go execute routine
        bra     PCX

PCD
	ldx	#PPCD		;go execute routine
	jsr	0,x
	bra	PCX

PCF
        ldx     #PPCF
        jsr     0,x             ;go execute routine
        bra     PCX

PCM
        ldx     #PPCM
        jsr     0,x             ;go execute routine
        bra     PCX

PCQ
        ldx     #PPCQ
        jsr     0,x             ;go execute routine
        bra     PCX

PCT
        ldx     #PPCT
        jsr     0,x             ;go execute routine
        bra     PCX

PCU
        ldx     #PPCU
        jsr     0,x             ;go execute routine
        bra     PCX

PXR
        ldx     #PPCR
        jsr     0,x             ;go execute routine
        bra     PCX

IncDACs
	inc	DAC0DR
	inc	DAC1DR
	bra	PCX

DecDACs
	dec	DAC0DR
	dec	DAC1DR
;	bra	PCX

PCX
PCXX
        RTS
;----------------------------------------------

PPCA
	ldx	#CrlfStr
	jsr	OutStr0	

        ldx     #M_PORTA
        jsr     OutStr0

        ldab    PORTA
        clra
        jsr     SendDecByte	;Display PORTA value

	ldx	#CrlfStr
	jsr	OutStr0

	lbra	PCXX

PPCB
	ldx	#CrlfStr
	jsr	OutStr0	

        ldx     #M_PORTB
        jsr     OutStr0

        ldab    PORTB
        clra
        jsr     SendDecByte	;Display PORTB value

	ldx	#CrlfStr
	jsr	OutStr0

	lbra	PCXX

PPCC
	ldx	#CrlfStr
	jsr	OutStr0	

	clr	PORTP		;clear all Port P outputs
	clr	DAC0DR		;clear DAC data registers to 0
	clr	DAC1DR
	ldx	#M_POFF
	jsr	OutStr0

	lbra	PPCX

PPCD
	ldx	#CrlfStr
	jsr	OutStr0	

	ldd	ad0		;mirror most significant 8 bits of analog channel 0 to DAC0
	lsrd
	lsrd
	stab	DAC0DR

        clra
        jsr     SendDecByte
	ldx	#CrlfStr
	jsr	OutStr0

	ldd	ad1		;mirror most significant 8 bits of analog channel 1 to DAC1
	lsrd
	lsrd
	stab	DAC1DR

        clra
        jsr     SendDecByte
	ldx	#CrlfStr
	jsr	OutStr0
	ldx	#M_DAC
	jsr	OutStr0

	lbra	PPCX

PPCF
        ldx     #M_BLINK
        jsr     OutStr0

	ldx	#CrlfStr
	jsr	OutStr0	

	bset	PORTP,#LED	;flash the LED once
	jsr	delay
	bclr	PORTP,#LED
	jsr	delay

	lbra	PCXX

PPCM
	ldx	#CrlfStr
	jsr	OutStr0

        ldx     #M_PORTM
        jsr     OutStr0

        ldab    PTIM		;display the value of PORTM inputs
        clra
        jsr     SendDecByte

	ldx	#CrlfStr
	jsr	OutStr0

	lbra	PCXX

PPCQ
	ldx	#CrlfStr
	jsr	OutStr0	

        ldx     #M_PORTQ	;display the value of PORTQ inputs
        jsr     OutStr0

        ldab    PTIQ
        clra
        jsr     SendDecByte

	ldx	#CrlfStr
	jsr	OutStr0

	bra	PPCX

PPCT
	ldx	#CrlfStr
	jsr	OutStr0	

        ldx     #M_PORTT
        jsr     OutStr0

        ldab    PTIT		;display the value of PORTT inputs
        clra
        jsr     SendDecByte

	ldx	#CrlfStr
	jsr	OutStr0

	bra	PPCX

PPCU
	ldx	#CrlfStr
	jsr	OutStr0	

        ldx     #M_PORTU
        jsr     OutStr0

        ldab    PTIU		;display the value of PORTU inputs
        clra
        jsr     SendDecByte

	ldx	#CrlfStr
	jsr	OutStr0

	bra	PPCX


PPCMenu
        jsr     ShowMenu
        bra     PPCX

PPCR
        staa    CONTINUOUS      ;set Continuous a/d display mode flag
        bra     PPCX

PPCX
        rts


;------------------------------------------------------------------------------
TogglePLine
        ldab    #1              ;number is between 0 and 5
        suba    #'0'            ;convert ASCII digit to binary number
        beq     TBL1

TBL0
        aslb                    ;set the appropriate mask bit
        dbne    a,TBL0

TBL1
        bitb    PORTP           ;if selected port bit is 1
        beq     TBM1            ;  then make it 0
        comb
        bra     TBM0

TBM1
        orab    PORTP
        stab    PORTP
        rts

TBM0
        andb    PORTP
        stab    PORTP
        rts

;------------------------------------------------------------------------------
;short delay
delay
	pshy
	ldy	#0

dly
	nop
	nop
	nop
	nop
	nop
	dbne	y,dly
	puly
	rts

;-----------------------------------------------------------------------------
ShowMenu
        ldx     #M_UNDLN
        jsr     OutStr0

Show_Prompt
        ldx     #M_PROMPT
        jsr     OutStr0
        rts

;-----------------------------------------------------------------------------
ShowAnalog
        inc     d_flag
        sei

        ldx     #M_AD0
        jsr     OutStr0
        ldx     #ad0
        jsr     HTOD

        ldaa    #SPACE
        jsr     SerOutput0

        ldx     #M_AD1
        jsr     OutStr0
        ldx     #ad1
        jsr     HTOD

        ldaa    #SPACE
        jsr     SerOutput0

        ldx     #M_AD2
        jsr     OutStr0
        ldx     #ad2
        jsr     HTOD

        ldaa    #SPACE
        jsr     SerOutput0

        ldx     #M_AD3
        jsr     OutStr0
        ldx     #ad3
        jsr     HTOD

        ldaa    #SPACE
        jsr     SerOutput0

        ldx     #M_AD4
        jsr     OutStr0
        ldx     #ad4
        jsr     HTOD

        ldaa    #SPACE
        jsr     SerOutput0

        ldx     #M_AD5
        jsr     OutStr0
        ldx     #ad5
        jsr     HTOD

        ldaa    #SPACE
        jsr     SerOutput0

        ldx     #M_AD6
        jsr     OutStr0
        ldx     #ad6
        jsr     HTOD

        ldaa    #SPACE
        jsr     SerOutput0

        ldx     #M_AD7
        jsr     OutStr0
        ldx     #ad7
        jsr     HTOD

        ldaa    #SPACE
        jsr     SerOutput0

        cli
        clr     d_flag

        ldaa    #CR
        jsr     SerOutput0
        rts

;------------------------------------------------------------------------------
HexByte2Dec

;               Supply hex value to be converted in B register
        clra
        ldx     #100    ;numerator in D, denominator in X
        idiv
;               remainder in D, quotient in X
;               B is lo-order byte
        xgdx
        stab    H       ;8 bit quotient in B
        xgdx
        ldx     #10     ;H=0X  TO=packed BCD
        idiv
        stab    TO
        xgdx
        aslb
        aslb
        aslb
        aslb
        orab    TO
        stab    TO
        rts

SendDecByte

; value is in B

        clra
        jsr     HexByte2Dec
        ldaa    H
        adda    #'0'
        jsr     SerOutput0
        ldaa    TO
        jsr     SendASCIIHex
        rts

SendASCIIHex
;value is in A

        psha
        anda    #$F0
        lsra
        lsra
        lsra
        lsra
        cmpa    #$A
        blt     SAH1
        suba    #$A
        adda    #'A'
        bra     SAH2
SAH1
        adda    #'0'
SAH2
        jsr     SerOutput0
        pula
        psha
        anda    #$0F
        cmpa    #$A
        blt     SAH3
        suba    #$A
        adda    #'A'
        bra     SAH4
SAH3
        adda    #'0'
SAH4
        jsr     SerOutput0
        pula
        rts

;-----------------------------------------------------------------------------
;HTOD-SUBROUTINE TO CONVERT A 16-BIT HEX NUMBER TO A 5 DIGIT DECIMAL
;This routine is taken from the HC11 manual.
;D=HEX VALUE TO BE CONVERTED
;X=ADDRESS WHERE THE NUMBER IS STORED TO BE CONVERTED

HTOD
        LDD     0,X             ;
	LDX     #10000
        IDIV                    ;
	XGDX
	ADDB    #$30
	STAB    DBUFR
	XGDX
	LDX     #1000
	IDIV
	XGDX
	ADDB    #$30
	STAB    DBUFR+1
	XGDX
	LDX     #100
	IDIV
	XGDX
	ADDB    #$30
	STAB    DBUFR+2
	XGDX
	LDX     #10
	IDIV
	ADDB    #$30
	STAB    DBUFR+4
	XGDX
	ADDB    #$30
	STAB    DBUFR+3
        LDX     #DBUFR+1        ;POINT AT DECIMAL
        BRA     P1K

P5DEC
        LDX     #DBUFR          ;POINT AT DECIMAL
	LDAA    #$30            ;CHECK FOR LEADING ZEROS
	CMPA    0,X             ;CHECK FOR 10,000S DIGIT

	BNE     P10K            ;START AT 10K DIGIT
        BSR     SKP1            ;INX AND PRINT A SPACE
	CMPA    0,X             ;CHECK FOR 1,000S

	BNE     P1K             ;START AT 1K DIGIT
        BSR     SKP1
        BSR     SKP1
	DEX
	CMPA    0,X             ;CHECK FOR 100S DIGIT

	BNE     P100            ;START AT 100 DIGIT
        BSR     SKP1
	CMPA    0,X             ;CHECK 10S DIGIT

	BNE     P10
        BSR     SKP1
	BRA     P1              ;START AT 1S DIGIT

P10K    LDAA    0,X             ;10,000 DIGIT
	PSHX
        jsr     SerOutput0
        PULX
	INX

P1K     LDAA    0,X
	PSHX
        jsr     SerOutput0
        PULX
	INX

P100    LDAA    0,X
	PSHX
        jsr     SerOutput0
	PULX
	INX

P10     LDAA    0,X
	PSHX
        jsr     SerOutput0
        PULX
	INX

P1      LDAA    0,X
        jsr     SerOutput0
        RTS

SKP1    PSHA
	INX
	LDAA    #$20
        jsr     SerOutput0
	PULA
	RTS

;------------------------------------------------------------------------------
OutStr0				; send a null terminated string to the display.
	ldaa	1,x+            ; get a character, advance pointer, null?
	beq	OutStrDone      ; yes. return.
	bsr	SerOutput0      ; no. send it out the SCI.
	bra	OutStr0          ; go get the next character.
OutStrDone
	rts

;------------------------------------------------------------------------------
SerOutput0
	brclr	SCI0SR1,TDREflag,SerOutput0     ;check if buffer is empty
	staa	SCI0DRL
	rts

*******************************************************************************
* User Messages
*******************************************************************************
M_UNDLN		dc.b	$D,$A,$D,$A
M_HEADER	dc.b	'ADAPT9S12E128 DEMO2 COMMAND MENU V1.00',$D,$A
		dc.b	'______________________________________',$D,$A
		dc.b	$D,$A
		dc.b	'A => SHOW PORT A STATUS',$D,$A
		dc.b	'B => SHOW PORT B STATUS',$D,$A

		dc.b	'C => CLEAR OUTPUT PORTS',$D,$A
		dc.b	'D => DAC0 and DAC1 MIRROR AD0 and AD1',$D,$A
		dc.b	'F => FLASH LED CONNECTED TO PP0',$D,$A

		dc.b	'M => SHOW PORT M STATUS',$D,$A
;		dc.b	'P => SHOW PORT P STATUS',$D,$A
		dc.b	'Q => SHOW PORT Q STATUS',$D,$A
		dc.b	'T => SHOW PORT T STATUS',$D,$A
		dc.b	'U => SHOW PORT U STATUS',$D,$A

		dc.b	'R => SHOW 10-bit REAL-TIME ANALOG VALUES',$D,$A
		dc.b	'0 TO 5 => TOGGLE THE SELECTED P PORT LINE',$D,$A,0

M_PROMPT	dc.b	'?',0

M_AD0		dc.b	'AD0=',0
M_AD1		dc.b	'AD1=',0
M_AD2		dc.b	'AD2=',0
M_AD3		dc.b	'AD3=',0
M_AD4		dc.b	'AD4=',0
M_AD5		dc.b	'AD5=',0
M_AD6		dc.b	'AD6=',0
M_AD7		dc.b	'AD7=',0

M_PORTA		dc.b	'PORTA=',0
M_PORTB		dc.b	'PORTB=',0
M_PORTS		dc.b	'Cleared Output Ports',$D,$A,0
M_DAC		dc.b	'DACs mirror AD0 and AD1',$D,$A,0

M_PORTM		dc.b	'PORTM=',0
M_PORTP		dc.b	'PORTP=',0
M_PORTQ		dc.b	'PORTQ=',0
M_PORTT		dc.b	'PORTT=',0
M_PORTU		dc.b	'PORTU=',0

M_POFF		dc.b	'OUTPUTS CLEARED',$D,$A,0
M_BLINK		dc.b	'>>> LED FLASH! <<<',$D,$A,0

CrlfStr		dc.b	$0a,$0d,$0

***********************************************************************
* Interrupt Handlers
***********************************************************************

* Real-time Interrupt Routine

RealTimeInt
	bset	CRGFLG,RTIF		;clear real-time interrupt flag
	cli				; so that other interrupts can be service

RTI6
	ldaa	adflag
	bne	rtiex

	brclr	ATDSTAT0,SCFflag,*	;Loop here until SCF of ATD is set
	bset	ATDSTAT0,SCFflag	;clear Flag for next conversion

	ldd	ATDDR0H
	std	ad0

	ldd	ATDDR1H
	std	ad1

	ldd	ATDDR2H
	std	ad2

	ldd	ATDDR3H
	std	ad3

	ldd	ATDDR4H
	std	ad4

	ldd	ATDDR5H
	std	ad5

	ldd	ATDDR6H
	std	ad6

	ldd	ATDDR7H
	std	ad7

rtiex
	rti


;==============================================================================
;  SCI Input Interrupt Handler

;  Gets bytes from SCI.  Sets COMMAND_PENDING flag.

SerInputInt0
	ldaa	SCI0SR1                  ;read register to clear flag RDRF
	movb	SCI0DRL,COMMAND          ;read receive buffer
	movb	#TRUE,COMMAND_PENDING

SIIX
	rti



********************************************************************************************
* Vector Table
********************************************************************************************
VECTOR	equ	$F780
	ORG	VECTOR

	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved

;	ORG	$FF88

	fdb	ResetFunc		;PWM Emergency Shutdown
	fdb	ResetFunc		;VREG LVI
	fdb	ResetFunc		;PMF Fault 3
	fdb	ResetFunc		;PMF Fault 2
	fdb	ResetFunc		;PMF Fault 1
	fdb	ResetFunc		;PMF Fault 0
	fdb	ResetFunc		;PMF Generator C Reload
	fdb	ResetFunc		;PMF Generator B Reload
	fdb	ResetFunc		;PMF Generator A Reload
	fdb	ResetFunc		;Pulse Accumulator 2 input edge
	fdb	ResetFunc		;Pulse Accumulator 2 overflow
	fdb	ResetFunc		;Standard Timer 2 Overflow
	fdb	ResetFunc		;Timer 2 Channel 7
	fdb	ResetFunc		;Timer 2 Channel 6
	fdb	ResetFunc		;Timer 2 Channel 5
	fdb	ResetFunc		;Timer 2 Channel 4
	fdb	ResetFunc		;Reserved

;	ORG	$FFAA

	fdb	ResetFunc		;Pulse Accumulator 1 input edge
	fdb	ResetFunc		;Pulse Accumulator 1 overflow
	fdb	ResetFunc		;Standard Timer 1 Overflow
	fdb	ResetFunc		;Timer 1 Channel 7
	fdb	ResetFunc		;Timer 1 Channel 6
	fdb	ResetFunc		;Timer 1 Channel 5
	fdb	ResetFunc		;Timer 1 Channel 4
	fdb	ResetFunc		;FLASH

	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved

;	ORG	$FFC0

	fdb	ResetFunc		;IIC Bus
	fdb	ResetFunc		;Reserved

;	ORG	$FFC4

	fdb	ResetFunc		;CRG Self Clock Mode
	fdb	ResetFunc		;CRG PLL lock
	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved

;	ORG	$FFCE

	fdb	ResetFunc		;Port AD (KWU)
	fdb	ResetFunc		;ATD
	fdb	ResetFunc		;SCI2
	fdb	ResetFunc		;SCI1
	fdb	SerInputInt0	;SCI0
	fdb	ResetFunc		;SPI
	fdb	ResetFunc		;Pulse Accumulator 0 input edge
	fdb	ResetFunc		;Pulse Accumulator 0 overflow
	fdb	ResetFunc		;Standard Timer 0 Overflow
	fdb	ResetFunc		;Timer 0 Channel 7
	fdb	ResetFunc		;Timer 0 Channel 6
	fdb	ResetFunc		;Timer 0 Channel 5
	fdb	ResetFunc		;Timer 0 Channel 4

	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved
	fdb	ResetFunc		;Reserved

;	ORG	$FFF0

	fdb	RealTimeInt		;Real Time Interrupt
	fdb	ResetFunc		;IRQ
	fdb	ResetFunc		;XIRQ
	fdb	ResetFunc		;SWI
	fdb	ResetFunc		;Instruction Trap
	fdb	ResetFunc		;COP failure
	fdb	ResetFunc		;Clock Monitor
	fdb	ResetFunc		;Power On Reset
