/*********************************************************************************************
*                                                                                            *
*    Nigel Johnson's Grand Universal Master Program (GUMP)                                   *
*                                                                                            *
*    Things this program MIGHT do well, as examples to use when you need to do these things: *
*                                                                                            *
*                                                                                            *
*    1) Show you how to access the LCD screen on the DEMH1 board in 4-bit mode               *
*                                                                                            *
*    2) Show you how to access the serial ports of the 9S12E, of which there are three       *
*                                                                                            *                                                 *
**********************************************************************************************
Edit history
                                                2014 04 23 CB edited comments
                                                2011 01 28 NWJ added beep
                                                2011 01 25 NWJ changes and adds 
                                                2009 10 29 NWJ original program
  This program does the following:
  
  Initialise the LCD screen
  Display a Welcome screen
  Produce a long beep on the speaker
  Create a random array of 32 bytes of data
  Output the data to the screen in hex format (each byte is two characters)
  Perform a bubble sort on the data, and every time it swaps two bytes it refreshes the screen
    and does a beep                                             
  Halts when the bubble sort is finished.
  
  Note:  To use the beep function, you need to connect external power to DEMH1
  
  This program also includes functions to access the serial port, but they have not been called
  from the main loop here.  They are included as examples.


  Use these pre-tested functions for your project
    
    The serial port functions are set up for serial port 2.  To change to another port, you
    just edit the names, i.e. SCI2BDH ===> SCI0BDH etc.
    
    Serial port 1 (SCI0)
       --- goes to the DE9S connector on the board
       
    Serial port 2 (SCI1)
       --- goes to the Molex header beside the DE9 - you can get a cable to go to a DE9 or it 
       goes to the Screw terminals if you have RS422 selected
       
    Serial port 3 (SCI2)
       --- goes to the IR transceiver and is available as TTL signals for interfacing to things 
       like bluetooth modems if you disable the IR transceiver using PQ5
    
    These may be useful for your project:
    
    LCD connections:
    
    The LCD is connected to the E128 via DEMH1 as follows:
    
    PS4 goes to LCD DB4
    PS5 goes to LCD DB5
    PS6 goes to LCD DB6
    PS7 goes to LCD DB7
    
    PP2 goes to LCD RS
    PP1 goes to LCD E
    PP0 goes to LCD R/-W
    
    CAUTION!  Do not mess with PS0 and PS1 as they are the RX and TX data leads respectively 
    for SCI0.  If you change these, you will disable the com port, and be unable to 
    connect to the serial monitor.

************************************************************************************************/
#include <hidef.h>          // common defines and macros
#include "derivative.h"     // derivative-specific definitions
#include "main_asm.h"       // interface to the assembly module
#include "stdlib.h"
#include "string.h"
#define debug
#define INIT 1              // arguments for the lcd routine
#define UPDATE 2            // 
#define CLEAROUT 3          // specially for lcds that have garbage after reset
#define RS 0x04             // these equates are to make referring to the LCD isgnals
#define E  0x02             // a little bit easier
#define RW 0x01 
#define SPKR 0x02           // bit on port M for the speaker (ext power required)
void lcd(int);
void initPorts(void);
void initSCI2(void);        // set up baud rate etc
void wait_LCD_Ready(void);
void delay(int);            // delay for 34.8 microseconds per count assuming 8MHz clock
void e(void);               // send the E pulse to the LCD
void out_Line(int); 
void sendSCI2(void);
void receiveSCI2(void);
void bubble(unsigned char a[],int n);
void swap (unsigned char *px,unsigned char *py);
void randomise(void);
void ascii(void);
void buzz(int);

const char line1[]="   Tech Arts    ";
const char line2[]=" Adapt9S12E128  ";
const char line3[]="Bubble Sort Demo";
const char line4[]="   for DEMH1    ";

char RAM_line1[16];
char RAM_line2[16];
char RAM_line3[16];
char RAM_line4[16];

unsigned char a[32];


char SCI2_RX_buf[20];        // where we put data received from SCI2 serial port, <cr> terminated
char SCI2_TX_buf[20]="ATI\r";// where we put output strings to send to SCI2 serial port, <cr> terminated

char *pointer;
char temp;
int char_Pos;               // where we are in LCD line
int dest;
int i;


void main(void) 
{
 EnableInterrupts;

 CLKSEL &= ~PLLCTL;          // turn off the PLL so MCU runs at the base speed
 _DISABLE_COP();             // this is not a safety critical app, therefore we don't need the hassle!


 delay(10000);               // seems to need a lot of time from power-up


 pointer=strncpy(&RAM_line1[0],&line1[0],16);  //Before we start, we copy the Welcome message from 
 pointer=strncpy(&RAM_line2[0],&line2[0],16);  // flash into the RAM buffer used for the LCD
 pointer=strncpy(&RAM_line3[0],&line3[0],16);
 pointer=strncpy(&RAM_line4[0],&line4[0],16);                          
 
randomise(); 
                            // from hardware reset, or run from CW
 for(;;) 
    {
    
    initPorts();    
    initSCI2();
    lcd(INIT);              // initialise the lcd for 4 line, 4 bit mode  
    delay(10000);            // this delay is way beyond spec but it seems to need it
    
    lcd(UPDATE);            // output 4 lines of text
    delay(10000);           // give the user a chance to read the Welcome message
    buzz(1000);
    while(1)
      {
      delay(10000);
      lcd(INIT);

      lcd(UPDATE);            // output 4 lines of text
      delay(10000);           
      bubble(&a[0],32);       // do a bubble sort
      ascii();                // copy the numbers and convert to ASCII
      asm "SWI";
      }                      // loop here to stall
    } // loop forever
}


/********************************************************************************
*
*
*       LCD DRIVER
*
*
*
*       All LCD operations done through this function.
*      
*                 Calling convention:
*                             
*                 Type = 1, initialise the lcd for 4 line mode, 4 bit transfers
*                 Type = 2, output 4 lines of text from buffer lline1,line2 etc
*
*********************************************************************************
 All accesses to the LCD must be done in pairs since we are in 4 bit mode and 
 only the top 4 bits of each write to Port S (PTS) are connected to the LCD and used
  
 so:  nothing can be in the lower nibble on Port S
 and: if we don't do it in pairs we will always be out by half a byte in following 
      commands since the LCD will be waiting for a second write cycle

***************************************************************************/

void lcd(int type)
{ 
    if (type==INIT) 
      {
      PTS = 0x00;          // 
      e();
      PTS = 0x10;          // Clear the display
      e();
      delay(5000); 
      /****************************************************************************/
      PTS = 0x00; 
      e();
      PTS = 0x20;          // set LCD for 4 bits
      e();
      delay(5000);
      /****************************************************************************/
      PTS = 0x00;          // clear out last
      e();
      PTS = 0x80;          // function set
      e();
      delay(5000);
      /****************************************************************************/
      PTS = 0x00;          // clear out last
      e();
      PTS = 0xE0;          // display on, etc
      e();
      delay(5000);
      /****************************************************************************/
      PTS = 0x00;          // clear out last
      e();
      PTS = 0x06;          // entry mode set
      e();
      delay(5000);
      /****************************************************************************/
      } 
  
    else if (type ==UPDATE)
      {
      out_Line(1);    
      out_Line(2);    
      out_Line(3);    
      out_Line(4);    
      }
      return;
}

/*********************************************************************************/
void sendSCI2(void) 
{
  pointer = &SCI2_TX_buf[0];

  while (char_Pos<20)
  {
    while (_SCI2SR1.Bits.TDRE)  // send data when transmit data buffer empty
    {
      SCI2DRL = pointer[char_Pos];
      if (pointer[char_Pos] == '\r')
        break;
      char_Pos++;
    }
  }
}
/*********************************************************************************/
void receiveSCI2(void) 
{

}
  
/*********************************************************************************/
void initSCI2()  
{
 SCI2BDH = 0x00;
 SCI2BDL = 0x1A;
 SCI2CR1 = 0x00;
 SCI2CR2 = 0x0C;            // TX and RX enabled
 
}
/*********************************************************************************/
void initPorts()
{
  
  DDRU |=0xFF;              // configure the ports for output
  DDRS |=0xFC;              // lower 2 bits are for SCI 0
  DDRP |=0xFF;   
  DDRM |=0x02;               // we are going to output to the audio transducer
}
/*********************************************************************************/ 
void wait_LCD_Ready()
{
 
 PTP &= ~RS;              // RS low selects control reg
 PTP |= RW;               // R/-W high selects READ mode
 PTP |= E;                // raise E pulse
  
 DDRS &= ~0x80;             // switch bit 7 to READ mode
 
 while ((PTS & 0x80)==0x80) {};
     
 DDRS |= 0x80;              // set bit 7 back to output
 PTP &=~E;               // drop the E pulse down again

 PTP &= ~RW;              // set R/-W low to go back to writing
 PTP |= RS;               // return to WRITE mode
}
/*********************************************************************************/
void e(void)                // send the E pulse to the lcd
{
  PTP |= E;
  delay(50);
  PTP &= ~E;
  delay(50);
  return;
}
/*********************************************************************************/       	
void delay(int microsecs)
{
 int i;
 for (i=1;i<microsecs;i++) {};
}
/*********************************************************************************/
void out_Line(int line)
{
  switch (line) 
  {
  case 1:
    PTP &= ~RS;           // access control register
    PTS &= 0x03;            // clear all bits other than 
    PTS |= 0x80;
    e();                    // 0 and 1 to select line 1
    PTS &= 0x03;            // start position on line
    e();
    PTP |=RS;             // access data register
    pointer=&RAM_line1[0];
    goto common;
    
  case 2:
    PTP &= ~RS;           // access control register
    PTS &= 0x03;            // clear all bits other than 
    PTS |= 0xC0;            // select line 2
    e();
    PTS &= 0x03;            // start position on line
    e();
    PTP |=RS;             // access data register
    pointer=&RAM_line2[0];
    goto common;
  
  case 3:
    PTP &= ~RS;           // access control register
    PTS &= 0x03;            // clear out last
    PTS |= 0x94;            // select line 3
    e();
    PTS &=0x03;             // start position on line
    e();
    PTP |=RS;             // access data register
    pointer=&RAM_line3[0];
    goto common;
    
  case 4:
    PTP &= ~RS;           // access control register
    PTS &= 0x03;            // clear out last
    PTS |= 0xD4;            // select line 4
    e();
    PTS &= 0x03;            // start position on line
    e();
    PTP |=RS;             // access data register
    pointer=&RAM_line4[0];
    
  common:
    for (char_Pos=0;char_Pos<16;char_Pos++)
    {
      temp = pointer[char_Pos];
      PTS = temp;
      e();
      temp <<=4;
      PTS = temp;
      e();
    }
  }
}  
/*********************************************************************************/
void bubble(unsigned char a[],int n) 
{
  int i,j;
  for (i=0;i<n-1;i++)
    for (j=0;j<n-1;j++)
      if (a[j]>a[j+1])
        swap (&a[j],&a[j+1]);

}      
/*********************************************************************************/
void swap (unsigned char *px, unsigned char *py) 
{
  unsigned char temp;
  temp = *px;
  *px = *py;
  *py = temp;
  buzz(200);
  ascii();
}
/*********************************************************************************/
void randomise(void)
{
  for (i=0; i<32; i++)        // set up some random numbers to sort
   a[i] = rand();  
}
/*********************************************************************************/
void ascii(void)              // this function takes the binary values in the 32 bytes
                              // of a[], converts each byte into two right-justified 
                              // nibbles, converts them into ASCII, and stores them
                              // in the 4 x 16 LCD buffer
{
  for(i=0;i<32;i++)
    {
    unsigned char temp=a[i];
   
    temp >>=4;                // move temp1 left a nibble  
    temp +=0x30;
      if (temp>0x39)
        temp+=7;
    RAM_line1[2*i]=temp;
  
    temp=a[i];                // get two copies of the byte
    temp &=0x0F;              // get rid of upper nibble of temp2
    temp +=0x30;
      if (temp>0x39)
        temp+=7;
    RAM_line1[2*i+1]=temp;
    delay(1000);              // every time we change the buffer
    lcd(UPDATE);              // the delay is needed due to bad specs for the lcd
    }
}


void buzz(int count)
  {
  for (i=0; i< count;i++)
    {
    PTM |=SPKR;
    delay(281);               // produces a tone of about 1 kHz    
    PTM &= ~SPKR;
    delay(281);
    }
  
  
  }