
/*------------------------------------------------------------------------------
	lan91c111.c
		
	Author : Pierre Morency <pmorency@globetrotter.net>

	Created: 14.apr.2004
	Revised: 02.jun.2004 - Pre-release version
		     08.jun.2004 - Release V1.0
--------------------------------------------------------------------------------

 Copyright  2004, Technological Arts <www.technologicalarts.com>
 Copyright  2004, Pierre Morency
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of the copyright holders nor the names of their
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

------------------------------------------------------------------------------*/

#include "mc9s12e_regs.h"
#include "lan91c111.h"



/*------------------------------------------------------------------------------
  Default MAC address for this interface card
  
  This address is used only when the lan card is not equipped with an EEPROM
  storing an unique IEEE MAC address. The address below comes from an old
  retired ISA network card. If you need to change it, we suggest you 'recycle'
  a valid address not in use instead of 'improvising' a new one from scratch.  
------------------------------------------------------------------------------*/

static MAC_address mac = { 0, 0, 0xC0, 0x16, 0xF6, 0xB2 };



/*------------------------------------------------------------------------------
  Local private structure to access LAN91C111 banked registers.
  
  The LAN91C111 is designed as a little endian architecture device. The HC12
  being big endian, the MSB and the LSB of the 16-bit data bus are connected to
  the LAN91C111 in a swapped order. By doing this most configuration registers
  appear swapped to the CPU, but the CPU can move data to and from the data
  register without having to swap the bytes in software. This makes accessing
  most control registers a bit more difficult to program, but makes data
  transfers much faster.
  
  NOTES:

  1) all registers are defined as 16-bit entities (word) to speed chip access.
     (see Motorola documentation on 16-bit word alignment issues) 
  2) the bank select register (BSR) is always accessible in any bank.
  3) all registers having high byte and low byte swapped by hardware
     are identified with their name in capital letters.
  4) no bit modify instruction is used on the registers to avoid byte
     access to the LAN91C111.   
------------------------------------------------------------------------------*/

typedef struct
{
	volatile uint16 TCR;	/* Transmit Control Register */
	volatile uint16 EPHSR;	/* EPH Status Register */
	volatile uint16 RCR;	/* Receive Control Register */
	volatile uint16 ECR;	/* Counter Register */
	volatile uint16 MIR;	/* Memory Information Register */
	volatile uint16 RPCR;	/* Receive/PHY Control Register */
	volatile uint16 no_use;
	volatile uint16 BSR; 	/* Bank Select Register (common to all) */

} BANK0;

typedef struct
{
	volatile uint16 CR;		/* Configuration Register */
	volatile uint16 BAR;	/* Base Address Register */
	volatile uint16 IAR0;	/* Individual Address word 0 Register (LSB) */ 
	volatile uint16 IAR1;	/* Individual Address word 1 Register */ 
	volatile uint16 IAR2;	/* Individual Address word 2 Register (MSB) */ 
	volatile uint16 GPR;	/* General Purpose Register */ 
	volatile uint16 CTR;	/* Control Register */ 
	volatile uint16 BSR; 	/* Bank Select Register (common to all) */

} BANK1;

typedef struct
{
	volatile uint16 MMUCR; 	/* MMU Command Register */
	volatile uint16 pnr; 	/* Packet Number Register */
	volatile uint16 FIFO; 	/* FIFO Ports Registers */		
	volatile uint16 PTR; 	/* Pointer Register */
	volatile uint16 DATA;	/* 16-bit Data Register */
	volatile uint16 no_use;	/* high word of data register */	
	volatile uint16 ist; 	/* Interrupt Status Registers */
	volatile uint16 BSR; 	/* Bank Select Register (common to all) */

} BANK2;

typedef struct
{
	volatile uint16 MT01; 	/* Multicast Table 0 and 1 Registers */
	volatile uint16 MT23; 	/* Multicast Table 2 and 3 Registers */
	volatile uint16 MT45; 	/* Multicast Table 4 and 5 Registers */
	volatile uint16 MT67; 	/* Multicast Table 6 and 7 Registers */
	volatile uint16 MGMT; 	/* Management Interface */
	volatile uint16 REV; 	/* Revision Register */
	volatile uint16 ERCV; 	/* Early RCV Register */
	volatile uint16 BSR; 	/* Bank Select Register (common to all) */

} BANK3;

typedef union
{
	volatile BANK0 b0;
	volatile BANK1 b1;
	volatile BANK2 b2;
	volatile BANK3 b3;
	
} LAN91C;


/*------------------------------------------------------------------------------
  Private local variables
------------------------------------------------------------------------------*/

static LAN91C *nic = NULL;	/* use this pointer to access lan card registers */

static bool opened = false;

static void (*delay50ms)() = NULL;	
static uint16 tmp, tmpi;



/*------------------------------------------------------------------------------
  Prototypes of private local functions
------------------------------------------------------------------------------*/

static uint16 swap(uint16 val);

static   void write_Z(void);
static   void write_off(void);
static uint16 read_mii(uint8);  
static   void write_mii(uint8, uint16);
static   void write_frame(uint8, uint16);
static   void write_bits(uint8, uint16);

static uint16 read_via_gpio(uint16);
static   void write_via_gpio(uint16, uint16);
static   void idle_gpio(void);

static   bool lan_card_access(void);

static   void auto_negociate(uint8 retries);
static  uint8 *write_data(void *, uint16);
static   void read_data(void *, uint16);



/*------------------------------------------------------------------------------
  Swap high byte and low byte of a word
	
  This inline function is very efficient with zero overhead when compiler
  creates constants (by swapping MSB value with LSB value). Only 1 extra
  instruction (exg A,B) is used when compiler obtain value from D register.
------------------------------------------------------------------------------*/

static __inline__ uint16 swap(uint16 val)
{
	return (val << 8)|(val >> 8);
}



/*------------------------------------------------------------------------------
  These local functions provide tools to read and write to selected PHY MII
  registers. These registers are accessible via a serial interface controlled
  by 4 bits of the MGMT register of bank 3.
------------------------------------------------------------------------------*/

static void write_Z()
{
	write_off();
	nic->b3.MGMT = swap(0x3334);	/* MDOE = 0, MDO = 0, MCLK = 1 */
}


static void write_off()
{
	nic->b3.MGMT = swap(0x3330);	/* bring all control lines to 0 */
}


static uint16 read_mii(uint8 reg)
{
	uint16 data = 0;
	uint8 i;

	/* 14-bit value is made of start bits (01) + command bits (10 = read) 
	 + PHY address (5-bit = 00000 for LAN91C111) + PHY register to read (5-bit) */
	 
	write_frame(14, 0x6000 + (reg << 2));

	write_Z();							/* send turnaround bit */
	for(i = 0; i < 16; i++)
	{
		data <<= 1;
		write_Z();						/* toggle MCLK line */
		if(swap(nic->b3.MGMT) & 0x0002)	/* test MDI bit */
			data |= 0x0001;	
	}	

	write_Z();							/* send turnaround bit */
	write_off();
	return data;
}


static void write_mii(uint8 reg, uint16 data)
{
	/* 16-bit value is made of start bits (01) + command bits (01 = write) 
	  + PHY address (5-bit = 00000 for LAN91C111) + PHY register to write
	  (5-bit) + turnaround bits (10) for write */
	
	write_frame(16, 0x5002 + (reg << 2));
	write_bits(16, data);
	write_off();
}


static void write_frame(uint8 n, uint16 bits)
{
	nic->b3.BSR = swap(0x0003);		/* select bank 3 */

	write_bits(16, 0xFFFF);					
	write_bits(16, 0xFFFF);			/* write 32 consecutive 1's */
	write_bits(n, bits);			/* then frame data */
}


static void write_bits(uint8 n, uint16 bits)
{
	while(n--)
	{
		if(bits & 0x8000)				/* shift serially, MSB first */
		{
			nic->b3.MGMT = swap(0x3339);/* MDOE = 1, MDO = 1, MCLK = 0 */
			nic->b3.MGMT = swap(0x333D);/* MDOE = 1, MDO = 1, MCLK = 1 */
		}
		else
		{
			nic->b3.MGMT = swap(0x3338);/* MDOE = 1, MDO = 0, MCLK = 0 */
			nic->b3.MGMT = swap(0x333C);/* MDOE = 1, MDO = 0, MCLK = 1 */
		}
		
		bits <<= 1;
	}	
}


/*------------------------------------------------------------------------------
  Engage the auto-negociation process.
  Repeat 'retries' number of time if unsuccessful.
------------------------------------------------------------------------------*/

static void auto_negociate(uint8 retries)
{
	if(delay50ms != NULL)
	{
		while(retries)
		{
			write_mii(0, 0x3200);	/* clear MII_DIS and re-enable auto negotiation */
			retries--;

			/* wait 1.5 second maximum for the 
			   negociation process to complete */
			uint8 i;
			for(i = 0; i < 30; i++)
			{	
				(delay50ms)();
				if(read_mii(1) & 0x0004) /* check LINK bit of reg #1 */
				{
					retries = 0;
					break;
				}
			}			 
		}							
	}
}


/*------------------------------------------------------------------------------
  Local routines to read and write 16-bit values to LAN91C111 using port
  A and B as multiplexed address and data bus by manipulating control
  lines R/W, LSTRB, ECLK of port E and XCS of port K. 
	
  Notes:

  1) Must be called while in single chip mode.
  2) LSB and MSB of data bus are inverted to match hardware implementation.

------------------------------------------------------------------------------*/
#define ECLK  BIT4
#define LSTRB BIT3
#define RW    BIT2

static uint16 read_via_gpio(uint16 address)
{
	PORTB.byte = (uint8)address;		/* put address in data register */
	PORTA.byte = (uint8)(address >> 8);
	DDRAB.word = 0xFFFF;				/* put address on data bus  */
	PORTK.bit.XCS = 0;					/* latch address into 74LVC573's */
	
	DDRAB.word = 0;						/* put data bus in read mode */
	PORTE.byte = 0x14;					/* setting ECLK = 1, R/W = 1 causes RD* */
										/* to let 74LVX245's drive the data bus */
										
	uint16 value = PORTA.byte + (PORTB.byte << 8);	/* read 16-bit value */
	idle_gpio();						

	return value;
}


static void write_via_gpio(uint16 address, uint16 data)
{
	PORTB.byte = (uint8)address;		/* put address in data register */
	PORTA.byte = (uint8)(address >> 8);
	DDRAB.word = 0xFFFF;				/* put address on data bus  */
	PORTK.bit.XCS = 0;					/* latch address into 74LVC573's */

	PORTA.byte = (uint8)data;			/* put data on data bus */
	PORTB.byte = (uint8)(data >> 8);	/* with inverted MSB and LSB */
	PORTE.bit.ECLK = 1;					/* enables WR* signal */
	idle_gpio();						/* terminates */	
}


static void idle_gpio()
{
 	PORTK.bit.XCS  = 1;		/* XCS = 1 */
 	PORTE.byte = 0;			/* ECLK = 0, LSTRB = 0 (always), RW = 0 (write) */	
	DDRAB.word = 0;			/* leave data bus in input mode */
}


/*------------------------------------------------------------------------------
  Returns true if CPU is running in expanded mode and if the
  LAN91C111 has been properly initialized by lan_card_init()
------------------------------------------------------------------------------*/

static bool lan_card_access()
{
	return MODE.bit.MODA == 1 && MODE.bit.MODB == 1 && nic != NULL;
} 


/*------------------------------------------------------------------------------
  Public section
------------------------------------------------------------------------------*/

void lan_card_init(void (*delay)())
{
	if(MODE.bit.MODA == 0 && MODE.bit.MODB == 0)	/* do in single chip mode */ 
	{
		/* set PQ6 as output line and initiate an hardware reset */
		DDRQ.bit.BIT6 = 1;				
		PTQ.bit.BIT6 = 0;				

		/* Set port E bit 4, bit 3 and bit 2 as output to emulate ECLK, 
		   LSTRB and R/W respectively, set port K bit 6 to be used as XCS */
		DDRE.byte |= BIT4 + BIT3 + BIT2;	
		DDRK.bit.BIT6 = 1;				
		idle_gpio();
		   
		/* No E clock on port E bit 4, set LSTRE and RDWE bits now to make 
		   LSTRB and R/W signals available when switching to expanded mode */   
		PEAR.byte |= NECLK + LSTRE + RDWE;

		/* terminate the hardware reset,
		   wait 50ms and save delay address */
		PTQ.bit.BIT6 = 1;				
		(*delay)();	
		delay50ms = delay;				

		/* The bank select register (base address + 0x0E) always return 0x33
		   in the high byte. It is used here to determine if the card responds
		   correctly at the default address of 0x0300, or at the selected
		   address defined by CARD_ADDRESS.	*/		   
		   
		if((read_via_gpio(0x030E) & 0xFF00) == 0x3300)
		{
			write_via_gpio(0x030E, 0x0001);			/* select bank 1 */
			write_via_gpio(0x0302, CARD_ADDRESS);	/* set lan card base address */		
		}
		
		/* Set local pointer to register only if card responds to defined address */

		if((read_via_gpio(CARD_ADDRESS + 0x0E) & 0xFF00) == 0x3300)
			nic = (LAN91C *)CARD_ADDRESS;
			
		/* prepare CPU to operate in normal expanded wide mode */
		EBICTL.bit.ESTR = 1;	/* strech E clock */
		PEAR.bit.NECLK = 0;		/* Port E bit 4 is now external E clock */
		MISC.byte = 0x05;		/* use 1 clock stretch + ROMON */
	}
}


/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/
uint16 ethernet_open(MAC_address *pa)	
{
	static uint16 autoneg = 0xFFFF;

	if(lan_card_access() && !opened)
	{
		uint8 save = PPAGE.byte;			/* save and change PPAGE register */
		PPAGE.byte = CARD_PAGE;
		
		if(autoneg == 0xFFFF)				/* start auto-negociation process */
		{									/* only after a power up */
			nic->b0.BSR = swap(0x3300);		/* select bank 0 */
			nic->b0.RPCR = swap(0x0810);	/* enables auto-negotiation */
											/* LED A = link detected */
			auto_negociate(3);				/* LED B = TX or RX packet occured */
			autoneg = read_mii(18);			/* save outcome of negociation */
		}	
			
		/* sets SWFDUP with state of DPLXDET bit from outcome of autoneg,
		   pads frames shorter than 64 bytes and enables transmitter */

		nic->b0.BSR = swap(0x3300);
		nic->b0.TCR = (autoneg & 0x0040) ? swap(0x8081) : swap(0x0081);

		nic->b0.RCR = swap(0x0300);		/* enable receiver + strip CRC */
		nic->b1.BSR = swap(0x0001);		/* select bank 1 */
		nic->b1.CR  = swap(0xA0B1);		/* default configuration */
		nic->b1.CTR = swap(0x1210);		/* do not use auto release feature */

		if((0xFFFE & nic->b1.IAR0) == 0 && nic->b1.IAR1 == 0 && nic->b1.IAR2 == 0)
		{
			/* set ethernet address using the value defined in this module
		       only when the address is not loaded from local EEPROM, ie
		       when iar0 = 0 (except bit 0), iar1 = 0 and iar2 = 0 */
		       
			nic->b1.IAR0 = mac.word[0]; 
			nic->b1.IAR1 = mac.word[1];
			nic->b1.IAR2 = mac.word[2]; 
		}									

		pa->word[0] = nic->b1.IAR0;		/* set MAC address to return */		
		pa->word[1] = nic->b1.IAR1;
		pa->word[2] = nic->b1.IAR2;
				
		nic->b2.BSR = swap(0x0002);		/* select bank 2 */
		nic->b2.MMUCR = swap(0x0040);	/* reset MMU to initial state */
	
		nic->b2.ist = 0x0400;			/* clear TX EMPTY flag */
		nic->b2.ist = 0x0200;			/* clear TX INT flag */
		nic->b2.ist = 0x1000;			/* clear RX OVERRUN INT flag */
 		nic->b2.ist = 0x0010;			/* set RX OVRN INT mask */
		
		INTCR.byte |= IRQEN + IRQE;		/* enable IRQ on falling edge */
		opened = true;					/* set flag to control access */
		PPAGE.byte = save;				/* restore PPAGE register */
		return autoneg;					 
	}
	else
		return 0xFFFF;	
}


void ethernet_close()
{
	if(lan_card_access() && opened)
	{
		uint8 save = PPAGE.byte;		/* change PPAGE register */
		PPAGE.byte = CARD_PAGE;

		nic->b0.BSR = swap(0x3300);		/* select bank 0 */

		tmp = nic->b0.RCR & swap(0xFEFF);
		nic->b0.RCR = tmp;				/* disable the receiver by clearing RXEN bit */

		tmp = nic->b0.TCR & swap(0xFFFE);
		nic->b0.TCR = tmp;				/* disable transmit by clearing TXENA bit */

		INTCR.bit.IRQEN = 0;			/* disable IRQ pin interrupt */
		opened = false;					/* clear flag to force re-open() */
		PPAGE.byte = save;				/* restore PPAGE register */
	}	
}


/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/
uint16 ethernet_get(void *buf, uint16 maxlen)
{
	uint16 retval = 0;

	if(lan_card_access() && opened)
	{
		uint8 save = PPAGE.byte;			/* change PPAGE register */
		PPAGE.byte = CARD_PAGE;	
							
		nic->b2.BSR = swap(0x0002);			/* select bank 2 */
		if(nic->b2.ist & 0x0100)			/* return if RCV INT flag is cleared */
		{					
			nic->b2.PTR = swap(0xE000);		/* set auto increment, read operation */
											/* and address pointer to status word */
											
			uint16 status = swap(nic->b2.DATA);		/* read received frame status  */
			uint16 count = swap(nic->b2.DATA) & 0x07FF;	/* byte count (always even)*/

			if(count > 6)
			{								
				count -= 6;					/* remove status, count and control */
				if(status & 0x1000)			/* words to obtain data size */
					count++;				/* increase count if ODD bit is on */

				if(count <= maxlen)				
				{
					read_data(buf, count);	/* read frame data and return */
					retval = count;			/* number of bytes placed in buf */		
				}
			}
											/* release memory (RCV INT also cleared) */
			nic->b2.MMUCR = swap(0x0080);	/* remove and release top of RX fifo */
			
			/* it is safe to exit now, even if the MMU is processing the command */
			//while(nic->b2.MMUCR & swap(0x0001));
		}						
		PPAGE.byte = save;					/* restore PPAGE register */
	}
	return retval;
}

static void read_data(void *buf, uint16 len)
{
	uint16 *p = (uint16 *)buf;
	while(p < (uint16 *)(buf +len))
		*p++ = nic->b2.DATA;				/* read data 2 bytes at a time */
}


/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/
void ethernet_put(void *buf, void *appdata, uint16 len, uint16 header_len)
{
	if(lan_card_access() && opened)
	{
		/* prepare data */
		
		uint8 save = PPAGE.byte;				/* change PPAGE register */
		PPAGE.byte = CARD_PAGE;						

		nic->b2.BSR = swap(0x0002);				/* select bank 2 */
		nic->b2.MMUCR = swap(0x0020);			/* ask MMU to allocate memory for TX */
		while(!(nic->b2.ist & 0x0800));			/* wait for completion */

		uint16 packno = nic->b2.pnr & 0x003F;	/* get allocated packet number */
		nic->b2.pnr = packno << 8;				/* copy the packet number to PNR */
											 
		nic->b2.PTR = swap(0x4002);				/* set auto inc., write operation */
		nic->b2.DATA = swap(len +2 +2 +2);		/* and address pointer to store */
												/* frame byte count */
		uint8 *p;										
		if(len <= header_len)
			p = write_data(buf, len);			/* write the uip buffer alone */
		else
		{
			p = write_data(buf, header_len);	/* write the uip buffer */
			uint16 count = len - header_len;
			if(header_len % 2)							
			{										
				uint8 *pa = (uint8 *)appdata;	/* write last byte of buf */
				uint16 val = *p << 8 | *pa;		/* and first byte of app */
				nic->b2.DATA = swap(val);		/* as a word if odd length, */
				appdata = ++pa;					/* adjust pointer and byte */	
				count--;						/* count for next write */
			}
			p = write_data(appdata, count);		/* then write the application data */
		}

		/* write the control word to complete the frame. The control
		   word contains the last byte when the byte count is odd.
		   0x0000 when EVEN, 0x20xx (xx = last byte) when ODD */
  
   		nic->b2.DATA = swap((len % 2) ? (0x2000 + *p) : 0x0000);


		/* transmit frame and ignore any TX error */   									

 		nic->b2.MMUCR = swap(0x00C0);			/* enqueue packet for transmission */
 		nic->b2.ist = 0x0002;					/* set TX INT mask and let IRQ */
 												/* service routine handle the outcome */
 		PPAGE.byte = save;						/* restore PPAGE register */
	}
}


static uint8 *write_data(void *buf, uint16 len)
{
	len &= 0xFFFE;						/* force an even number of writes */
	uint16 *p = (uint16 *)buf;
	while(p < (uint16 *)(buf +len))
		nic->b2.DATA = *p++;			/* transfer data 2 bytes at a time */
		
	return (uint8 *)p;					/* when len is odd, p points */
										/* on last byte not written yet */	
}


/*------------------------------------------------------------------------------
	IRQ service routine to acknowledge TX operations and RX overruns
------------------------------------------------------------------------------*/

void ethernet_service()
{
	uint8 save = PPAGE.byte;				/* save and change PPAGE register */
	PPAGE.byte = CARD_PAGE;	
	uint16 bsr = nic->b0.BSR;				/* save current bank selection */

	nic->b2.BSR = swap(0x0002);				/* select bank 2 */		
	uint16 ist = nic->b2.ist;				/* read interrupt status register */


	/* handle TX interrupt */

	if(ist & 0x0200)
	{
		nic->b2.ist = 0x0200;				/* clear TX INT flag */
		nic->b2.MMUCR = swap(0x00A0);		/* release the packet used for TX */
											/* since we ignore TX errors, there is */
											/* no need to verify the cause */
		nic->b0.BSR = swap(0x3300);
		tmpi = nic->b0.TCR | 0x0100;		/* set TXENA bit to transmit again */
		nic->b0.TCR = tmpi;					/* in case there was an error */
	}
	
	
	/* handle RX overrun interrupt */

	if(ist & 0x1000)
	{
		nic->b0.BSR = swap(0x3300);
		if(nic->b0.MIR == swap(0x0004))		/* check if no memory available */
		{
			nic->b2.BSR = swap(0x0002);		/* no, then select bank 2 */		
			nic->b2.MMUCR = swap(0x0040);	/* issue reset MMU command */
		}
		nic->b2.BSR = swap(0x0002);			/* select bank 2 */		
		nic->b2.ist = 0x1000;				/* clear RX overrun INT flag */
	}
	
	nic->b0.BSR = bsr;						/* restore curent bank selection */
	PPAGE.byte = save;						/* restore PPAGE register */
}

/*------------------------------------------------------------------------------
	END
------------------------------------------------------------------------------*/
