
/*------------------------------------------------------------------------------
	mc9s12e_regs.h

	Most registers of the MC9S12E family are defined here.
	 
	Author : Pierre Morency <pmorency@globetrotter.net>

	Created: 19.apr.2004
	Revised: 27.jun.2004
--------------------------------------------------------------------------------

 Copyright  2004, Pierre Morency.
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of author nor the names of its contributors
      may be used to endorse or promote products derived from this
      software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

------------------------------------------------------------------------------*/

#ifndef MC9S12E_REGS_H
#define MC9S12E_REGS_H 1

#include "datatypes.h"

#ifndef REG_BASE
 enum { REG_BASE = 0x0000 };
#endif


/*------------------------------------------------------------------------------
	Definition of generic 8-bit register type
------------------------------------------------------------------------------*/

typedef union {
	uint8 byte;
	struct {
		uint8 BIT7 : 1;
		uint8 BIT6 : 1;
		uint8 BIT5 : 1;
		uint8 BIT4 : 1;
		uint8 BIT3 : 1;
		uint8 BIT2 : 1;
		uint8 BIT1 : 1;
		uint8 BIT0 : 1;
	}bit;
}reg08;

enum {	BIT7 = 0x80,		/* bit masks */
		BIT6 = 0x40,
		BIT5 = 0x20,
		BIT4 = 0x10,
		BIT3 = 0x08,
		BIT2 = 0x04,
		BIT1 = 0x02,
		BIT0 = 0x01 };


/*------------------------------------------------------------------------------
	Definition of generic 16-bit register type
------------------------------------------------------------------------------*/

typedef union {
	uint16 word;
	struct {
		uint16 BIT15 : 1;
		uint16 BIT14 : 1;
		uint16 BIT13 : 1;
		uint16 BIT12 : 1;
		uint16 BIT11 : 1;
		uint16 BIT10 : 1;
		uint16 BIT9  : 1;
		uint16 BIT8  : 1;
		uint16 BIT7  : 1;
		uint16 BIT6  : 1;
		uint16 BIT5  : 1;
		uint16 BIT4  : 1;
		uint16 BIT3  : 1;
		uint16 BIT2  : 1;
		uint16 BIT1  : 1;
		uint16 BIT0  : 1;
	}bit;
}reg16;

enum {	BIT15 = 0x8000,		/* bit masks */
		BIT14 = 0x4000,
		BIT13 = 0x2000,
		BIT12 = 0x1000,
		BIT11 = 0x0800,
		BIT10 = 0x0400,
		BIT9  = 0x0200,
		BIT8  = 0x0100 };


/*------------------------------------------------------------------------------
	M E B I 
------------------------------------------------------------------------------*/

#define PORTA (*((volatile reg08*)(0x0000 + REG_BASE)))
#define PORTB (*((volatile reg08*)(0x0001 + REG_BASE)))

#define DDRAB (*((volatile reg16*)(0x0002 + REG_BASE)))
#define DDRA  (*((volatile reg08*)(0x0002 + REG_BASE)))
#define DDRB  (*((volatile reg08*)(0x0003 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 BIT7 : 1;
		uint8 BIT6 : 1;
		uint8 BIT5 : 1;
		uint8 BIT4 : 1;
		uint8 BIT3 : 1;
		uint8 BIT2 : 1;
		uint8      : 2;
	}bit;
}porte;

#define PORTE (*((volatile porte*)(0x0008 + REG_BASE)))
#define DDRE  (*((volatile porte*)(0x0009 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 NOACCE : 1;
		uint8        : 1;
		uint8 PIPOE  : 1;
		uint8 NECLK  : 1;
		uint8 LSTRE  : 1;
		uint8 RDWE   : 1;
		uint8        : 2;
	}bit;
}pear;

enum {	NOACCE = 0x80,	/* bit masks */
		PIPOE  = 0x20,
		NECLK  = 0x10,
		LSTRE  = 0x08,
		RDWE   = 0x04 };

#define PEAR (*((volatile pear*)(0x000A + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 MODC : 1;
		uint8 MODB : 1;
		uint8 MODA : 1;
		uint8      : 1;
		uint8 IVIS : 1;
		uint8      : 1;
		uint8 EMK  : 1;
		uint8 EME  : 1;
	}bit;
}mode;

enum {	MODC = 0x80,		/* bit masks */
		MODB = 0x40,
		MODA = 0x20,
		IVIS = 0x08,
		EMK  = 0x02,
		EME  = 0x01 };

#define MODE (*((volatile mode*)(0x000B + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 PUPKE : 1;
		uint8       : 2;
		uint8 PUPEE : 1;
		uint8       : 2;
		uint8 PUPBE : 1;
		uint8 PUPAE : 1;
	}bit;
}pucr;

enum {	PUPKE = 0x80,		/* bit masks */
		PUPEE = 0x10,
		PUPBE = 0x02,
		PUPAE = 0x01 };

#define PUCR (*((volatile pucr*)(0x000C + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 RDPK : 1;
		uint8      : 2;
		uint8 RDPE : 1;
		uint8      : 2;
		uint8 RDPB : 1;
		uint8 RDPA : 1;
	}bit;
}rdriv;

enum {	RDPK = 0x80,		/* bit masks */
		RDPE = 0x10,
		RDPB = 0x02,
		RDPA = 0x01 };

#define RDRIV (*((volatile rdriv*)(0x000D + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8      : 7;
		uint8 ESTR : 1;
	}bit;
}ebictl;

enum {	ESTR = 0x01 };		/* bit mask */

#define EBICTL (*((volatile ebictl*)(0x000E + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 IRQE  : 1;
		uint8 IRQEN : 1;
		uint8       : 6;
	}bit;
}intcr;

enum {	IRQE  = 0x80,		/* bit masks */
		IRQEN = 0x40 };

#define INTCR (*((volatile intcr*)(0x001E + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 ECS   : 1;
		uint8 XCS   : 1;
		uint8 XAB19 : 1;
		uint8 XAB18 : 1;
		uint8 XAB17 : 1;
		uint8 XAB16 : 1;
		uint8 XAB15 : 1;
		uint8 XAB14 : 1;
	}bit;
}portk;

enum {	ECS   = 0x80,		/* bit masks */
		XCS   = 0x40,
		XAB19 = 0x20,
		XAB18 = 0x10,
		XAB17 = 0x08,	
		XAB16 = 0x04,
		XAB15 = 0x02,
		XAB14 = 0x01 };

#define PORTK (*((volatile portk*)(0x0032 + REG_BASE)))
#define DDRK  (*((volatile reg08*)(0x0033 + REG_BASE)))


/*------------------------------------------------------------------------------
	M M C 
------------------------------------------------------------------------------*/

typedef union {
	uint8 byte;
	struct {
		uint8 RAM15  : 1;
		uint8 RAM14  : 1;
		uint8 RAM13  : 1;
		uint8 RAM12  : 1;
		uint8 RAM11  : 1;
		uint8        : 2;
		uint8 RAMHAL : 1;
	}bit;
}initrm;

enum {	RAM15  = 0x80,		/* bit masks */
		RAM14  = 0x40,
		RAM13  = 0x20,
		RAM12  = 0x10,
		RAM11  = 0x08,
		RAMHAL = 0x01 };

#define INITRM (*((volatile initrm*)(0x0010 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8       : 1;
		uint8 REG14 : 1;
		uint8 REG13 : 1;
		uint8 REG12 : 1;
		uint8 REG11 : 1;
		uint8       : 3;
	}bit;
}initrg;

enum {	REG14 = 0x40,		/* bit masks */
		REG13 = 0x20,
		REG12 = 0x10,
		REG11 = 0x08 };

#define INITRG (*((volatile initrg*)(0x0011 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 EE15 : 1;
		uint8 EE14 : 1;
		uint8 EE13 : 1;
		uint8 EE12 : 1;
		uint8 EE11 : 1;
		uint8      : 2;
		uint8 EEON : 1;
	}bit;
}initee;

enum {	EE15 = 0x80,		/* bit masks */
		EE14 = 0x40,
		EE13 = 0x20,
		EE12 = 0x10,
		EE11 = 0x08,
		EEON = 0x01 };

#define INITEE (*((volatile initee*)(0x0012 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8        : 4;
		uint8 EXSTR1 : 1;
		uint8 EXSTR0 : 1;
		uint8 ROMHM  : 1;
		uint8 ROMON  : 1;
	}bit;
}misc;

enum {	EXSTR1 = 0x08,		/* bit masks */
		EXSTR0 = 0x04,
		ROMHM  = 0x02,
		ROMON  = 0x01 };

#define MISC (*((volatile misc*)(0x0013 + REG_BASE)))

#define MTST0 (*((volatile reg08*)(0x0014 + REG_BASE)))
#define MTST1 (*((volatile reg08*)(0x0017 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 REG_SW0 : 1;
		uint8         : 1;
		uint8 EEP_SW1 : 1;
		uint8 EEP_SW0 : 1;
		uint8         : 1;
		uint8 RAM_SW2 : 1;
		uint8 RAM_SW1 : 1;
		uint8 RAM_SW0 : 1;
	}bit;
}memsiz0;

enum {	REG_SW0 = 0x80,		/* bit masks */
		EEP_SW1 = 0x20,
		EEP_SW0 = 0x10,
		RAM_SW2 = 0x04,
		RAM_SW1 = 0x02,
		RAM_SW0 = 0x01 };

#define MEMSIZ0 (*((volatile memsiz0*)(0x001C + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 ROM_SW1 : 1;
		uint8 ROM_SW0 : 1;
		uint8         : 4;
		uint8 PAG_SW1 : 1;
		uint8 PAG_SW0 : 1;
	}bit;
}memsiz1;

enum {	ROM_SW1 = 0x80,		/* bit masks */
		ROM_SW0 = 0x40,
		PAG_SW1 = 0x02,
		PAG_SW0 = 0x01 };

#define MEMSIZ1 (*((volatile memsiz1*)(0x001C + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8      : 2;
		uint8 PIX5 : 1;
		uint8 PIX4 : 1;
		uint8 PIX3 : 1;
		uint8 PIX2 : 1;
		uint8 PIX1 : 1;
		uint8 PIX0 : 1;
	}bit;
}ppage;

enum {	PIX5 = 0x20,			/* bit masks */
		PIX4 = 0x10,
		PIX3 = 0x08,
		PIX2 = 0x04,
		PIX1 = 0x02,
		PIX0 = 0x01 };

#define PPAGE (*((volatile ppage*)(0x0030 + REG_BASE)))


/*------------------------------------------------------------------------------
	I N T  +  M I S C   
------------------------------------------------------------------------------*/

typedef union {
	uint8 byte;
	struct {
		uint8       : 3;
		uint8 WRINT : 1;
		uint8 ADR3  : 1;
		uint8 ADR2  : 1;
		uint8 ADR1  : 1;
		uint8 ADR0  : 1;
	}bit;
}itcr;

enum {	WRINT = 0x10,		/* bit masks */
		ADR3  = 0x08,
		ADR2  = 0x04,
		ADR1  = 0x02,
		ADR0  = 0x01 };

#define ITCR (*((volatile itcr*)(0x0015 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 INTE : 1;	
		uint8 INTC : 1;	
		uint8 INTA : 1;
		uint8 INT8 : 1;
		uint8 INT6 : 1;
		uint8 INT4 : 1;
		uint8 INT2 : 1;
		uint8 INT0 : 1;
	}bit;
}itest;

enum {	INTE = 0x80,		/* bit masks */
		INTC = 0x40,
		INTA = 0x20,
		INT8 = 0x10,	
		INT6 = 0x08,
		INT4 = 0x04,
		INT2 = 0x02,
		INT0 = 0x01 };

#define ITEST (*((volatile itest*)(0x0016 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 PSEL7 : 1;	
		uint8 PSEL6 : 1;	
		uint8 PSEL5 : 1;
		uint8 PSEL4 : 1;
		uint8 PSEL3 : 1;
		uint8 PSEL2 : 1;
		uint8 PSEL1 : 1;
		uint8       : 1;
	}bit;
}hprio;

enum {	PSEL7 = 0x80,		/* bit masks */
		PSEL6 = 0x40,
		PSEL5 = 0x20,
		PSEL4 = 0x10,	
		PSEL3 = 0x08,
		PSEL2 = 0x04,
		PSEL1 = 0x02 };

#define HPRIO (*((volatile hprio*)(0x001F + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8      : 5;
		uint8 LVDS : 1;		/* read only bit */
		uint8 LVIE : 1;
		uint8 LVIF : 1;
	}bit;
}vregctrl;

enum {	LVDS = 0x04,		/* bit masks */
		LVIE = 0x02,
		LVIF = 0x01 };

#define VREGCTRL (*((volatile vregctrl*)(0x0019 + REG_BASE)))

#define PARTID (*((volatile reg16*)(0x001A + REG_BASE)))


/*------------------------------------------------------------------------------
	C R G   
------------------------------------------------------------------------------*/

typedef union {
	uint8 byte;
	struct {
		uint8      : 2;
		uint8 SYN5 : 1;
		uint8 SYN4 : 1;
		uint8 SYN3 : 1;
		uint8 SYN2 : 1;
		uint8 SYN1 : 1;
		uint8 SYN0 : 1;
	}bit;
}synr;

enum {	SYN5 = 0x20,		/* bit masks */
		SYN4 = 0x10,
		SYN3 = 0x08,
		SYN2 = 0x04,
		SYN1 = 0x02,
		SYN0 = 0x01 };

#define SYNR (*((volatile synr*)(0x0034 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8        : 4;
		uint8 REFDV3 : 1;
		uint8 REFDV2 : 1;
		uint8 REFDV1 : 1;
		uint8 REFDV0 : 1;
	}bit;
}refdv;

enum {	REFDV3 = 0x08,		/* bit masks */
		REFDV2 = 0x04,
		REFDV1 = 0x02,
		REFDV0 = 0x01 };

#define REFDV (*((volatile refdv*)(0x0035 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 TOUT7 : 1;	
		uint8 TOUT6 : 1;	
		uint8 TOUT5 : 1;
		uint8 TOUT4 : 1;
		uint8 TOUT3 : 1;
		uint8 TOUT2 : 1;
		uint8 TOUT1 : 1;
		uint8 TOUT0 : 1;
	}bit;
}ctflg;

enum {	TOUT7 = 0x80,		/* bit masks */
		TOUT6 = 0x40,
		TOUT5 = 0x20,
		TOUT4 = 0x10,	
		TOUT3 = 0x08,
		TOUT2 = 0x04,
		TOUT1 = 0x02,
		TOUT0 = 0x01 };

#define CTFLG (*((volatile ctflg*)(0x0036 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 RTIF   : 1;	
		uint8 PROF   : 1;	
		uint8        : 1;
		uint8 LOCKIF : 1;
		uint8 LOCK   : 1;
		uint8 TRACK  : 1;
		uint8 SCMIF  : 1;
		uint8 SCM    : 1;
	}bit;
}crgflg;

enum {	RTIF   = 0x80,		/* bit masks */
		PROF   = 0x40,
		LOCKIF = 0x10,	
		LOCK   = 0x08,
		TRACK  = 0x04,
		SCMIF  = 0x02,
		SCM    = 0x01 };

#define CRGFLG (*((volatile crgflg*)(0x0037 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 RTIE   : 1;	
		uint8        : 2;	
		uint8 LOCKIE : 1;
		uint8        : 2;
		uint8 SCMIE  : 1;
		uint8        : 1;
	}bit;
}crgint;

enum {	RTIE   = 0x80,		/* bit masks */
		LOCKIE = 0x10,	
		SCMIE  = 0x02 };

#define CRGINT (*((volatile crgint*)(0x0038 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 PLLSEL : 1;	
		uint8 PSTP   : 1;	
		uint8 SYSWAI : 1;
		uint8 ROAWAI : 1;
		uint8 PLLWAI : 1;
		uint8 CWAI   : 1;
		uint8 RTIWAI : 1;
		uint8 COPWAI : 1;
	}bit;
}clksel;

enum {	PLLSEL = 0x80,		/* bit masks */
		PSTP   = 0x40,
		SYSWAI = 0x20,
		ROAWAI = 0x10,	
		PLLWAI = 0x08,
		CWAI   = 0x04,
		RTIWAI = 0x02,
		COPWAI = 0x01 };

#define CLKSEL (*((volatile clksel*)(0x0039 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 CME   : 1;	
		uint8 PLLON : 1;	
		uint8 AUTO  : 1;
		uint8 ACQ   : 1;
		uint8       : 1;
		uint8 PRE   : 1;
		uint8 PCE   : 1;
		uint8 SCME  : 1;
	}bit;
}pllctl;

enum {	CME   = 0x80,		/* bit masks */
		PLLON = 0x40,
		AUTO  = 0x20,
		ACQ   = 0x10,	
		PRE   = 0x04,
		PCE   = 0x02,
		SCME  = 0x01 };

#define PLLCTL (*((volatile pllctl*)(0x003A + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8      : 1;	
		uint8 RTR6 : 1;	
		uint8 RTR5 : 1;
		uint8 RTR4 : 1;
		uint8 RTR3 : 1;
		uint8 RTR2 : 1;
		uint8 RTR1 : 1;
		uint8 RTR0 : 1;
	}bit;
}rtictl;

enum {	RTR6 = 0x40,		/* bit masks */
		RTR5 = 0x20,
		RTR4 = 0x10,	
		RTR3 = 0x08,
		RTR2 = 0x04,
		RTR1 = 0x02,
		RTR0 = 0x01 };

#define RTICTL (*((volatile rtictl*)(0x003B + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 WCOP  : 1;	
		uint8 RSBCK : 1;	
		uint8       : 3;
		uint8 CR2   : 1;
		uint8 CR1   : 1;
		uint8 CR0   : 1;
	}bit;
}copctl;

enum {	WCOP  = 0x80,		/* bit masks */
		RSBCK = 0x40,
		CR2   = 0x04,
		CR1   = 0x02,
		CR0   = 0x01 };

#define COPCTL (*((volatile copctl*)(0x003C + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 RTIBYP : 1;	
		uint8 COPBYP : 1;	
		uint8        : 1;
		uint8 PLLBYP : 1;	
		uint8        : 1;
		uint8        : 1;
		uint8 FCM    : 1;
		uint8        : 1;
	}bit;
}forbyp;

enum {	RTIBYP = 0x80,		/* bit masks */
		COPBYP = 0x40,
		PLLBYP = 0x10,
		FCM    = 0x02 };

#define FORBYP (*((volatile forbyp*)(0x003D + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 TCTL7 : 1;	
		uint8 TCTL6 : 1;	
		uint8 TCTL5 : 1;
		uint8 TCTL4 : 1;
		uint8 TCTL3 : 1;
		uint8 TCTL2 : 1;
		uint8 TCTL1 : 1;
		uint8 TCTL0 : 1;
	}bit;
}ctctl;

enum {	TCTL7 = 0x80,		/* bit masks */
		TCTL6 = 0x40,
		TCTL5 = 0x20,
		TCTL4 = 0x10,	
		TCTL3 = 0x08,
		TCTL2 = 0x04,
		TCTL1 = 0x02,
		TCTL0 = 0x01 };

#define CTCTL (*((volatile ctctl*)(0x003E + REG_BASE)))

#define ARMCOP (*((volatile reg08*)(0x003F + REG_BASE)))


/*------------------------------------------------------------------------------
	T I M   ( 3 )	   
------------------------------------------------------------------------------*/

typedef union {
	uint8 byte;
	struct {
		uint8 IOS7 : 1;
		uint8 IOS6 : 1;
		uint8 IOS5 : 1;
		uint8 IOS4 : 1;
		uint8      : 4;
	}bit;
}tios;
enum {	IOS7 = 0x80,		/* bit masks */
		IOS6 = 0x40,
		IOS5 = 0x20,
		IOS4 = 0x10 };


typedef union {
	uint8 byte;
	struct {
		uint8 FOC7 : 1;
		uint8 FOC6 : 1;
		uint8 FOC5 : 1;
		uint8 FOC4 : 1;
		uint8      : 4;
	}bit;
}cforc;
enum {	FOC7 = 0x80,		/* bit masks */
		FOC6 = 0x40,
		FOC5 = 0x20,
		FOC4 = 0x10 };


typedef union {
	uint8 byte;
	struct {
		uint8 OC7M7 : 1;
		uint8 OC7M6 : 1;
		uint8 OC7M5 : 1;
		uint8 OC7M4 : 1;
		uint8       : 4;
	}bit;
}oc7m;
enum {	OC7M7 = 0x80,		/* bit masks */
		OC7M6 = 0x40,
		OC7M5 = 0x20,
		OC7M4 = 0x10 };


typedef union {
	uint8 byte;
	struct {
		uint8 OC7D7 : 1;
		uint8 OC7D6 : 1;
		uint8 OC7D5 : 1;
		uint8 OC7D4 : 1;
		uint8       : 4;
	}bit;
}oc7d;
enum {	OC7D7 = 0x80,		/* bit masks */
		OC7D6 = 0x40,
		OC7D5 = 0x20,
		OC7D4 = 0x10 };


typedef union {
	uint8 byte;
	struct {
		uint8 TEN   : 1;
		uint8 TSWAI : 1;
		uint8 TSFRZ : 1;
		uint8 TFFCA : 1;
		uint8       : 4;
	}bit;
}tscr1;
enum {	TEN   = 0x80,		/* bit masks */
		TSWAI = 0x40,
		TSFRZ = 0x20,
		TFFCA = 0x10 };


typedef union {
	uint8 byte;
	struct {
		uint8 TOV7 : 1;
		uint8 TOV6 : 1;
		uint8 TOV5 : 1;
		uint8 TOV4 : 1;
		uint8      : 4;
	}bit;
}ttov;
enum {	TOV7 = 0x80,		/* bit masks */
		TOV6 = 0x40,
		TOV5 = 0x20,
		TOV4 = 0x10 };


typedef union {
	uint8 byte;
	struct {
		uint8 OM7 : 1;	
		uint8 OL7 : 1;	
		uint8 OM6 : 1;
		uint8 OL6 : 1;
		uint8 OM5 : 1;
		uint8 OL5 : 1;
		uint8 OM4 : 1;
		uint8 OL4 : 1;
	}bit;
}tctl1;
enum {	OM7 = 0x80,			/* bit masks */
		OL7 = 0x40,
		OM6 = 0x20,
		OL6 = 0x10,	
		OM5 = 0x08,
		OL5 = 0x04,
		OM4 = 0x02,
		OL4 = 0x01 };


typedef union {
	uint8 byte;
	struct {
		uint8 EDG7B : 1;	
		uint8 EDG7A : 1;	
		uint8 EDG6B : 1;
		uint8 EDG6A : 1;
		uint8 EDG5B : 1;
		uint8 EDG5A : 1;
		uint8 EDG4B : 1;
		uint8 EDG4A : 1;
	}bit;
}tctl3;
enum {	EDG7B = 0x80,		/* bit masks */
		EDG7A = 0x40,
		EDG6B = 0x20,
		EDG6A = 0x10,	
		EDG5B = 0x08,
		EDG5A = 0x04,
		EDG4B = 0x02,
		EDG4A = 0x01 };


typedef union {
	uint8 byte;
	struct {
		uint8 C7I : 1;
		uint8 C6I : 1;
		uint8 C5I : 1;
		uint8 C4I : 1;
		uint8     : 4;
	}bit;
}tie;
enum {	C7I = 0x80,			/* bit masks */
		C6I = 0x40,
		C5I = 0x20,
		C4I = 0x10 };


typedef union {
	uint8 byte;
	struct {
		uint8 TOI  : 1;	
		uint8      : 3;	
		uint8 TCRE : 1;
		uint8 PR2  : 1;
		uint8 PR1  : 1;
		uint8 PR0  : 1;
	}bit;
}tscr2;
enum {	TOI  = 0x80,		/* bit masks */
		TCRE = 0x08,
		PR2  = 0x04,
		PR1  = 0x02,
		PR0  = 0x01 };


typedef union {
	uint8 byte;
	struct {
		uint8 C7F : 1;
		uint8 C6F : 1;
		uint8 C5F : 1;
		uint8 C4F : 1;
		uint8     : 4;
	}bit;
}tflg1;
enum {	C7F = 0x80,			/* bit masks */
		C6F = 0x40,
		C5F = 0x20,
		C4F = 0x10 };

		
typedef union {
	uint8 byte;
	struct {
		uint8 TOF : 1;	
		uint8     : 7;
	}bit;
}tflg2;
enum {	TOF = 0x80 };		/* bit mask */


typedef union {
	uint8 byte;
	struct {
		uint8       : 1;	
		uint8 PAEN  : 1;	
		uint8 PAMOD : 1;
		uint8 PEDGE : 1;
		uint8 CLK1  : 1;
		uint8 CLK0  : 1;
		uint8 PAOVI : 1;
		uint8 PAI   : 1;
	}bit;
}pactl;
enum {	PAEN  = 0x40,		/* bit masks */
		PAMOD = 0x20,
		PEDGE = 0x10,	
		CLK1  = 0x08,
		CLK0  = 0x04,
		PAOVI = 0x02,
		PAI   = 0x01 };


typedef union {
	uint8 byte;
	struct {
		uint8       : 6;	
		uint8 PAOVF : 1;
		uint8 PAIF  : 1;
	}bit;
}paflg;
enum {	PAOVF = 0x02,		/* bit masks */
		PAIF  = 0x01 };


typedef struct {
	volatile tios  TIOS;	/* timer input capture/output compare select */
	volatile cforc CFORC;	/* timer compare force register */
	volatile oc7m  OC7M;	/* output compare 7 mask register */
	volatile oc7d  OC7D;	/* output compare 7 data register */
	volatile reg16 TCNT;	/* timer count register */
	volatile tscr1 TSCR1;	/* timer system control register 1 */
	volatile ttov  TTOV;	/* timer toggle on overflow register */
	volatile tctl1 TCTL1;	/*	timer control register 1 */
	volatile reg08 r0;		/* reserved (maintaining memory map) */
	volatile tctl3 TCTL3;	/* timer control register 3 */
	volatile reg08 r1;
	volatile tie   TIE;		/* timer interrupt mask 1 */
	volatile tscr2 TSCR2;	/* timer system control register 2 */
	volatile tflg1 TFLG1;	/* timer interrupt flag 1 */
	volatile tflg2 TFLG2;	/* timer interrupt flag 2 */
	volatile reg08 r2[8];
	volatile reg16 TC4;		/* timer input capture/output compare register */
	volatile reg16 TC5;		/* timer input capture/output compare register */
	volatile reg16 TC6;		/* timer input capture/output compare register */
	volatile reg16 TC7;		/* timer input capture/output compare register */
	volatile pactl PACTL;	/* pulse accumulator control register */
	volatile paflg PAFLG;	/* pulse accumulator flag register */
	volatile reg16 PACNT;	/* pulse accumulator count register */
	volatile reg08 r3[12];
}tim;	

#define TIM0 (*((volatile tim*)(0x0040 + REG_BASE)))
#define TIM1 (*((volatile tim*)(0x0140 + REG_BASE)))
#define TIM2 (*((volatile tim*)(0x0180 + REG_BASE)))


/*------------------------------------------------------------------------------
	A T D  	   
------------------------------------------------------------------------------*/

typedef union {
	uint8 byte;
	struct {
		uint8 ADPU    : 1;	
		uint8 AFFC    : 1;	
		uint8 AWAI    : 1;
		uint8 ETRIGLE : 1;
		uint8 ETRIGP  : 1;
		uint8 ETRIG   : 1;
		uint8 ASCIE   : 1;
		uint8 ASCIF   : 1;
	}bit;
}atdctl2;

enum {	ADPU    = 0x80,		/* bit masks */
		AFFC    = 0x40,
		AWAI    = 0x20,
		ETRIGLE = 0x10,	
		ETRIGP  = 0x08,
		ETRIG   = 0x04,
		ASCIE   = 0x02,
		ASCIF  = 0x01 };

#define ATDCTL2 (*((volatile atdctl2*)(0x0082 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8      : 1;	
		uint8 S8C  : 1;	
		uint8 S4C  : 1;
		uint8 S2C  : 1;
		uint8 S1C  : 1;
		uint8 FIFO : 1;
		uint8 FRZ1 : 1;
		uint8 FRZ0 : 1;
	}bit;
}atdctl3;

enum {	S8C  = 0x40,		/* bit masks */
		S4C  = 0x20,
		S2C  = 0x10,	
		S1C  = 0x08,
		FIFO = 0x04,
		FRZ1 = 0x02,
		FRZ0 = 0x01 };

#define ATDCTL3 (*((volatile atdctl3*)(0x0083 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 SRES8 : 1;	
		uint8 SMP1  : 1;	
		uint8 SMP0  : 1;
		uint8 PRS4  : 1;
		uint8 PRS3  : 1;
		uint8 PRS2  : 1;
		uint8 PRS1  : 1;
		uint8 PRS0  : 1;
	}bit;
}atdctl4;

enum {	SRES8 = 0x80,		/* bit masks */
		SMP1  = 0x40,		
		SMP0  = 0x20,
		PRS4  = 0x10,	
		PRS3  = 0x08,
		PRS2  = 0x04,
		PRS1  = 0x02,
		PRS0  = 0x01 };

#define ATDCTL4 (*((volatile atdctl4*)(0x0084 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 DJM  : 1;	
		uint8 DSGN : 1;	
		uint8 SCAN : 1;
		uint8 MULT : 1;
		uint8      : 1;
		uint8 CC   : 1;
		uint8 CB   : 1;
		uint8 CA   : 1;
	}bit;
}atdctl5;

enum {	DJM  = 0x80,		/* bit masks */
		DSGN = 0x40,		
		SCAN = 0x20,
		MULT = 0x10,	
		CC   = 0x04,
		CB   = 0x02,
		CA   = 0x01 };

#define ATDCTL5 (*((volatile atdctl5*)(0x0085 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 SCF   : 1;	
		uint8       : 1;	
		uint8 ETORF : 1;
		uint8 FIFOR : 1;
		uint8       : 1;
		uint8 CC2   : 1;
		uint8 CC1   : 1;
		uint8 CC0   : 1;
	}bit;
}atdstat0;

enum {	SCF   = 0x80,		/* bit masks */
		ETORF = 0x20,
		FIFOR = 0x10,	
		CC2   = 0x04,
		CC1   = 0x02,
		CC0   = 0x01 };

#define ATDSTAT0 (*((volatile atdstat0*)(0x0086 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8    : 7;	
		uint8 SC : 1;
	}bit;
}atdtest1;

enum {	SC = 0x01 };		/* bit mask */

#define ATDTEST1 (*((volatile atdtest1*)(0x0089 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 CCF15 : 1;	
		uint8 CCF14 : 1;	
		uint8 CCF13 : 1;
		uint8 CCF12 : 1;
		uint8 CCF11 : 1;
		uint8 CCF10 : 1;
		uint8 CCF9  : 1;
		uint8 CCF8  : 1;
	}bit;
}atdstat2;

enum {	CCF15 = 0x80,		/* bit masks */
		CCF14 = 0x40,		
		CCF13 = 0x20,
		CCF12 = 0x10,	
		CCF11 = 0x08,
		CCF10 = 0x04,
		CCF9  = 0x02,
		CCF8  = 0x01 };

#define ATDSTAT2 (*((volatile atdstat2*)(0x008A + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 CCF7 : 1;	
		uint8 CCF6 : 1;	
		uint8 CCF5 : 1;
		uint8 CCF4 : 1;
		uint8 CCF3 : 1;
		uint8 CCF2 : 1;
		uint8 CCF1 : 1;
		uint8 CCF0 : 1;
	}bit;
}atdstat1;

enum {	CCF7 = 0x80,		/* bit masks */
		CCF6 = 0x40,		
		CCF5 = 0x20,
		CCF4 = 0x10,	
		CCF3 = 0x08,
		CCF2 = 0x04,
		CCF1 = 0x02,
		CCF0 = 0x01 };

#define ATDSTAT1 (*((volatile atdstat1*)(0x008B + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 IEN15 : 1;	
		uint8 IEN14 : 1;	
		uint8 IEN13 : 1;
		uint8 IEN12 : 1;
		uint8 IEN11 : 1;
		uint8 IEN10 : 1;
		uint8 IEN9  : 1;
		uint8 IEN8  : 1;
	}bit;
}atddien0;

enum {	IEN15 = 0x80,		/* bit masks */
		IEN14 = 0x40,		
		IEN13 = 0x20,
		IEN12 = 0x10,	
		IEN11 = 0x08,
		IEN10 = 0x04,
		IEN9  = 0x02,
		IEN8  = 0x01 };

#define ATDDIEN0 (*((volatile atddien0*)(0x008C + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 IEN7 : 1;	
		uint8 IEN6 : 1;	
		uint8 IEN5 : 1;
		uint8 IEN4 : 1;
		uint8 IEN3 : 1;
		uint8 IEN2 : 1;
		uint8 IEN1 : 1;
		uint8 IEN0 : 1;
	}bit;
}atddien1;

enum {	IEN7 = 0x80,		/* bit masks */
		IEN6 = 0x40,		
		IEN5 = 0x20,
		IEN4 = 0x10,	
		IEN3 = 0x08,
		IEN2 = 0x04,
		IEN1 = 0x02,
		IEN0 = 0x01 };

#define ATDDIEN1 (*((volatile atddien1*)(0x008D + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 PTAD15 : 1;	
		uint8 PTAD14 : 1;	
		uint8 PTAD13 : 1;
		uint8 PTAD12 : 1;
		uint8 PTAD11 : 1;
		uint8 PTAD10 : 1;
		uint8 PTAD9  : 1;
		uint8 PTAD8  : 1;
	}bit;
}portad0;

enum {	PTAD15 = 0x80,		/* bit masks */
		PTAD14 = 0x40,		
		PTAD13 = 0x20,
		PTAD12 = 0x10,	
		PTAD11 = 0x08,
		PTAD10 = 0x04,
		PTAD9  = 0x02,
		PTAD8  = 0x01 };

#define PORTAD0 (*((volatile portad0*)(0x008E + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 PTAD7 : 1;	
		uint8 PTAD6 : 1;	
		uint8 PTAD5 : 1;
		uint8 PTAD4 : 1;
		uint8 PTAD3 : 1;
		uint8 PTAD2 : 1;
		uint8 PTAD1 : 1;
		uint8 PTAD0 : 1;
	}bit;
}portad1;

enum {	PTAD7 = 0x80,		/* bit masks */
		PTAD6 = 0x40,		
		PTAD5 = 0x20,
		PTAD4 = 0x10,	
		PTAD3 = 0x08,
		PTAD2 = 0x04,
		PTAD1 = 0x02,
		PTAD0 = 0x01 };

#define PORTAD1 (*((volatile portad1*)(0x008F + REG_BASE)))


#define ATDDR0  (*((volatile reg16*)(0x0090 + REG_BASE)))
#define ATDDR1  (*((volatile reg16*)(0x0092 + REG_BASE)))
#define ATDDR2  (*((volatile reg16*)(0x0094 + REG_BASE)))
#define ATDDR3  (*((volatile reg16*)(0x0096 + REG_BASE)))
#define ATDDR4  (*((volatile reg16*)(0x0098 + REG_BASE)))
#define ATDDR5  (*((volatile reg16*)(0x009A + REG_BASE)))
#define ATDDR6  (*((volatile reg16*)(0x009C + REG_BASE)))
#define ATDDR7  (*((volatile reg16*)(0x009E + REG_BASE)))
#define ATDDR8  (*((volatile reg16*)(0x00A0 + REG_BASE)))
#define ATDDR9  (*((volatile reg16*)(0x00A2 + REG_BASE)))
#define ATDDR10 (*((volatile reg16*)(0x00A4 + REG_BASE)))
#define ATDDR11 (*((volatile reg16*)(0x00A6 + REG_BASE)))
#define ATDDR12 (*((volatile reg16*)(0x00A8 + REG_BASE)))
#define ATDDR13 (*((volatile reg16*)(0x00AA + REG_BASE)))
#define ATDDR14 (*((volatile reg16*)(0x00AC + REG_BASE)))
#define ATDDR15 (*((volatile reg16*)(0x00AE + REG_BASE)))


/*------------------------------------------------------------------------------
	S C I  ( 3 )	   
------------------------------------------------------------------------------*/

typedef union {
	uint16 word;
	struct {
		uint16 IREN  : 1;
		uint16 TNP1  : 1;
		uint16 TNP0  : 1;
		uint16 SBR12 : 1;
		uint16 SBR11 : 1;
		uint16 SBR10 : 1;
		uint16 SBR9  : 1;
		uint16 SBR8  : 1;
		uint16 SBR7  : 1;
		uint16 SBR6  : 1;
		uint16 SBR5  : 1;
		uint16 SBR4  : 1;
		uint16 SBR3  : 1;
		uint16 SBR2  : 1;
		uint16 SBR1  : 1;
		uint16 SBR0  : 1;
	}bit;
}scibd;

enum {	IREN  = 0x8000,		/* bit masks */
		TNP1  = 0x4000,
		TNP0  = 0x2000,
		SBR12 = 0x1000,
		SBR11 = 0x0800,
		SBR10 = 0x0400,
		SBR9  = 0x0200,
		SBR8  = 0x0100,
		SBR7  = 0x80,		
		SBR6  = 0x40,
		SBR5  = 0x20,
		SBR4  = 0x10,
		SBR3  = 0x08,
		SBR2  = 0x04,
		SBR1  = 0x02,
		SBR0  = 0x01 };

typedef union {
	uint8 byte;
	struct {
		uint8 LOOPS   : 1;
		uint8 SCISWAI : 1;
		uint8 RSRC    : 1;
		uint8 M       : 1;
		uint8 WAKE    : 1;
		uint8 ILT     : 1;
		uint8 PE      : 1;
		uint8 PT      : 1;
	}bit;
}scicr1;

enum {	LOOPS   = 0x80,		/* bit masks */
		SCISAWI = 0x40,
		RSRC    = 0x20,
		M       = 0x10,
		WAKE    = 0x08,
		ILT     = 0x04,
		PE      = 0x02,
		PT      = 0x01 };

typedef union {
	uint8 byte;
	struct {
		uint8 TIE  : 1;
		uint8 TCIE : 1;
		uint8 RIE  : 1;
		uint8 ILIE : 1;
		uint8 TE   : 1;
		uint8 RE   : 1;
		uint8 RWU  : 1;
		uint8 SBK  : 1;
	}bit;
}scicr2;

enum {	TIE  = 0x80,		/* bit masks */
		TCIE = 0x40,
		RIE  = 0x20,
		ILIE = 0x10,
		TE   = 0x08,
		RE   = 0x04,
		RWU  = 0x02,
		SBK  = 0x01 };

typedef union {
	uint8 byte;
	struct {
		uint8 TDRE : 1;
		uint8 TC   : 1;
		uint8 RDRF : 1;
		uint8 IDLE : 1;
		uint8 OR   : 1;
		uint8 NF   : 1;
		uint8 FE   : 1;
		uint8 PF   : 1;
	}bit;
}scisr1;

enum {	TDRE = 0x80,		/* bit masks */
		TC   = 0x40,
		RDRF = 0x20,
		IDLE = 0x10,
		OR   = 0x08,
		NF   = 0x04,
		FE   = 0x02,
		PF   = 0x01 };

typedef union {
	uint8 byte;
	struct {
		uint8       : 5;
		uint8 BRK13 : 1;
		uint8 TXDIR : 1;
		uint8 RAF   : 1;
	}bit;
}scisr2;

enum {	BRK13 = 0x04,		/* bit masks */
		TXDIR = 0x02,
		RAF   = 0x01 };

typedef union {
	uint8 byte;
	struct {
		uint8    : 1;
		uint8 T8 : 1;
		uint8    : 6;
	}bit;
}scidrh;

enum {	T8 = 0x40 };		/* bit mask */


typedef struct {
	volatile scibd  SCIBD;
	volatile scicr1 SCICR1;
	volatile scicr2 SCICR2;
	volatile scisr1 SCISR1;
	volatile scisr2 SCISR2;
	volatile scidrh SCIDRH;
	volatile reg08  SCIDRL;
}sci;	

#define SCI0 (*((volatile sci*)(0x00C8 + REG_BASE)))
#define SCI1 (*((volatile sci*)(0x00D0 + REG_BASE)))
#define SCI2 (*((volatile sci*)(0x00E8 + REG_BASE)))


/*------------------------------------------------------------------------------
	F L A S H 	   
------------------------------------------------------------------------------*/

typedef union {
	uint8 byte;
	struct {
		uint8 FDIVLD : 1;	
		uint8 PRDIV8 : 1;	
		uint8 FDIV5  : 1;
		uint8 FDIV4  : 1;
		uint8 FDIV3  : 1;
		uint8 FDIV2  : 1;
		uint8 FDIV1  : 1;
		uint8 FDIV0  : 1;
	}bit;
}fclkdiv;

enum {	FDIVLD = 0x80,		/* bit masks */
		PRDIV8 = 0x40,		
		FDIV5  = 0x20,
		FIDV4  = 0x10,	
		FDIV3  = 0x08,
		FDIV2  = 0x04,
		FDIV1  = 0x02,
		FDIV0  = 0x01 };

#define FCLKDIV (*((volatile fclkdiv*)(0x0100 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 KEYEN1 : 1;	
		uint8 NV6    : 1;	
		uint8 NV5    : 1;
		uint8 NV4    : 1;
		uint8 NV3    : 1;
		uint8 NV2    : 1;
		uint8 SEC1   : 1;
		uint8 SEC0   : 1;
	}bit;
}fsec;

enum {	KEYEN1 = 0x80,		/* bit masks */
		NV6    = 0x40,		
		NV5    = 0x20,
		NV4    = 0x10,	
		NV3    = 0x08,
		NV2    = 0x04,
		SEC1   = 0x02,
		SEC0   = 0x01 };

#define FSEC (*((volatile fsec*)(0x0101 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 CBEIE  : 1;	
		uint8 CCIE   : 1;	
		uint8 KEYACC : 1;
		uint8        : 5;
	}bit;
}fcnfg;

enum {	CBEIE  = 0x80,		/* bit masks */
		CCIE   = 0x40,		
		KEYACC = 0x20 };

#define FCNFG (*((volatile fcnfg*)(0x0103 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 FPOPEN : 1;	
		uint8 NV6    : 1;	
		uint8 FPHDIS : 1;
		uint8 FPHS1  : 1;
		uint8 FPHS0  : 1;
		uint8 FPLDIS : 1;
		uint8 FPLS1  : 1;
		uint8 FPLS0  : 1;
	}bit;
}fprot;

enum {	FPOPEN = 0x80,		/* bit masks */
		FPHDIS = 0x20,
		FPHS1  = 0x10,	
		FPHS0  = 0x08,
		FPLDIS = 0x04,
		FPLS1  = 0x02,
		FPLS0  = 0x01 };

#define FPROT (*((volatile fprot*)(0x0104 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8 CBEIF  : 1;	
		uint8 CCIF   : 1;	
		uint8 PVIOL  : 1;
		uint8 ACCERR : 1;
		uint8        : 1;
		uint8 BLANK  : 1;
		uint8        : 2;
	}bit;
}fstat;

enum {	CBEIF  = 0x80,		/* bit masks */
		CCIF   = 0x40,		
		PVIOL  = 0x20,
		ACCERR = 0x10,	
		BLANK  = 0x04 };

#define FSTAT (*((volatile fstat*)(0x0105 + REG_BASE)))

typedef union {
	uint8 byte;
	struct {
		uint8       : 1;	
		uint8 CMDB6 : 1;	
		uint8 CMDB5 : 1;
		uint8       : 2;
		uint8 CMDB2 : 1;
		uint8       : 1;
		uint8 CMDB0 : 1;
	}bit;
}fcmd;

enum {	CMDB6 = 0x40,			/* bit masks */
		CMDB5 = 0x20,
		CMDB2 = 0x04,
		CMDB0 = 0x01,
		
		MASS_ERASE   = 0x41,	/* valid commands */
		SECTOR_ERASE = 0x40,
		WORD_PROGRAM = 0x20,
		ERASE_VERIFY = 0x05 };

#define FCMD (*((volatile fcmd*)(0x0106 + REG_BASE)))


/*------------------------------------------------------------------------------
	P O R T  T	   
------------------------------------------------------------------------------*/

#define PTT   (*((volatile reg08*)(0x0240 + REG_BASE)))
#define PTIT  (*((volatile reg08*)(0x0241 + REG_BASE)))
#define DDRT  (*((volatile reg08*)(0x0242 + REG_BASE)))
#define RDRT  (*((volatile reg08*)(0x0243 + REG_BASE)))
#define PERT  (*((volatile reg08*)(0x0244 + REG_BASE)))
#define PPST  (*((volatile reg08*)(0x0245 + REG_BASE)))


/*------------------------------------------------------------------------------
	P O R T  S	   
------------------------------------------------------------------------------*/

#define PTS   (*((volatile reg08*)(0x0248 + REG_BASE)))
#define PTIS  (*((volatile reg08*)(0x0249 + REG_BASE)))
#define DDRS  (*((volatile reg08*)(0x024A + REG_BASE)))
#define RDRS  (*((volatile reg08*)(0x024B + REG_BASE)))
#define PERS  (*((volatile reg08*)(0x024C + REG_BASE)))
#define PPSS  (*((volatile reg08*)(0x024D + REG_BASE)))
#define WOMS  (*((volatile reg08*)(0x024E + REG_BASE)))


/*------------------------------------------------------------------------------
	P O R T  P   [5..0]	   
------------------------------------------------------------------------------*/

#define PTP   (*((volatile reg08*)(0x0258 + REG_BASE)))
#define PTIP  (*((volatile reg08*)(0x0259 + REG_BASE)))
#define DDRP  (*((volatile reg08*)(0x025A + REG_BASE)))
#define RDRP  (*((volatile reg08*)(0x025B + REG_BASE)))
#define PERP  (*((volatile reg08*)(0x025C + REG_BASE)))
#define PPSP  (*((volatile reg08*)(0x025D + REG_BASE)))


/*------------------------------------------------------------------------------
	P O R T  Q	   
------------------------------------------------------------------------------*/

#define PTQ   (*((volatile reg08*)(0x0260 + REG_BASE)))
#define PTIQ  (*((volatile reg08*)(0x0261 + REG_BASE)))
#define DDRQ  (*((volatile reg08*)(0x0262 + REG_BASE)))
#define RDRQ  (*((volatile reg08*)(0x0263 + REG_BASE)))
#define PERQ  (*((volatile reg08*)(0x0264 + REG_BASE)))
#define PPSQ  (*((volatile reg08*)(0x0265 + REG_BASE)))


/*------------------------------------------------------------------------------
	P O R T  U	   
------------------------------------------------------------------------------*/

#define PTU   (*((volatile reg08*)(0x0268 + REG_BASE)))
#define PTIU  (*((volatile reg08*)(0x0269 + REG_BASE)))
#define DDRU  (*((volatile reg08*)(0x026A + REG_BASE)))
#define RDRU  (*((volatile reg08*)(0x026B + REG_BASE)))
#define PERU  (*((volatile reg08*)(0x026C + REG_BASE)))
#define PPSU  (*((volatile reg08*)(0x026D + REG_BASE)))
#define MODRR (*((volatile reg08*)(0x026E + REG_BASE)))

#endif
