
/*------------------------------------------------------------------------------
	netlog.c

	Simple demo application to route uIP log messages
	to an active connection on TCP port 23 (telnet).
	
	Author : Pierre Morency <pmorency@globetrotter.net>
	
	Created: 09.jun.2004
	Revised: 
--------------------------------------------------------------------------------

 Copyright  2004, Pierre Morency.
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of author nor the names of its contributors
      may be used to endorse or promote products derived from this
      software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

------------------------------------------------------------------------------*/

#include "uip.h"
#include "netlog.h"



/*------------------------------------------------------------------------------
	Local private
------------------------------------------------------------------------------*/

enum { BUFFER_SIZE = 300,

	   DORMANT     = 1,		/* application state */
	   CLOSE       = 2,
	   OPENED      = 3,
	   
		IAC  = 255,			/* telnet command codes */
		WILL = 251,
		WONT = 252,
		DO   = 253,
		DONT = 254,
		
		ESC  = 0x27 };
 	   

static char echo_cmd[] = { IAC, WILL, 1, 0 };	/* will echo command */

static char *hello = 

"\r\n\
-------------------------------\r\n\
 Welcome to uIP event logging\r\n\
-------------------------------\r\n\
Press <space> bar to terminate.\r\n";


static char buffer[BUFFER_SIZE];
static uint16 index;
static uint8 state = DORMANT;

static bool netlog_receive(void);



/*------------------------------------------------------------------------------
  Parses received characters from remote end.
  
  It ignores all option commands having the form 'IAC verb option' (see RFC 855),
  all ESC sequences and all data characters except the blank (space) character.
  Returns true when a blank character is found.
------------------------------------------------------------------------------*/

static bool netlog_receive()
{
	uint16 len = uip_datalen();

	while(len-- > 0)
	{
		switch(*uip_appdata++)
		{
			case IAC:
			case WILL:	
			case WONT:
			case DO:
			case DONT:
			case ESC:	return false;
						break;
			case ' ':	return true;
						break;
		}
	}
	return false;
}



/*------------------------------------------------------------------------------
	Public section
------------------------------------------------------------------------------*/

void netlog_open()
{
	uip_listen(HTONS(23));
}


bool netlog_put(const char *msg)
{
	if(state != OPENED)
		return false;
		
	char *ps = &buffer[index];
	while(*msg && index < BUFFER_SIZE)
	{
		/* store message in buffer */
		*ps++ = *msg++;
		index++;
	}
	return true;		
}


/*------------------------------------------------------------------------------
	main event loop for the telnet application
	called by uIP in response to TCP/IP events.	
------------------------------------------------------------------------------*/
   
void netlog_job(void)
{
	if(uip_connected())
	{
		/* allow only one active telnet connection */
		uip_unlisten(HTONS(23));
		state = OPENED;
		index = 0;
		
		/* send logon message */
		netlog_put(hello);
		
		/* send 'will echo' command to force single character mode */
		netlog_put(echo_cmd);
	}
	
	if(state == CLOSE)
	{
		/* application has requested a closure */
		uip_close();		
		state = DORMANT;

		/* allow new connection */
		uip_listen(HTONS(23));
	}
	
	if(uip_acked())
		/* reset pointer to store data */
		index = 0;

	if(uip_newdata() && netlog_receive())
		state = CLOSE;
				
	if(index && (uip_poll() || uip_connected() || uip_rexmit()))
		/* send buffered data */
		uip_send(buffer, index);
}

/*------------------------------------------------------------------------------
	END
------------------------------------------------------------------------------*/
