
/*------------------------------------------------------------------------------
	timers.c

	Author : Pierre Morency <pmorency@globetrotter.net>
	
	Created: 11.may.2004
	Revised: 04.jun.2004 - To use 100ms rate
--------------------------------------------------------------------------------

 Copyright  2004, Pierre Morency.
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of author nor the names of its contributors
      may be used to endorse or promote products derived from this
      software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

------------------------------------------------------------------------------*/

#include "mc9s12e_regs.h"
#include "critical.h"
#include "timers.h"



/*------------------------------------------------------------------------------
	Local private variables
------------------------------------------------------------------------------*/

typedef struct
{
	uint16	count;		/* current value of countdown timer */
	uint16	initval;	/* initial value of timer when timer is set */
	bool	enabled;	/* flag indicating whether timer is enabled */ 

} TIMER_CTRL;


/* table of timers used by this module */
static TIMER_CTRL control[MAX_TIMERS];	

/* module running up time in seconds */
static uint32 uptime = 0;		


/* value is set by timers_open() 
   and cleared by timers_close() */
static uint8 max_timers = 0;	



/*------------------------------------------------------------------------------
	Public section
------------------------------------------------------------------------------*/
								
void timers_service()
{
	static uint8 ticks = 0;

	TIMER_CTRL *p = &control[0];
	uint8 i;
	
	/* decrement all enabled timers */
	for(i = 0; i < max_timers; i++)	
	{
		if(p->enabled && p->count > 0)
			p->count--;			
		p++;
	}
	
	/* track running time */	
	if(++ticks == TICKS_PER_SEC)
	{
		ticks = 0;
		uptime++;					
	}
	
	/* clear interrupt flag and return */
	CRGFLG.bit.RTIF = 1;			
}


void timers_open(uint8 RTIrate)
{
	if(max_timers == 0 && (RTIrate & (RTR6 +RTR5 +RTR4)) != 0)			
	{								
		max_timers = MAX_TIMERS;		/* initialise maximum value. also */
										/* used as a flag to control access */
		TIMER_CTRL *p = &control[0];
		uint8 i;
		for(i = 0; i < max_timers; i++)	/* initialize control table */
		{
			p->count = 0; 
			p->initval = 0;
			p->enabled = false;
			p++;
		}
		RTICTL.byte = RTIrate;			/* set interrupt rate */
		CRGFLG.bit.RTIF = 1;			/* clear interrupt flag */
		CRGINT.bit.RTIE = 1;			/* enable RTI interrupt */
	}
}


uint32 timers_uptime(void)
{
	ENTER_CRITICAL();
	uint32 val = uptime;
	EXIT_CRITICAL();
	return val;
}


void timers_close()
{
	if(max_timers)
	{
		CRGINT.bit.RTIE = 0;				/* disable RTI interrupt */
		RTICTL.byte &= ~(RTR6 +RTR5 +RTR4);	/* disable RTI counter */
		max_timers = 0;						/* and timer module */
		uptime = 0;
	}
}


void timer_set(uint8 no, uint16 x100ms)
{
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		p->initval = x100ms;		/* set this timer with specified */ 
		p->count   = x100ms;		/* number of 1/10 seconds */
		EXIT_CRITICAL();
	}
}


void timer_start(uint8 no)
{
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		p->enabled = true;			/* start this timer */
		EXIT_CRITICAL();
	}
}


void timer_stop(uint8 no)
{
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		p->enabled = false;			/* stop this timer */
		EXIT_CRITICAL();
	}
}


void timer_reload(uint8 no)
{
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		p->count = p->initval;	/* reload the counter immediately */
		EXIT_CRITICAL();
	}
}


uint16 timer_check(uint8 no)
{
	uint16 remain = 0;
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		remain = p->count;	
		EXIT_CRITICAL();
	}
	return remain;		/* return time left before timer expires */	
}

/*------------------------------------------------------------------------------
	END
------------------------------------------------------------------------------*/
