
/*------------------------------------------------------------------------------
	timers.h

	Author : Pierre Morency <pmorency@globetrotter.net>
	
	Created: 11.may.2004
	Revised: 04.jun.2004 - To use 100ms rate
--------------------------------------------------------------------------------

 Copyright  2004, Pierre Morency.
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of author nor the names of its contributors
      may be used to endorse or promote products derived from this
      software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

------------------------------------------------------------------------------*/

#ifndef TIMERS_H
#define TIMERS_H 1

#include "datatypes.h"



/*------------------------------------------------------------------------------
	Configuration options
------------------------------------------------------------------------------*/

enum { MAX_TIMERS = 3,			/* number of timers available */
	   TICKS_PER_SEC = 10 };	/* number of RTI service calls per second */
	   							/* this setting must reflect RTI rate */


/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/


/* This is the timer module interrupt service routine. This address should be
   placed in the vector table at the RTI entry. (see mc9s12e_vectors.s file)  */
 
void timers_service(void) __attribute__((interrupt));



/* Initializes timer module and sets the RTI rate to obtain 100ms tick. 
   RTIrate = 0x7B selects a divisor of (12 x 2**16) for an 8MHz oscillator.
   See Motorola CRG block user guide p.23 */

void timers_open(uint8 RTIrate);



/* Returns the running time of the timer module in seconds */

uint32 timers_uptime(void);



/* Stops the RTI sub-system. 
   Further calls to timer_set(), timer_start(), timer_reset()
   will be ignored and timer_check() will return zero */

void timers_close(void);



/* Select a timer to use (number between 0 and MAX_TIMERS -1)
   and set the time in 1/10s (100ms). To set 36.5 seconds for
   example, use value of 365. Changing the timer value does
   not enable the timer. Countdown is initiated by calling
   timer_start(no) */
         
void timer_set(uint8 no, uint16 x100ms);


/* Initiates the countdown of the selected timer.
   The coundown must be set prior to calling this function
   by either timer_set() or timer_reload(). */
    
void timer_start(uint8 no);


/* Suspends the countdown of selected timer.
   Countdown can be resumed later by calling timer_start() */
   
void timer_stop(uint8 no);


/* Restarts the countdown process of the selected timer to
   its initial value established by timer_set(). If the timer
   is already running, the change is applied immediately */
    
void timer_reload(uint8 no);


/* This function allows to check the countdown progress of the
   selected timer. It returns the time remaining (in 1/10s)
   until the timer expires, or 0 when timer has expired. */ 
   
uint16 timer_check(uint8 no);

#endif
