
/*------------------------------------------------------------------------------
	ethernet.c
		
	Ethernet driver for the MC9S12NE64 MCU selected when EMAC_STATISTICS == 0 
	 	
	Author : Pierre Morency <pmorency@globetrotter.net>

	Created: 25.jun.2004
	Revised: 
--------------------------------------------------------------------------------

 Copyright  2004, Technological Arts <www.technologicalarts.com>
 Copyright  2004, Pierre Morency
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of the copyright holders nor the names of their
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

------------------------------------------------------------------------------*/

#include "mc9s12ne_regs.h"
#include "ethernet.h"

#if EMAC_STATISTICS == 0


/*------------------------------------------------------------------------------
  Private local definitions and variables
------------------------------------------------------------------------------*/

enum { EPHY_ADDRESS = 0,			/* address of attached PHY device */
	   BUS_CLOCK    = 25,			/* bus clock frequency in MHz */
	   RAM_BASE     = 0x2000 };		/* RAM base address used by set_ptr() */
	   
static bool opened = false;

static void *rxa = NULL;
static void *rxb = NULL;
static void *tx  = NULL;


/*------------------------------------------------------------------------------
  Default MAC address for this interface card
  
  The address below comes from an old retired ISA network card. If you need to
  change it, we suggest you recycle a valid MAC address not in use instead of
  improvising a new one from scratch.  
------------------------------------------------------------------------------*/

static MAC_address mac = { 0, 0, 0xC0, 0x42, 0xF9, 0xB2 };


/*------------------------------------------------------------------------------
  Prototypes of private local functions
------------------------------------------------------------------------------*/

static uint16 read_mii(uint8 reg);  
static void write_mii(uint8 reg, uint16 value);
static uint16 set_ptr(uint8 no);


static uint16 read_mii(uint8 reg)
{
	MRADR.byte = reg;
	MCMST.bit.OP = READ;
	while(!(IEVENT.word & MMCIF));
	IEVENT.bit.MMCIF = 1;
	return MRDATA.word;
}


static void write_mii(uint8 reg, uint16 value)
{
	MRADR.byte = reg;
	MWDATA.word = value;
	MCMST.bit.OP = WRITE;
	while(!(IEVENT.word & MMCIF));
	IEVENT.bit.MMCIF = 1;
}

 
/*------------------------------------------------------------------------------
  This function sets RXA, RXB and TX pointers to ram area to be used for
  receive and transmit operations. It uses BUFMAP number as parameter and
  returns the maximum receive frame size allowed for the selected BUFMAP.
------------------------------------------------------------------------------*/
  
static uint16 set_ptr(uint8 no)
{
	static uint16 rx_size[] = { 128, 256, 512, 1024, 1536 };
	
	if(no > 4)
		return 0;
							 
	rxa = (void *) RAM_BASE;
	rxb = (void *)(RAM_BASE + rx_size[no]);
	tx  = (void *)(RAM_BASE + (2 * rx_size[no]));

	return rx_size[no];					  
}


/*------------------------------------------------------------------------------
  Public section
------------------------------------------------------------------------------*/

uint16 ethernet_open(uint8 bufmap, uint8 mode, void (*delay)(uint8), MAC_address *pa)
{
	uint16 maxfl;
	
	if(!opened && (maxfl = set_ptr(bufmap)) != 0)
	{
		opened = true;
		
		/* set EPHY address in both modules */
		EPHYCTL1.byte = EPHY_ADDRESS;
		MPADR.byte = EPHY_ADDRESS;
		
		/* set EPHY module with PHY clocks disabled, LED enabled */ 
		EPHYCTL0.byte = ANDIS | DIS100 | DIS10 | LEDEN;
		if(mode == AUTONEG)
			EPHYCTL0.bit.ANDIS = 0;

		/* enables EPHY module */
		EPHYCTL0.bit.EPHYEN = 1;
		(*delay)(60);
		
		/* initialise EMAC including MII management interface */
		
		while(MCMST.bit.BUSY);			/* wait if operation in progress */
		SWRST.bit.MACRST = 1;			/* EMAC software reset */
		(*delay)(50);
		
		MCMST.bit.MDCSEL = BUS_CLOCK /5;/* set MII management clock rate */	
		BUFCFG.bit.BUFMAP = bufmap;		/* set FIFO configuration */
		BUFCFG.bit.MAXFL = maxfl;		/* set receive maximum frame length */

		ETCTL.byte = ARP + IPV4;		/* accept only ARP and IPV4 messages */
		RXCTS.byte = 0;					/* receive control set to no filtering */
		NETCTL.byte = 0;				/* EMAC operate in wait mode, internal */
										/* PHY, normal operation, half duplex */
		if(mode & 0x01)
		{								/* if full duplex is used */
			NETCTL.bit.FDX = 1;			/* set full duplex mode */
			RXCTS.bit.RFCE = 1;			/* pause frame detection on receive */
		}
					
		/* store MAC address in EMAC module and in passed argument for uIP */
		memcpy((void *)&MACAD.byte[0], &mac, sizeof(MAC_address));
		memcpy(pa, &mac, sizeof(MAC_address));

		/* clear interrupt mask and enable EMAC module */
		IMASK.word = 0;
		NETCTL.bit.EMACE = 1;

		
		if(mode == AUTONEG)
		{
			/* prepare auto-negociate advertisement and make capable of only
			   half duplex mode at both speeds, which is best for uIP */
			   
			uint16 advert = read_mii(4) | BIT7 | BIT5;
			advert &= ~(BIT8 + BIT6);
			write_mii(4, advert);

			EPHYCTL0.byte &= ~(DIS100 + DIS10);	/* start both EPHY PLL clocks */
			while(!(read_mii(1) & BIT5));		/* wait for A/N to complete */
		}
		else
		{
			/* set speed and duplex mode manually */
			
			uint16 control = read_mii(0) & ~(BIT8 | BIT13);	/* half duplex, 10Mbs */
			if(mode & 0x01)
				control |= BIT8;	/* set full duplex */
			if(mode & 0x02)
				control |= BIT13;	/* set 100Mbs */
			write_mii(0, control);

			EPHYCTL0.byte &= ~(DIS100 + DIS10);	/* enables both EPHY PLL clocks */
		}		

		/* clear all event flags */
		IEVENT.word =
			RFCIF|BREIF|RXEIF|RXAOIF|RXBOIF|RXACIF|RXBCIF|MMCIF|LCIF|ECIF|TXCIF;
			
		/* enable LCIE, ECIE and TXCIE transmitter related interrupts */	
		IMASK.word |= LCIE|ECIE|TXCIE;
	}
	
	return maxfl;
}


/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/
void ethernet_close(void)
{
	if(opened)
	{
		opened = false;
		NETCTL.bit.EMACE = 0;	/* disables EMAC */
		IMASK.word = 0;			/* masks all EMAC interrupts */
	}
}				


/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/
uint16 ethernet_get(void *buf, uint16 maxlen)
{
	uint16 len = 0;
	
	if(opened)
	{
		if(IEVENT.bit.RXACIF)
		{
			memcpy(buf, rxa, len = RXAEFP.word +1 -4);	/* get data from buffer A */
			IEVENT.bit.RXACIF = 1;						/* clear event */
			return len;
		}

		if(IEVENT.bit.RXBCIF)
		{
			memcpy(buf, rxb, len = RXBEFP.word +1 -4); /* get data from buffer B */
			IEVENT.bit.RXBCIF = 1;					   /* clear event */
		}
	}
	return len;
}


/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/
void ethernet_put(void *buf, void *appdata, uint16 len, uint16 header_len)
{
	if(opened)
	{
		while(TXCTS.byte & TXACT);		/* wait for transmitter to be idle */

		if(len <= header_len)
			memcpy(tx, buf, len);		/* write the uip buffer alone */
		else
		{	
			memcpy(tx, buf, header_len);	/* write uip buffer + app. data */		
			memcpy(tx +header_len, appdata, len - header_len);	
		}	

		TXEFP.word = len -1;			/* store offset of last byte written */
		TXCTS.bit.TCMD = START;			/* issue a transmit command */
		}
}


/*------------------------------------------------------------------------------
  RX related interrupt services routines.
  
  None of them is used by this module. 
------------------------------------------------------------------------------*/

/* service babbling receive error interrupt */
void emac_BRE_service() {}

/* service receive buffer overrun interrupt */
void emac_RXO_service() {}

/* service receive error interrupt */
void emac_RXE_service() {}

/* service receive flow control interrupt */
void emac_RFC_service() {}


/*------------------------------------------------------------------------------
  TX related interrupt services routines.
  
  Their main task is to clear their related event flag to allow for transmitter
  to keep on working, and for put_ethernet() not to wait for transmission to
  be completed before returning to caller.
------------------------------------------------------------------------------*/

/* service late collision interrupt */
void emac_LC_service()
{
	IEVENT.bit.LCIF = 1;
}

/* service excessive collision interrupt */
void emac_EC_service()
{
	IEVENT.bit.ECIF = 1;
}

/* service frame transmission complete interrupt */
void emac_TXC_service()
{
	IEVENT.bit.TXCIF = 1;
}

#endif /* EMAC_STATISTICS */

/*------------------------------------------------------------------------------
	END
------------------------------------------------------------------------------*/
