
/*------------------------------------------------------------------------------
	ethernet.h
		
	Author : Pierre Morency <pmorency@globetrotter.net>

	Created: 25.jun.2004
	Revised: 
--------------------------------------------------------------------------------

 Copyright  2004, Technological Arts <www.technologicalarts.com>
 Copyright  2004, Pierre Morency
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of the copyright holders nor the names of their
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

------------------------------------------------------------------------------*/

#ifndef ETHERNET_H
#define ETHERNET_H 1

#include "datatypes.h"


/*------------------------------------------------------------------------------
  Compiling option determines which ethernet module get compiled.

  0 compiles the content of ethernet.c driver file.
  1 compiles the content the ethernet_stats.c file which handles statistics.

  Note: both files must be part of the project.
------------------------------------------------------------------------------*/

#define EMAC_STATISTICS 0


/*------------------------------------------------------------------------------
  Prototypes of interrupt service routines. 

  8 interrupt vectors must be set in the mc9s12ne_vectors.s file:
  
  emac_EC_service  at 0xFFA0
  emac_LC_service  at 0xFFA2
  emac_BRE_service at 0xFFA4
  emac_RXO_service at 0xFFA6
  emac_RXO_service at 0xFFA8
  emac_RXE_service at 0xFFAA
  emac_RFC_service at 0xFFAE
  emac_TXC_service at 0xFFB0
	
------------------------------------------------------------------------------*/

/* service excessive collision interrupt */
void emac_EC_service(void) __attribute__((interrupt));

/* service late collision interrupt */
void emac_LC_service(void) __attribute__((interrupt));

/* service babbling receive error interrupt */
void emac_BRE_service(void) __attribute__((interrupt));

/* service receive buffer overrun interrupt */
void emac_RXO_service(void) __attribute__((interrupt));

/* service receive error interrupt */
void emac_RXE_service(void) __attribute__((interrupt));

/* service receive flow control interrupt */
void emac_RFC_service(void) __attribute__((interrupt));

/* service frame transmission complete interrupt */
void emac_TXC_service(void) __attribute__((interrupt));



/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/

/* 48-bit MAC address type for local use */
typedef union		
{
	uint8  byte[6];
	uint16 word[3];

} MAC_address;	


enum {	HALF10  = 0,	/* mode settings with odd number being full duplex */
		FULL10  = 1,
		HALF100 = 2,
		FULL100 = 3,
		AUTONEG = 4 };


/* This function should be called first to enable the ethernet module.
   Caller must specify 'bufmap' to use and communication mode, must provide
   the address of a delay routine and a pointer to a MAC address structure
   which be filled by the ethernet module. Useful 'bufmap' values are:

   2 - for RXA, RXB and TX buffers of 512 bytes,
   3 - for RXA, RXB and TX buffers of 1024 bytes,
   4 - for RXA, RXB and TX buffers of 1536 bytes.
   
   (see MC9S12NE64 data sheet, section 12.3.3.15 FIFO Buffer Configuration)

   NOTE: the size of UIP_BUFSIZE must be smaller or equal to this value.
      	 
   A non zero return value indicates a success, and the value is the size of
   the three buffers assigned by the ethernet driver to RX and TX operations. */
   
uint16 ethernet_open(uint8 bufmap, uint8 mode, void(*delay)(uint8), MAC_address *pa);


/* Calling this function stops the ethernet module from receiving and transmitting.
   A call to ethernet_open() is necessary to restart communication */
   
void ethernet_close(void);				


/* This function returns immediately with zero when no data has been received.
   When data is available, it returns the number of bytes obtained from the
   network and places them in the caller's buffer having 'maxlen' size. */
   
uint16 ethernet_get(void *buf, uint16 maxlen);


/* This function is called to transmit data over the network.
   Data is fetched from both the uip_buffer and the application data buffer */
   
void ethernet_put(void *buf, void *appdata, uint16 len, uint16 header_len);


#if EMAC_STATISTICS

typedef struct {
	uint32 rx_bytes;			/* number of bytes received */
	uint32 rxa_frames;			/* ethernet frames received by RXA */
	uint32 rxb_frames;			/* ethernet frames received by RXB */
	uint32 pause_frames;		/* pause frames received */
	uint32 buffer_overruns;		/* number of time frame data > buffer */
	uint32 receive_errors;		/* receive errors */
	uint32 babbling_errors;		/* babbling receive errors */
	
	uint32 tx_bytes;			/* number of bytes transmitted */
	uint32 tx_frames;			/* ethernet frames transmitted */
	uint32 late_collisions;		/* number of late collisions */
	uint32 exces_collisions;	/* number of excessive collisions */

} EMAC_STATS;


/* This function fills an EMAC_STATS structure with data accumulated 
   from the time ethernet_open() is called. If a NULL pointer is passed,
   the statistics are reset to zero. */
   
void ethernet_stats(EMAC_STATS *p);

#endif /* EMAC_STATISTICS */

#endif

/*------------------------------------------------------------------------------
	END
------------------------------------------------------------------------------*/


