
/*------------------------------------------------------------------------------
	main.c
	
	uIP main control loop

	Author : Pierre Morency <pmorency@globetrotter.net>
	
	Created: 25.jun.2004
	Revised: 20.jul.2004 - Release of uIP HCS12NE V1.0
--------------------------------------------------------------------------------

 Copyright  2004, Pierre Morency.
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of author nor the names of its contributors
      may be used to endorse or promote products derived from this
      software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

------------------------------------------------------------------------------*/

#include "mc9s12ne_regs.h"
#include "ethernet.h"
#include "critical.h"
#include "timer.h"

#include "uip.h"
#include "uip_arp.h"



/*------------------------------------------------------------------------------
	Prototypes of local functions 
------------------------------------------------------------------------------*/

#if UIP_LOGGING != 0
void uip_log(char *msg);
#endif

static void transmit(void);



/*------------------------------------------------------------------------------
  This function is called by uIP whenever a log message is generated.
  It is enabled by setting the UIP_LOGGING option to 1, and it uses
  SCI channel 1 (at 57600 bauds) in polling mode or TCP port 23 (telnet)
  when an active connection is established on the port.
------------------------------------------------------------------------------*/

#if UIP_LOGGING != 0

void uip_log(char *msg)
{
	static bool initialized = false;
	
	if(!initialized)
	{
		SCI1.SCIBD.word = 27;			/* set baud rate to 57600 */
		SCI1.SCICR1.byte = 0;			/* normal, 8-N-1 */
		uint8 x = SCI1.SCIDRL.byte;		/* read any left over character */
		SCI1.SCICR2.byte |= TE + RE;	/* enable transmitter and receiver */
		initialized = true;
	}

	/* use TCP port 23 */
	if(netlog_put("\r\n"))
		netlog_put(msg);
	else
	{	
		/* or SCI channel 1 */
		while(!SCI1.SCISR1.bit.TDRE);
		SCI1.SCIDRL.byte = '\r';
		while(!SCI1.SCISR1.bit.TDRE);
		SCI1.SCIDRL.byte = '\n';

		while(*msg)
		{
			while(!SCI1.SCISR1.bit.TDRE);
			SCI1.SCIDRL.byte = *msg++;
		}
	}
}
#endif


/*------------------------------------------------------------------------------
	P R E - M A I N
------------------------------------------------------------------------------*/
void __premain()
{
	/* configures PLL to run chip at 25MHz (OSCCLK is 25MHz)
	   and enables interrupts before leaving */
	
	SEI();						/* disable interrupts for now */
	CLKSEL.bit.PLLSEL = 0;		/* system clock now derived from OSCCLK */
	PLLCTL.bit.PLLON = 0;		/* turn off the PLL */
	SYNR.byte = 0;				/* set PLL multiplier */
	REFDV.byte = 0;				/* set PLL divider */
	PLLCTL.bit.PLLON = 1;		/* turn on PLL */
	NOP();
	NOP();
	NOP();
	NOP();
	while(!CRGFLG.bit.LOCK);	/* wait for PLL to lock */
	CLKSEL.bit.PLLSEL = 1;		/* switch clock source to PLL */
	CLI();						/* enables interrupts */
	
	PPAGE.byte = 0x3D;			/* page = $3D for continuous 46K memory block */
}


/*------------------------------------------------------------------------------
	Transmit a packet	
------------------------------------------------------------------------------*/
static void transmit()
{
	ethernet_put(&uip_buf[0],(void *)uip_appdata,
					uip_len, UIP_LLH_LEN + UIP_TCPIP_HLEN);
}


/*------------------------------------------------------------------------------
	M A I N
------------------------------------------------------------------------------*/

#define BUF ((struct uip_eth_hdr *)&uip_buf[0])

int main()
{
	uip_log("NE64 on");
	
	/* enable software timers module */
	timers_open(7, 19531);
	
	/* init the uIP TCP/IP stack */
	uip_init();
	uip_sethostaddr(uip_ipaddr(192,168,10,20));
	uip_setnetmask (uip_ipaddr(255,255,255,0));
	uip_setdraddr  (uip_ipaddr(192,168,10,10));
	
	/* open ethernet driver and obtain MAC address */	
	ethernet_open(4, AUTONEG, delay_ms, (MAC_address *)&uip_ethaddr);
	
	/* init ARP cache */
	uip_arp_init();
	
	/* init application (see netlog.c) */
	netlog_open();
	
	/* set timer #0 for 0.5 second period 
	   and timer #1 for 10 seconds period */
	timer_set(0, 5); 
	timer_start(0);
	timer_set(1, 100);
	timer_start(1);


	/* enter main control loop */
	for(;;)
	{
		/* call ethernet receiver */
		uip_len = ethernet_get(&uip_buf[0], UIP_BUFSIZE);
		
		if(uip_len == 0)
		{
			if(timer_check(0) == 0)
			{
				/* call periodic function for each connection */
				timer_reload(0);
				uint8 i;
				for(i = 0; i < UIP_CONNS; i++)
				{
					uip_periodic(i);
					if(uip_len > 0)
					{
						uip_arp_out();
						transmit();
					}
				}
			}
			if(timer_check(1) == 0)
			{
				/* call the ARP timer function every 10 seconds
				   to flush old entries in the ARP table */
				timer_reload(1);
				uip_arp_timer();
			}
		}
		else
		{
			if(BUF->type == htons(UIP_ETHTYPE_IP))
			{
				/* add the source to the ARP cache and set
				   the ethernet packet length before processing */
				uip_arp_ipin();
				uip_input();
				if(uip_len > 0)
				{
					uip_arp_out();
					transmit();
				}
			}
			else if(BUF->type == htons(UIP_ETHTYPE_ARP))
			{
				/* process an ARP packet */
				uip_arp_arpin();
				if(uip_len > 0)
					transmit();
			}
		}
	}

	return 0;
}

/*------------------------------------------------------------------------------
	END
------------------------------------------------------------------------------*/
