
/*------------------------------------------------------------------------------
	timer.c

	Author : Pierre Morency <pmorency@globetrotter.net>
	
	Created: 10.jul.2004 - Module to replace timers.c
					     - Uses TIM channel 4 for software timers,
					       and TIM channel 5 for delay_ms() routine.
	Revised: 
--------------------------------------------------------------------------------

 Copyright  2004, Pierre Morency.
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of author nor the names of its contributors
      may be used to endorse or promote products derived from this
      software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

------------------------------------------------------------------------------*/

#include "mc9s12ne_regs.h"
#include "critical.h"
#include "timer.h"



/*------------------------------------------------------------------------------
	Local private definitions, variables and prototypes
------------------------------------------------------------------------------*/

typedef struct
{
	uint16	count;				/* current value of countdown timer */
	uint16	initval;			/* initial value of timer when timer is set */
	bool	enabled;			/* flag indicating whether timer is enabled */ 

} TIMER_CTRL;

static TIMER_CTRL control[MAX_TIMERS];	/* table of timers offered by module */

static uint32 uptime;			/* total running up time in seconds */
   
static uint8 max_timers = 0;	/* value is set by timers_open() 
								   and cleared by timers_close() */

static uint16 oc4count;			/* count to reload on each 100msec interrupt */
static uint16 oc5count;			/* count to be used by delay_ms routine */

enum { TICKS_PER_SEC = 10 };	/* number of interrupt service calls per second */

void timers_service(void) __attribute__((interrupt));



/*------------------------------------------------------------------------------
  The timer module interrupt service routine. It's address should be placed in
  the vector table at the 'timer channel 4' entry.(mc9s12ne_vectors.s file). 
------------------------------------------------------------------------------*/
								
void timers_service()
{
	static uint8 ticks = 0;
	
	TIM.TC4.word += oc4count;	/* set new compare count and clear int. flag */ 

	uint8 i;
	TIMER_CTRL *p = &control[0];
	for(i = 0; i < max_timers; i++)		/* decrement all enabled timers */
	{
		if(p->enabled && p->count > 0)
			p->count--;			
		p++;
	}
	
	if(++ticks == TICKS_PER_SEC)		/* track module running time */
	{
		ticks = 0;
		uptime++;					
	}
}


/*------------------------------------------------------------------------------
	Public section
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Initializes the timer module and sets the OC4 (output compare 4) function to
  generate an interrupt every 100ms. Must be called with a prescaler value and
  an output compare count to obtain 100ms rate. This same count is also divided
  by 100 to be used as base count for the delay_ms() function.
  
  (see Motorola TIM16B4 p.19)
------------------------------------------------------------------------------*/

void timers_open(uint8 prescaler, uint16 count)
{
	if(max_timers == 0)
	{
		uptime = 0;
		max_timers = MAX_TIMERS;				/* initialise maximum value; also */
												/* used as a flag to control access */
		TIMER_CTRL *p = &control[0];
		uint8 i;
		for(i = 0; i < max_timers; i++)			/* initialize control table */
		{
			p->count = 0; 
			p->initval = 0;
			p->enabled = false;
			p++;
		}

		TIM.TSCR2.byte |= (prescaler & 0x07);	/* set prescaler */
		oc4count = count;						/* save count to reload on each int. */
		oc5count = count /100;					/* save count for delay_ms routine */
		
		TIM.TIOS.byte |= IOS5 + IOS4;			/* uses OC5 and OC4 functions */
		TIM.TCTL1.byte &= ~(OL5 +OM5 +OL4 +OM4);/* disconnect from output pins */
		TIM.TSCR1.byte |= TEN + TFFCA;			/* ena. TCNT + fast timer flag clear */
		
		TIM.TFLG1.byte |= C5F + C4F;			/* clear output 5 and 4 compare flags */
		TIM.TC4.word = TIM.TCNT.word + count;	/* start an OC4 action */
		TIM.TIE.bit.C4I = 1;					/* set interrupt on OC4 compare */
	}
}


/*------------------------------------------------------------------------------
  Returns the running time of the timer module in seconds
------------------------------------------------------------------------------*/

uint32 timers_uptime(void)
{
	ENTER_CRITICAL();
	uint32 val = uptime;
	EXIT_CRITICAL();
	return val;
}


/*------------------------------------------------------------------------------
  Stops this timer module. Further calls to timer_set(), timer_start(),
  timer_reset() will be ignored, timer_check() will return zero and 
  timer_delay_ms() will return immediately 
------------------------------------------------------------------------------*/

void timers_close()
{
	if(max_timers)
	{
		TIM.TIE.bit.C4I = 0;	/* disable interrupt on OC4 compare */
		max_timers = 0;			/* and timer module */
	}
}


/*------------------------------------------------------------------------------
  Selects a timer to use (number between 0 and MAX_TIMERS -1) and sets the time
  in 1/10s (100ms). To set 36.5 seconds for example, use value of 365. Changing
  the timer value does not enable the timer. Countdown is initiated by calling
  timer_start(no)
------------------------------------------------------------------------------*/

void timer_set(uint8 no, uint16 x100ms)
{
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		p->initval = x100ms;		/* set this timer with specified */ 
		p->count   = x100ms;		/* number of 1/10 seconds */
		EXIT_CRITICAL();
	}
}


/*------------------------------------------------------------------------------
  Initiates the countdown of the selected timer. The coundown must be set
  prior to calling this function by either timer_set() or timer_reload().
------------------------------------------------------------------------------*/

void timer_start(uint8 no)
{
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		p->enabled = true;			
		EXIT_CRITICAL();
	}
}


/*------------------------------------------------------------------------------
  Suspends the countdown of selected timer.
  Countdown can be resumed later by calling timer_start().
------------------------------------------------------------------------------*/

void timer_stop(uint8 no)
{
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		p->enabled = false;			
		EXIT_CRITICAL();
	}
}


/*------------------------------------------------------------------------------
  Restarts the countdown process of the selected timer to its initial value
  established by timer_set(). If the timer is currently running, the change
  is applied immediately. 
 ------------------------------------------------------------------------------*/

void timer_reload(uint8 no)
{
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		p->count = p->initval;		
		EXIT_CRITICAL();
	}
}


/*------------------------------------------------------------------------------
  This function allows to check the countdown progress of the selected timer.
  It returns the time remaining (in 1/10s) until the timer expires, or 0
  when timer has expired. 
------------------------------------------------------------------------------*/

uint16 timer_check(uint8 no)
{
	uint16 remain = 0;
	if(no < max_timers)
	{
		TIMER_CTRL *p = &control[no];
		ENTER_CRITICAL();
		remain = p->count;	
		EXIT_CRITICAL();
	}
	return remain;			
}


/*------------------------------------------------------------------------------
  This function offer a delay in the range of 0 to 100ms.
------------------------------------------------------------------------------*/

void delay_ms(uint8 ms)
{
	if(ms > 0 && max_timers)
	{
		if(ms > 100) ms = 100;							/* set to max allowed */
		TIM.TC5.word = TIM.TCNT.word + (ms * oc5count);	/* set delay in ms */
		while(TIM.TFLG1.bit.C5F == 0);					/* wait for event */
	}
}

/*------------------------------------------------------------------------------
	END
------------------------------------------------------------------------------*/
