// filename  ***************  LCD.C ****************************** 
// LCD Display (HD44780) on Port AD,M of the 9S12C32   
// Jonathan W. Valvano 8/15/06
// Modified for Tech Arts Esduino 5/25/2011

//  This example accompanies the books
//   "Embedded Microcomputer Systems: Real Time Interfacing",
//        Thomson Engineering, copyright (c) 2006,
//   "Introduction to Embedded Microcomputer Systems: 
//    Motorola 6811 and 6812 Simulation", Brooks-Cole, copyright (c) 2002

// Copyright 2006 by Jonathan W. Valvano, valvano@mail.utexas.edu 
//    You may use, edit, run or distribute this file 
//    as long as the above copyright notice remains 

/*   
  size is 1*16 
  ground = pin 1    Vss
  power  = pin 2    Vdd   +5V
  ground = pin 3    Vlc   grounded for highest contrast
  PM5    = pin 4    RS    (1 for data, 0 for control/status) => DIG13
  ground = pin 5    R/W   (0 for write)
  PM4    = pin 6    E     (enable) =>DIG11 (ensure that JB3 is at 1-2 position)
     (4-bit data)
  PM3    = pin 14   DB7 => DIG10 (ensure that JB2 is at 2-3 position)
  PM2    = pin 13   DB6 => DIG13
  PM1    = pin 12   DB5 => located on pin 1 of J8
  PM0    = pin 11   DB4 => located on pin 2 of J8
*/

#include <mc9s12c32.h>  /* derivative information */
#include "LCD.H"

static unsigned short OpenFlag=0;

//---------------------wait---------------------
// time delay
// Input: time in 0.667usec
// Output: none
void static wait(unsigned short delay){ 
unsigned short startTime;
  startTime = TCNT;
  while((TCNT-startTime) <= delay){}  
}
  
//---------------------outCsr---------------------
// sends one command code to the LCD control/status
// Input: command is 8-bit function to execute
// Output: none
static void outCsr(unsigned char command){
  PTM = 0x0F&(command>>4); // ms nibble, E=0, RS=0
  PTM |= 0x10;             // E goes 0,1
  PTM &= ~0x10;            // E goes 1,0
  PTM = command&0x0F;      // ls nibble, E=0, RS=0
  PTM |= 0x10;             // E goes 0,1
  PTM &= ~0x10;            // E goes 1,0
  wait(135);               // blind cycle 90 us wait
}

//---------------------LCD_Clear---------------------
// clear the LCD display, send cursor to home
// Input: none
// Output: true if successful
short LCD_Clear(void){
  if(OpenFlag==0){
    return 0;  // not open
  }
  outCsr(0x01);        // Clear Display
  wait(2460);          // 1.64ms wait
  outCsr(0x02);        // Cursor to home
  wait(2460);          // 1.64ms wait
  return 1;		         // success
}

//---------------------LCD_Open---------------------
// initialize the LCD display, called once at beginning
// Input: display determines increment and shift option
//        cursor determines cursor options
//        move determines cursor movement
//        size sets display size
// Output: true if successful
short LCD_Open(char display, char cursor, char move, char size){
int i;
  if(OpenFlag){
    return 0;      // error if already open
  }
  DDRAD |= 0x01;   // PortAD bit 0 is output to R/W
  ATDDIEN |= 0x01; // enable AD bit 0 digital driver  
  PTAD &= ~0x01;   // PAD0=R/W=0 means write
  DDRM = 0x3F;     // PM5-0 output to LCD
  TSCR1 = 0x80;   // Enable TCNT, 24MHz boot mode, 4MHz in run mode
  TSCR2 = 0x04;   // divide by 16 TCNT prescale, TCNT at 667nsec
  PACTL = 0;      // timer prescale used for TCNT
/* Bottom three bits of TSCR2 (PR2,PR1,PR0) determine TCNT period
    divide  at 24MHz    
000   1     42ns  TOF  2.73ms  
001   2     84ns  TOF  5.46ms  
010   4    167ns  TOF  10.9ms      
011   8    333ns  TOF  21.8ms 	 
100  16    667ns  TOF  43.7ms 	 
101  32   1.33us  TOF  87.4ms		 
110  64   2.67us  TOF 174.8ms   
111 128   5.33us  TOF 349.5ms    */ 
// Be careful, TSCR1 and TSCR2 maybe set in other rituals
  for(i=100;i;i--){
    wait(1500);    // 100ms to allow LCD powerup
  }
  PTM = 0x02;		 // DL=0 4-bit, RS=0 (command), E=0
  PTM |= 0x10;   // E goes 0,1
  PTM &= ~0x10;  // E goes 1,0
  wait(135);     // 90 us wait
  outCsr(0x04+display); // I/D Increment, S displayshift
  outCsr(0x0C+cursor);  // D=1 displayon, C cursoron/off, B blink on/off
  outCsr(0x14+move);    // S/C cursormove, R/L shiftright
  outCsr(0x20+size);    // DL=0 4bit, N 1/2 line, F 5by7 or 5by10
  OpenFlag = 1;         // device open
  return LCD_Clear();   // clear display
}

//---------------------LCD_OutChar---------------------
// sends one ASCII to the LCD display
// Input: letter is ASCII code
// Output: true if successful
short LCD_OutChar(unsigned char letter){
  if(OpenFlag==0){
    return 0;  // not open
  }
  PTM = 0x20+(0x0F&(letter>>4));   // ms nibble, E=0, RS=1
  PTM |= 0x10;       // E goes 0,1
  PTM &= ~0x10;      // E goes 1,0
  PTM = 0x20+(letter&0x0F);        // ls nibble, E=0, RS=1
  PTM |= 0x10;       // E goes 0,1
  PTM &= ~0x10;      // E goes 1,0
  wait(135);         // 90 us wait
  return 1;	         // success
}

//---------------------LCD_OutString--------------
// Display String
// Input: pointer to NULL-terminationed ASCII string 
// Output: true if successful
short LCD_OutString(char *pt){ 
  if(OpenFlag==0){
    return 0;  // not open
  }
  while(*pt){
    if(LCD_OutChar((unsigned char)*pt)==0){
      return 0;
    }
    pt++;
  }
  return 1;	  // success
}

//-----------------------LCD_OutDec-----------------------
// Output a 16-bit number in unsigned decimal format
// Input: 16-bit unsigned number 
// Output: true if successful
// fixed size 3 digits of output, right justified 
short LCD_OutDec(unsigned short n){
  if(OpenFlag==0){
    return 0;  // not open
  }
  if(n < 10){
    if(LCD_OutString("    ")==0){
      return 0;
    }
    if(LCD_OutChar(n+'0')==0){ /* n is between 0 and 9 */
      return 0;
    }
  } else if(n<100){
    if(LCD_OutString("   ")==0){
      return 0;
    }
    if(LCD_OutChar(n/10+'0')==0){ /* tens digit */
      return 0;
    }
    if(LCD_OutChar(n%10+'0')==0){ /* ones digit */
      return 0;
    }
  } else if(n<1000){
      if(LCD_OutString("  ")==0){
        return 0;
      }
      if(LCD_OutChar(n/100+'0')==0){ /* hundreds digit */
        return 0;
      }
      n = n%100;
      if(LCD_OutChar(n/10+'0')==0){ /* tens digit */
        return 0;
      }
      if(LCD_OutChar(n%10+'0')==0){ /* ones digit */
        return 0;
      }
    }
    else if(n<10000){
      if(LCD_OutChar(' ')==0){  
        return 0;
      }
      if(LCD_OutChar(n/1000+'0')==0){ /* thousands digit */
        return 0;
      }
      n = n%1000;
      if(LCD_OutChar(n/100+'0')==0){ /* hundreds digit */
        return 0;
      }
      n = n%100;
      if(LCD_OutChar(n/10+'0')==0){ /* tens digit */
        return 0;
      }
      if(LCD_OutChar(n%10+'0')==0){ /* ones digit */
       return 0;
      } 
    }
    else {
      if(LCD_OutChar(n/10000+'0')==0){ /* ten-thousands digit */
        return 0;
      }
      n = n%10000;
      if(LCD_OutChar(n/1000+'0')==0){ /* thousands digit */
        return 0;
      }
      n = n%1000;
      if(LCD_OutChar(n/100+'0')==0){ /* hundreds digit */
        return 0;
      }
      n = n%100;
      if(LCD_OutChar(n/10+'0')==0){ /* tens digit */
        return 0;
      }
      if(LCD_OutChar(n%10+'0')==0){ /* ones digit */
       return 0;
      }
    }
  return 1;
}


//-----------------------LCD_GoTo-----------------------
// Move cursor
// Input: line number is 1 to 4, column from 1 to 20 
// Output: true if successful
// errors: it will check for legal address
short LCD_GoTo(int line, int column){
  unsigned char DDaddr;
  if(OpenFlag==0){
    return 0;  // not open
  }
  if((line<1) || (line>4)) return 0;
  if((column<1) || (column>20)) return 0;
  if(line==1) DDaddr = column-1;
  if(line==2) DDaddr = 0x3F+column;
  if(line==3) DDaddr = 0x13+column;
  if(line==4) DDaddr = 0x53+column;
  DDaddr += 0x80;
  outCsr(DDaddr);
  return 1;
}

//-----------------------LCD_SetFont-----------------------
// Define a new character font
// Input: font is the font number, n0-n7 is the 5 by 8 pixel map 
// Output: true if successful
// errors: font is 0 to 7
short LCD_SetFont(unsigned char font, unsigned char n0, 
unsigned char n1, unsigned char n2, unsigned char n3, 
unsigned char n4, unsigned char n5,unsigned char n6, 
unsigned char n7){ unsigned char CGaddr;
  if(OpenFlag==0){
    return 0;  // not open
  }
  if(font>7){
    return 0;  // only 0 to 7
  }
	CGaddr=((font&0x07)<<3)+0x40;
	outCsr(CGaddr); // set CG RAM address
	LCD_OutChar(n0);
	LCD_OutChar(n1);
	LCD_OutChar(n2);
	LCD_OutChar(n3);
	LCD_OutChar(n4);
	LCD_OutChar(n5);
	LCD_OutChar(n6);
	LCD_OutChar(n7);
	outCsr(0x80); // revert back to DD RAM
  return 1;
}


