// filename ******** Main.C ************** 
// LCD Display (HD44780) on Port PAD,M of the 9S12C32   
// Jonathan W. Valvano 1/14/07
// Purpose: test program for 4-bit LCD.C driver
// revised for Tech Arts Esduino 5/25/2011

/*   
  size is 1*16 
  ground = pin 1    Vss
  power  = pin 2    Vdd   +5V
  ground = pin 3    Vlc   grounded for highest contrast
  PM5    = pin 4    RS    (1 for data, 0 for control/status) => DIG13
  ground = pin 5    R/W   (0 for write)
  PM4    = pin 6    E     (enable) =>DIG11 (ensure that JB3 is at 1-2 position)
     (4-bit data)
  PM3    = pin 14   DB7 => DIG10 (ensure that JB2 is at 2-3 position)
  PM2    = pin 13   DB6 => DIG13
  PM1    = pin 12   DB5 => located on pin 1 of J8
  PM0    = pin 11   DB4 => located on pin 2 of J8
16 characters are configured as 2 rows of 8
addr  00 01 02 03 04 05 06 07 40 41 42 43 44 45 46 47
*/
#include <hidef.h>      /* common defines and macros */
#include <mc9s12c32.h>  /* derivative information */
#include "LCD.H"
#include "PLL.H"


#pragma LINK_INFO DERIVATIVE "mc9s12c32"

//--------------------TimerInit---------------
// initialize timer module to 0.667us(Boot Mode) TCNT clock
// inputs: none
// outputs: none
void TimerInit(void){
  DDRT |= 0x03;   // PortT bits 0,1  are outputs to optional LED
  TSCR1 = 0x80;   // Enable TCNT, 24MHz in both boo tand run modes
  TSCR2 = 0x04;   // divide by 16 TCNT prescale, TCNT at 667nsec
  PACTL = 0;      // timer prescale used for TCNT
/* Bottom three bits of TSCR2 (PR2,PR1,PR0) determine TCNT period
    divide  24MHz E clock    
000   1     42ns  TOF  2.73ms  
001   2     84ns  TOF  5.46ms   
010   4    167ns  TOF  10.9ms       
011   8    333ns  TOF  21.8ms 	 
100  16    667ns  TOF  43.7ms 	 
101  32   1.33us  TOF  87.4ms		 
110  64   2.67us  TOF 174.8ms   
111 128   5.33us  TOF 349.5ms    */ 
// Be careful, TSCR1 and TSCR2 maybe set in other rituals
}

//---------------------mwait---------------------
// wait specified number of msec
// Input: number of msec to wait
// Output: none
// assumes TCNT timer is running at 667ns
void mwait(unsigned short msec){ 
unsigned short startTime;
  for(; msec>0; msec--){
    startTime = TCNT;
    while((TCNT-startTime) <= 1500){} 
  }
}

//---------------------check---------------------
// if LCD broken toggle LED1 at 2Hz
// Input: last LCD status, 0 means bad
// Output: none
// Error: if status is zero, this function will not return
void check(short status){	 // 0 if LCD is broken
  if(status ==0){		   
    for(;;) {
      PTT ^= 0x01;   // fast toggle LED
      mwait(250);    // 0.25 sec wait
    }
  }
}
//---------------------swait---------------------
// wait specified 2 sec, then clear LCD
// Input: none
// Output: none
// uses mswait and TCNT timer
void swait(void){			
  PTT ^= 0x01;     // toggle LED0
  mwait(2000);     // 2 sec wait
  check(LCD_Clear());
}

//---------------------blank---------------------
// move cursor to second half of LCD display
// 32 spaces from address 08 to 40
// Input: none
// Output: none
void blank(void){
  check(LCD_OutString("                                "));
}

void main(void) {
  PLL_Init();       // set E clock to 24 MHz
  TimerInit();      // enable Port T 
  check(LCD_Open(LCDINC+LCDNOSHIFT,
    LCDNOCURSOR+LCDNOBLINK,
    LCDNOSCROLL+LCDLEFT,
    LCD2LINE+LCD7DOT));
  check(LCD_OutString("9S12C32 ")); blank();
  check(LCD_OutString("Esduino ")); swait();

  for(;;) {
    check(LCD_OutString("ABCDEFGH")); blank();
    check(LCD_OutString("IJKLMNOP")); swait();
    check(LCD_OutString("01234567")); blank();
    check(LCD_OutString("890,./<>")); swait();
    check(LCD_OutString("abcdefgh")); blank();
    check(LCD_OutString("ijklmnop")); swait();
    check(LCD_OutString("!@#$%^&*")); blank();
    check(LCD_OutString("()_+-=[]")); swait();
  } 
}