/////////////////////////////////////////////////////////////////////////////////////////
//
// LIN sample for Freescale TWR-S12G128
//
// --------------------------------------------------------------------------------------
/////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////////////
//
// NOTES:
//
// - LIN bus communication is set to work about at 20Kb/s.
// - Only standard checksum is used.
//
/////////////////////////////////////////////////////////////////////////////////////////

#include "mc9s12g128.h"
#include "lin.h"

#pragma MESSAGE DISABLE C12056 

/////////////////////////////////////////////////////////////////////////////////////////
// Variables
/////////////////////////////////////////////////////////////////////////////////////////

unsigned volatile char *lin_periph[MAX_LIN] = 
{
  &SCI0BDH,
  &SCI1BDH,
  &SCI2BDH,
};

struct frame rx[MAX_LIN];

/////////////////////////////////////////////////////////////////////////////////////////
// LINInit
// --------------------------------------------------------------------------------------
// SCI Peripheral Initialization
/////////////////////////////////////////////////////////////////////////////////////////
void LINInit(unsigned char lin_num)
{

  unsigned volatile char *lin_pt;
  int sbr, i, j;

  lin_pt = lin_periph[lin_num];
  
  // Calculates SCI baud rate value
  sbr=(SCI_CLOCK/SCI_BAUDRATE/16);          
  lin_pt[SCIBDH]=(unsigned char)(sbr>>8);
  lin_pt[SCIBDL]=(unsigned char)sbr;
  // Enables RDRF and OR interrupt and enables. SCI transmitter and receiver
  // Break character is 13 or 14 bit long
  lin_pt[SCICR2]=0x2C;   
  lin_pt[SCISR2]=0x04;                      
  
  // Initializes receive LIN frames
  for (i=0;i<MAX_LIN;i++) 
  {
    rx[i].protected_id=0;
    rx[i].state=IDLE;
    rx[i].error=0;
    rx[i].check=0;
    for (j=0;j<MAX_DATA;j++)
      rx[i].data[j]=0;
  }
    
}

/////////////////////////////////////////////////////////////////////////////////////////
// LINCalcParity
// --------------------------------------------------------------------------------------
// Parity LIN Calculation Routine
/////////////////////////////////////////////////////////////////////////////////////////
unsigned char LINCalcParity(unsigned char id)
{

  unsigned char parity, p0,p1;
  
  parity=id;
  p0=(BIT(parity,0)^BIT(parity,1)^BIT(parity,2)^BIT(parity,4))<<6;
  p1=(~(BIT(parity,1)^BIT(parity,3)^BIT(parity,4)^BIT(parity,5)))<<7;
  parity|=(p0|p1);
  return parity;
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// LINCalcChecksum
// --------------------------------------------------------------------------------------
// Checksum LIN Calculation Routine
/////////////////////////////////////////////////////////////////////////////////////////
unsigned char LINCalcChecksum(unsigned char *data)
{

  unsigned int sum = 0;
  unsigned char i;

  for(i = 0; i < MAX_DATA; i++)
  {
    sum += data[i];
    if(sum&0xFF00)
      // Add carry                         
      sum = (sum&0x00FF) + 1;
  }
  sum ^= 0x00FF;
  return((unsigned char)sum);
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// LINGetChar
// --------------------------------------------------------------------------------------
// SCI data receive routine
/////////////////////////////////////////////////////////////////////////////////////////
Bool LINGetChar(unsigned char lin_num)
{

  unsigned volatile char *lin_pt, ch;

  lin_pt = lin_periph[lin_num];
    
  // State of the LIN receive channel
  switch(rx[lin_num].state++) 
  {
    case IDLE:
      //Clear RDRF or OR SCI Flag 
      if(!(lin_pt[SCISR1]&0x22))
        return(FALSE);
      if(lin_pt[SCIDRL])
        return(FALSE);
      break;
    case _BREAK:
      //Clear RDRF Flag 
      if(!(lin_pt[SCISR1]&0x20))
        return(FALSE);
      if(lin_pt[SCIDRL] != 0x55)
        return(FALSE);
      break;  
    case SYNCH:
      //Clear RDRF Flag 
      if(!(lin_pt[SCISR1]&0x20))
        return(FALSE);    
      ch = lin_pt[SCIDRL];
      rx[lin_num].protected_id = ch;
      break;
      
    case PROTECTED_IDENTIFIER:
    case DATA_0:
    case DATA_1:
    case DATA_2:
    case DATA_3:
    case DATA_4:
    case DATA_5:
    case DATA_6:
      //Clear RDRF Flag 
      if(!(lin_pt[SCISR1]&0x20))
        return(FALSE);    
      ch = lin_pt[SCIDRL];
      rx[lin_num].data[rx[lin_num].state-DATA_0] = ch;
      if((rx[lin_num].state-DATA_0+1) == MAX_DATA)
        rx[lin_num].state = DATA_7;
      break;
    case DATA_7:
      //Clear RDRF Flag 
      if(!(lin_pt[SCISR1]&0x20))
        return(FALSE);    
      ch = lin_pt[SCIDRL];
      rx[lin_num].check = ch;
      break;
    case CHECKSUM:
      return(FALSE);
  }
  return(TRUE);
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// LINGetMsg
// --------------------------------------------------------------------------------------
// LIN Message Receive Routine
/////////////////////////////////////////////////////////////////////////////////////////
Bool LINGetMsg(unsigned char lin_num, Bool get_data, struct  message *msg)
{
  
  unsigned char i;
  
  if(rx[lin_num].state < PROTECTED_IDENTIFIER)
    return(FALSE);
  msg->identifier = (rx[lin_num].protected_id&0x3F);
  if (get_data)
  {
    if(rx[lin_num].state == CHECKSUM) 
    {
      for(i = 0; i < MAX_DATA; i++)
        msg->data_field[i] = rx[lin_num].data[i];
      rx[lin_num].state = IDLE;
    }
    else
      for(i = 0; i < MAX_DATA; i++)
        msg->data_field[i] = 0;
  }
  return TRUE; 
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// LINCheckSend
// --------------------------------------------------------------------------------------
// Sent byte field check routine
/////////////////////////////////////////////////////////////////////////////////////////
Bool LINCheckSend(unsigned char lin_num, enum lin_state status, unsigned char val)
{

  // While until sci data has been received 
  while(rx[lin_num].state < status)
    if(rx[lin_num].error)
      return(FALSE);
    
  switch(status) 
  {
    case _BREAK:
    case SYNCH:
      break;
      
    case PROTECTED_IDENTIFIER:
      if(rx[lin_num].protected_id != val)
        return(FALSE);
      break;
        
    case DATA_0:
    case DATA_1:
    case DATA_2:
    case DATA_3:
    case DATA_4:
    case DATA_5:
    case DATA_6:
    case DATA_7:
      if(rx[lin_num].data[status-DATA_0] != val)
        return(FALSE);
      break;
        
    case CHECKSUM:
      if(rx[lin_num].check != val)
        return(FALSE);
      break;      
  }
  return(TRUE);
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// LINSendChar
// --------------------------------------------------------------------------------------
// SCI byte fields sending routine
/////////////////////////////////////////////////////////////////////////////////////////
Bool LINSendChar(unsigned char lin_num, Bool brk, unsigned char ch)
{

  unsigned volatile char *lin_pt;

  lin_pt = lin_periph[lin_num];
  //Wait until transmit data registry is empty (TDRE=1)
  while(!(lin_pt[SCISR1]&0x80))
    ;
  // If breack field, transmit one breack character (13-14 bits of dominant value) 
  if(brk) 
  {
    lin_pt[SCICR2] |= 0x01;
    lin_pt[SCICR2] &= ~0x01;
  }
  else
    
    lin_pt[SCIDRL] = ch;
    
  return(TRUE);
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// LINSendMsg
// --------------------------------------------------------------------------------------
// LIN Message Sending Routine
/////////////////////////////////////////////////////////////////////////////////////////
Bool LINSendMsg(unsigned char lin_num, Bool master, Bool send_data, struct message msg)
{

  unsigned char check_sum, parity_id, i;

  //if(rx[lin_num].state != IDLE)
    //return(FALSE);
  
  rx[lin_num].error = 0;
  
  if(master) 
  {
    // Sends breack feald
    if(!LINSendChar(lin_num, TRUE, 0x00))
      return(FALSE);
    // Check breack sending
    if(!LINCheckSend(lin_num, _BREAK, 0x00))
      return(FALSE);    
    
    // Sends synch feald
    if(!LINSendChar(lin_num, FALSE, 0x55))
      return(FALSE);
    // Check synch sending
    if(!LINCheckSend(lin_num, SYNCH, 0x55))
      return(FALSE);
    
    parity_id=LINCalcParity(msg.identifier);
    // Sends protected identifier feald
    if(!LINSendChar(lin_num, FALSE, parity_id))
      return(FALSE);
    // Check protected identifier sending
    if(!LINCheckSend(lin_num, PROTECTED_IDENTIFIER, parity_id))
      return(FALSE);
  }
  if (send_data)
  {
    for(i=0; i < MAX_DATA; i++) 
    {
      // Send data field 
      if(!LINSendChar(lin_num, FALSE, msg.data_field[i]))
        return(FALSE);
      // Check data sending
      if(!LINCheckSend(lin_num, DATA_0+i, msg.data_field[i]))
        return(FALSE); 
    }
    check_sum = LINCalcChecksum(msg.data_field);
    // Send checksum field
    if(!LINSendChar(lin_num, FALSE, check_sum))
      return(FALSE);
    // Check checksum sending
    if(!LINCheckSend(lin_num, CHECKSUM, check_sum))
      return(FALSE);
    rx[lin_num].state = IDLE;
  } 
  return(TRUE);
      
}

/////////////////////////////////////////////////////////////////////////////////////////
// LINCheckState
// --------------------------------------------------------------------------------------
// LIN State Check Routine
/////////////////////////////////////////////////////////////////////////////////////////
enum lin_state LINCheckState(unsigned char lin_num)
{

  return(rx[lin_num].state);

}


#pragma CODE_SEG __NEAR_SEG NON_BANKED

/////////////////////////////////////////////////////////////////////////////////////////
// SCI0_ISR
// --------------------------------------------------------------------------------------
// SCI0 Interrupt Service Routine
/////////////////////////////////////////////////////////////////////////////////////////
interrupt 20 void SCI0_ISR(void)
{

  if(!LINGetChar(LIN_0)) 
  {
    rx[LIN_0].error = 1;
    rx[LIN_0].state = IDLE;
  }
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// SCI1_ISR
// --------------------------------------------------------------------------------------
// SCI1 Interrupt Service Routine
/////////////////////////////////////////////////////////////////////////////////////////
interrupt 21 void SCI1_ISR(void)
{

  if(!LINGetChar(LIN_1)) 
  {
    rx[LIN_1].error = 1;
    rx[LIN_1].state = IDLE;
  }
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// SCI2_ISR
// --------------------------------------------------------------------------------------
// SCI2 Interrupt Service Routine
/////////////////////////////////////////////////////////////////////////////////////////
interrupt 22 void SCI2_ISR(void)
{

  if(!LINGetChar(LIN_2)) 
  {
    rx[LIN_2].error = 1;
    rx[LIN_2].state = IDLE;
  }
  
}


#pragma CODE_SEG DEFAULT
