/*******************************************************************************
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2005 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by FREESCALE in this matter are performed AS IS and without 
any warranty. CUSTOMER retains the final decision relative to the total design 
and functionality of the end product. FREESCALE neither guarantees nor will be 
held liable by CUSTOMER for the success of this project.
FREESCALE DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING, 
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR 
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE OR ADVISE SUPPLIED TO THE PROJECT
BY FREESCALE, AND OR NAY PRODUCT RESULTING FROM FREESCALE SERVICES. IN NO EVENT
SHALL FREESCALE BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold FREESCALE harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial, 
contractual, or tortuous, rising directly or indirectly as a result of an advise
or assistance supplied CUSTOMER in connection with product, services or goods 
supplied under this Agreement.
********************************************************************************
* File      main.c
* Owner     B34874
* Version   1.0
* Date      Mar-1-2012
* Classification   General Business Information
* Brief     Example code for LIN comunication at S12G128
********************************************************************************
* Detailed Description:
*
* - The SW demonstrates LIN comunication between two TWR-S12G128 boards 
*
* - tested on: TWR-S12G128
* - clock source = IRC1M = 1MHz, BUSCLK = 6.25MHz as default state after reset
* - Reference to documentation: MC9S12GRMV1 Rev.1.10
*
* - LIN communication data from master to slave
*
* - This example must be programmed in two TWR-S12G128 Demo Boards connected
*   through the LIN bus.
*   
* - This program sends the potentiometer value of the Master board through
*   the LIN bus and displays the value received on the PTA[3..0] LEDs of the Slave board.
*  
* - A LIN cable must connect the two boards.
*
* - +12V power source must be conected to E1,E2 pins at one board
*  
* - Before to run this example, verify that all jumpers are in their default
*   position. Moreover, the example on the Master board must be compiled with
*  
* - #define _MASTER 1
*  
* - while the Slave board must be compiled with
*  
* - #define _MASTER 0
*  
* - See the "Summary of Jumper and Connector Settings" chapter in
*   the user's manual.
*
* - COM_EN - jumpers between pins 1-3 and 2-4
*  
* - NOTE: This example serves the only purpose of showing how to manage the SCI
*   peripheral to reconstruct a LIN bus, and is not intended to be a starting
*   point for a real-world LIN application.
*  
* - This project has been written for MC9S12XEP100 in CodeWarrior 4.5 for HC(S)12
*   original code from SofTec. Copyright (c) 2006 SofTec Microsystems // http://www.softecmicro.com/
*   Updated by B34874 in CodeWarrior 5.1 for HC(S)12 2012-03-1
*
********************************************************************************
Revision History:
Version  Date         Author  Description of Changes
1.0      Mar-1-2012   B34874  Initial version
*******************************************************************************/

#include <hidef.h>            /* common defines and macros */
#include "derivative.h"       /* derivative-specific definitions */
#include "lin.h"              /* lin functions */

#define     _ID         0x01
#define     _MASTER     1       // 0 = SLAVE TASK
                                // 1 = MASTER TASK
// Unsecured status flash
const unsigned char flash_security  @0xFF0F = 0xFE;

// Variables
Bool lin_send_enable = FALSE;
unsigned char potentiometer_value;
struct message msg_send, msg_get;

/////////////////////////////////////////////////////////////////////////////////////////
// PeriphInit
// --------------------------------------------------------------------------------------
// Initializes various registers and peripherals
/////////////////////////////////////////////////////////////////////////////////////////
void PeriphInit(void)
{
  // Configures PT[4..7] port as output  
  PTT = 0x00;                                                    
  DDRT = 0xF0;                        

  // Configures PortAD as input                                   
  DDR0AD = 0x00;
  
  // Configures PAD[4..7] as digital input (enable digital buffer)                                   
  ATDDIEN = 0xF0;                                   
    
  // Enables pull-ups on portAD
  PER0AD |= 0xF0;                                   

  // Configures the ATD peripheral
  // 8 bit data resolution
  ATDCTL1 = 0x00; 
  // Left justified data, 1 conversion sequence and non-FIFO mode
  // fBUS=6.25MHz, fATDCLK = 1.04 MHz (PRESCLAER = 2) Select 4 Sample Time
  ATDCTL4 = 0x02;
  
  // Configures the RTI (Real Time Interrupt) 
  // to generate a periodic interrupt of 10ms (Real Time Interrupt)
  if(_MASTER)  // Master 
    {
    CPMURTI = 0x89;   // Decimal based divider, divider 10x10^3, 1MHz IRC clock as default source
    CPMUINT = 0x80;   // enable RTI interrupt
    }
  
  LINInit(LIN_0);     // LIN Intialize
   
  EnableInterrupts;   // Enable Interrupts
}


#pragma CODE_SEG __NEAR_SEG NON_BANKED
/////////////////////////////////////////////////////////////////////////////////////////
// RTI_ISR
// --------------------------------------------------------------------------------------
// RTI Interrupt Service Routine
/////////////////////////////////////////////////////////////////////////////////////////
interrupt 7 void RTI_ISR(void)
{

  lin_send_enable = TRUE;                                //set flag

  CPMUFLG = 0x80; //cler RTI flag
  
}

#pragma CODE_SEG DEFAULT


/////////////////////////////////////////////////////////////////////////////////////////
// MAIN
/////////////////////////////////////////////////////////////////////////////////////////
void main(void)
{

  PeriphInit();                                          //device initialization
  
  for(;;)                                                //run forever
  {
    #if(_MASTER)  // Master 
      if (lin_send_enable)                               //check flag
      {
        // Select AN00 as input channel and single conversion
        ATDCTL5 = 0x00;                                  //start conversion
        while(!(ATDSTAT0 & 0x80))                        //waiting till end of conversion
          ; 
        // Sends the potentiometer value to the LIN bus
        potentiometer_value = ATDDR0H;                   //read ATD value
        msg_send.identifier = _ID;                       //set ID
        msg_send.data_field[0] = potentiometer_value;    //set LIN data
        (void)LINSendMsg(LIN_0, TRUE, TRUE, msg_send);   //send LIN mesage
        PTT = (potentiometer_value<<4);                  //display ATD value at LEDs (low order half byte)
        lin_send_enable = FALSE;                         //clear flag
      }
    #else // Slave
      if (LINCheckState(LIN_0) == CHECKSUM)              //check checksum
      {
        // Reads the potentiometer value from the LIN bus
        if (LINGetMsg(LIN_0, TRUE, &msg_get))
        {
          if (msg_get.identifier == _ID)                 //check ID 
            // Shows the potentiometer value on the LEDs
            PTT = (msg_get.data_field[0]<<4);            //display received data at LEDs (low order half byte)
        }  
      }
    #endif
  }
  
}      




