/////////////////////////////////////////////////////////////////////////////////////////
//
// Sample for Freescale TWR-S12G128
//
// LIN communication data from master to slave
//
// This example must be programmed in two TWR-S12G128 Demo Boards connected
// through the LIN bus.
//
// This program sends the potentiometer value of the Master board through
// the LIN bus and displays the value received on the PTA[3..0] LEDs of the Slave board.
//
// A LIN cable must connect the two boards.
//
// Before to run this example, verify that all jumpers are in their default
// position. Moreover, the example on the Master board must be compiled with
//
// #define _MASTER 1
//
// while the Slave board must be compiled with
//
// #define _MASTER 0
//
// See the "Summary of Jumper and Connector Settings" chapter in
// the user's manual.
//
// NOTE: This example serves the only purpose of showing how to manage the SCI
// peripheral to reconstruct a LIN bus, and is not intended to be a starting
// point for a real-world LIN application.
//
// --------------------------------------------------------------------------------------
//
// This project has been written for MC9S12XEP100 in CodeWarrior 4.5 for HC(S)12
// and updated by B34874 in CodeWarrior 5.1 for HC(S)12 2012-02-27
//
// --------------------------------------------------------------------------------------
/////////////////////////////////////////////////////////////////////////////////////////
#include <hidef.h>      /* common defines and macros */
#include "derivative.h"      /* derivative-specific definitions */
#include "lin.h"

#define     _ID         0x01
#define     _MASTER     1       // 0 = SLAVE TASK
                                // 1 = MASTER TASK
// Unsecured status flash
const unsigned char flash_security  @0xFF0F = 0xFE;

Bool lin_send_enable = FALSE;
unsigned char potentiometer_value;
struct message msg_send, msg_get;

/////////////////////////////////////////////////////////////////////////////////////////
// PeriphInit
// --------------------------------------------------------------------------------------
// Initializes various registers and peripherals
/////////////////////////////////////////////////////////////////////////////////////////
void PeriphInit(void)
{
  // Configures PT[4..7] port as output  
  PTT = 0x00;                                                    
  DDRT = 0x3C;                        

  // Configures PortAD as input                                   
  DDR0AD = 0x00;
  ATDDIEN = 0xF0;                                   
    
  // Enables pull-ups on portAD
  PER0AD |= 0xF0;                                   

  // Configures the ATD peripheral
  // 8 bit data resolution
  ATDCTL1 = 0x00; 
  // Left justified data, 1 conversion sequence and non-FIFO mode
  // fBUS=6.25MHz, fATDCLK = 1.04 MHz (PRESCLAER = 2) Select 4 Sample Time
  ATDCTL4 = 0x02;
  
  // Configures the RTI (Real Time Interrupt) 
  // to generate a periodic interrupt of 10ms (Real Time Interrupt)
  
  if(_MASTER)  // Master 
    {
    CPMURTI = 0x89;   // Decimal based divider, divider 10x10^3, 1MHz IRC clock as default source
    CPMUINT = 0x80;   // enable RTI interrupt
    }
  LINInit(LIN_0);
   
  EnableInterrupts;
}


#pragma CODE_SEG __NEAR_SEG NON_BANKED
/////////////////////////////////////////////////////////////////////////////////////////
// PIT0_ISR
// --------------------------------------------------------------------------------------
// PIT0 Interrupt Service Routine
/////////////////////////////////////////////////////////////////////////////////////////
interrupt 7 void RTI_ISR(void)
{

  lin_send_enable = TRUE;

  CPMUFLG = 0x80; //cler RTI flag
  
}

#pragma CODE_SEG DEFAULT

/////////////////////////////////////////////////////////////////////////////////////////
// Interrupt catcher
/////////////////////////////////////////////////////////////////////////////////////////
volatile unsigned int number_of_ISR = 0;


void Unimplemented_ISR(void)
{
  asm BGND;         //If the code stops here, check the "number_of_ISR"
}



/////////////////////////////////////////////////////////////////////////////////////////
// MAIN
/////////////////////////////////////////////////////////////////////////////////////////
void main(void)
{

  PeriphInit();
  
  for(;;) 
  {
    #if(_MASTER)  // Master 
      if (lin_send_enable)
      {
        // Select AN00 as single input channel and single conversion
        ATDCTL5 = 0x00;   
        while(!(ATDSTAT0 & 0x80))
          ; 
        // Sends the potentiometer value to the LIN bus
        potentiometer_value = ATDDR0H;
        msg_send.identifier = _ID;
        msg_send.data_field[0] = potentiometer_value;
        (void)LINSendMsg(LIN_0, TRUE, TRUE, msg_send);
        PTT = potentiometer_value;
        lin_send_enable = FALSE;  
      }
    #else // Slave
      if (LINCheckState(LIN_0) == CHECKSUM)
      {
        // Reads the potentiometer value from the LIN bus
        if (LINGetMsg(LIN_0, TRUE, &msg_get))
        {
          if (msg_get.identifier == _ID)
            // Shows the potentiometer value on the LEDs
            PTT = msg_get.data_field[0];
        }  
      }
    #endif
  }
  
}      




