/**************************************************************************
* I2C.C          Bit-bang master I2C interface in C for HC11 family.
*
* This file contains an I2C Driver which implements an I2C bit-bang master
* in C on two Port D pins. See I2C.H for more information.
*
* Assumes the following connections (but these are easily changed):
*
*                       SCL  SDA
*      .----.----.----.-----.----.----.----.----.
*      | -- | -- | SS*| SCK |MOSI|MISO| TX | RX |  Port D
*      `----^----^----^-----^----^----^----^----'
*        b7   b6   b5   b4    b3   b2   b1   b0
*
* The port is operated in wire-or mode so pull-up resisitors (~10k) will
* be needed on all 6 Port D pins.
*
* Also, it is assumed that the bit-banged clock pulses will be no faster
* than allowed by the I2C spec.  If an exceedingly fast micro is used, it
* may be necessary to set the CLOCKDELAY parameter to some non-zero value.
* Measure the bit times to make sure that the clock hi and lo pulse widths
* are at least 4.7us.
*
* BIDIRECTIONALSCL is a parameter that turns on/off the ability of a
* peripheral device to "wait-state" the I2C transfer by holding SCL low.
* Most I2C capable IC's do NOT do this (consult the data sheets), but
* a slave CPU may.  Disabling clock-stretching (BIDIRECTIONALSCL=0) will
* improve throughput by about 15%.
*
* Other processor implementations (ie. 8051, x86, etc) are available.
* Bug reports and/or inquiries to: grantb@freenet.edmonton.ab.ca
*
***************************************************************************
* Version 1.00
* (c) Copyright 2000 Grant Beattie
* This software is the property of Grant Beattie who specifically grants
* the user the right to modify, use and distribute this software provided
* this notice is not removed or altered.  All other rights are reserved.
* NO WARRANTY OF ANY KIND IS IMPLIED.  NO LIABILITY IS ASSUMED FOR
* INCIDENTAL OR CONSEQUENTIAL DAMAGES IN CONNECTION WITH OR ARISING FROM
* THE FURNISHING, PERFORMANCE, OR USE OF THIS SOFTWARE.
***************************************************************************/


#include "i2c.h"


/*
| Change or remove these to mate with your current CPU Register file(s).
*/
#define  PORTD  *(volatile unsigned char *)0x1008
#define  DDRD   *(volatile unsigned char *)0x1009
#define  SPCR   *(volatile unsigned char *)0x1028


#define  SDAbit          0x08     // Indicate the SDA bit on Port D.
#define  SCLbit          0x10     // Indicate the SCL bit on Port D.


#define  CLOCKDELAY        0      // Count value to ensure 4.7us pulse width.
#define  BIDIRECTIONALSCL  0      // Turns on/off SCL stretching.


#define  SCLTIMEOUT       255     // How long to wait on SCL stretching.
#define  SDATIMEOUT       255     // How many clocks to give on SDA hung.



/*
| Prototypes: Internal to this file only (private).
*/
void I2cStart(void);
void I2cStop(void);

void SdaLo(void);
void SdaHi(void);
void SclLo(void);
void SclHi(void);

unsigned char GetSda(void);
unsigned char I2cRead(unsigned char count);
unsigned char I2cWrite(unsigned char count);
unsigned char I2cWriteByte(unsigned char byte);

#if  CLOCKDELAY
void I2cDelay(void);
#else
#define I2cDelay()
#endif

#if  BIDIRECTIONALSCL
unsigned char GetScl(void);
#else
#define GetScl() 1
#endif


/*
| Global variables (private).
*/
unsigned char gI2cErrors;


/*
| Global variables (public).
*/
I2CPACKET gI2C;




/*
.--------------------------------------------------------------------------
| unsigned char I2cInit(void);
|
| Initialzes the I2C driver and pins (by sending a stop condition).  If one
| of the pins is jammed, returns failure.  Also clears gI2cError variable.
| This function should be called prior to the first I2C transaction.
|
| Pins In:  No requirements
| Pins Out: Both SCL and SDA are high if successful
|
| Requires: Nothing
| Returns:  0 on success, nonzero on failure
|           gI2cErrors is modified on failure
`--------------------------------------------------------------------------
*/
unsigned char I2cInit(void)
{
gI2cErrors = 0;


/*
| From reset it is assumed that Port D is set as inputs and there are 10k
| pullups on PD0 - PD5. No other devices should be pulling these lines down.
*/
PORTD |= (SDAbit|SCLbit);         // Both pins high.
DDRD  |= (SDAbit|SCLbit);         // Both pins as outputs.
SPCR  |= 0x20;                    // DWOM = 1, puts PortD into wired-OR mode,
                                  // (requires pull-ups).

SclLo();
if(PORTD & SCLbit)                // Only able to test register bit on HC11.
   {                              // (Cannot read pin when set as output)
   gI2cErrors |= I2CERR_BUS;
   return 1;
   }

SdaLo();                          // SDA low (data can change while the clock is low).
if(PORTD & SDAbit)                // Only able to test register bit on HC11.
   {                              // (Cannot read pin when set as output)
   gI2cErrors |= I2CERR_BUS;
   return 1;
   }

I2cDelay();             // Do a simulated stop, complete with correct timing.
                        // Requires a 4.7us clock low period before stop.

I2cStop();
if(gI2cErrors)
   {
   gI2cErrors |= I2CERR_BUS;
   return 1;
   }

return 0;
}



/*
.--------------------------------------------------------------------------
| void I2cDelay(void); 
|
| Provides the necessary 4.7us delay required for some of the I2C 
| transitions.  The actual delay time depends on the processor and it's
| crystal frequency.
|
| Pins In:  No requirements
| Pins Out: No changes
|
| Requires: Nothing
| Returns:  Nothing
`--------------------------------------------------------------------------
*/
#if CLOCKDELAY
void I2cDelay(void)
{
unsigned char count = CLOCKDELAY;

while(count)
   count--;
return;
}
#endif



/*
.--------------------------------------------------------------------------
| void I2cStart(void);
|
| Creates a start condition on the I2C pins (a START is defined as SDA
| going low while SCL is high), allowing for proper timing.
|
|
| Pins In:  Assumes both SCL and SDA are high.
| Pins Out: Both SCL and SDA are low.
|
| Requires: Nothing
| Returns:  Nothing
`--------------------------------------------------------------------------
*/
void I2cStart(void)
{
SdaLo();
I2cDelay();
SclLo();
I2cDelay();
}



/*
.--------------------------------------------------------------------------
| void I2cStop(void);
|
| Creates a stop condition on the I2C pins (a STOP is defined as SDA
| going high while SCL is high), allowing for proper timing.  If SDA is
| hung, an attempt is made to free the bus by clocking SCL until SDA is
| released.
|
| Pins In:  Assumes SCL is low (SDA indeterminate - holds last ACK bit).
| Pins Out: Both SCL and SDA are high if successful.
|
| Requires: Nothing
| Returns:  Nothing (gI2cErrors is modified on failure)
`--------------------------------------------------------------------------
*/
void I2cStop(void)
{
unsigned char count;

SdaLo();                          // SCL is initally low, OK to bring SDA
SclHi();                          // low in preparation for the stop.
I2cDelay();                       // Bring SCL high then SDA high for the
SdaHi();                          // STOP condition.
I2cDelay();

if( GetSda() )                    // Successful STOP.
   return;

gI2cErrors |= I2CERR_NOSTOP;      // STOP error, SDA hung (low).
count = SDATIMEOUT;               // Someone is holding SDA low.
while(count)                      // Attempt to clock the bus out of the
   {                              // SDA hung state.
   SclLo();
   I2cDelay();
   SclHi();
   if( GetSda() )
      return;
   count--;
   }
}



/*
.--------------------------------------------------------------------------
| void SdaLo(void)
|
| Sets the SDA pin as an output and low.
| 
| Requires: Nothing
| Returns:  Nothing
`--------------------------------------------------------------------------
*/
void SdaLo(void)
{
PORTD &= ~SDAbit;                 // SDA is low (set level first).
DDRD  |=  SDAbit;                 // SDA is an output (set direction).
}



/*
.--------------------------------------------------------------------------
| void SdaHi(void)
|
| Sets the SDA pin as an output and high.
| 
| Requires: Nothing
| Returns:  Nothing
`--------------------------------------------------------------------------
*/
void SdaHi(void)
{
//PORTD |= SDAbit;                  // SDA is high (set level first).
//DDRD  |= SDAbit;                  // SDA is an output (set direction).

DDRD &= ~SDAbit;                  // Making SDA an input allows it to go high.
}



/*
.--------------------------------------------------------------------------
| void SclLo(void)
|
| Sets the SCL pin as an output and low.
|
| Requires: Nothing
| Returns:  Nothing
`--------------------------------------------------------------------------
*/
void SclLo(void)
{
PORTD &= ~SCLbit;                 // SCL is low (set level first).
#if  BIDIRECTIONALSCL
DDRD  |= SCLbit;                  // SCL is an output (set direction).
#endif
}



/*
.--------------------------------------------------------------------------
| void SclHi(void); 
|
| Releases SCL and verifies that it is high before returning (to allow for
| clock-stretching peripherals).  If SCL is frozen low, re-tries until
| finally giving up on a timeout.
|
| Pins In:  Assumes SCL is low.
| Pins Out: Sets SCL high if successful
|
| Requires: Nothing 
| Returns:  Nothing (gI2cErrors is modified on failure)
`--------------------------------------------------------------------------
*/
#if  BIDIRECTIONALSCL
void SclHi(void)
{
unsigned char count;

//PORTD |= SCLbit;                  // SCL is high (set level first).
//DDRD  |= SCLbit;                  // SCL is an output (set direction).

DDRD  &= ~SCLbit;                 // Making SCL an input allows it to go high.
if(PORTD & SCLbit)                // Release SCL, if it goes high, exit OK.
   return;

count = SCLTIMEOUT;               // Someone is holding it low.
while(count)                      // Give them until the TIMEOUT val or
   {                              // else fatal bus hung error.
   if(PORTD & SCLbit)
      return;
   count--;
   }

gI2cErrors |= I2CERR_BUS;
}
#else
void SclHi(void)
{
PORTD |= SCLbit;                  // SCL is high (set level first).
}
#endif



/*
.--------------------------------------------------------------------------
| unsigned char GetSda(void)
|
| Returns the bit value of the SDA pin.
|
| Requires: Nothing
| Returns:  0 on pin low, non-zero on pin high
`--------------------------------------------------------------------------
*/
unsigned char GetSda(void)
{
DDRD  &= ~SDAbit;                 // Make SDA an input.
return(PORTD & SDAbit);
}




/*
.--------------------------------------------------------------------------
| unsigned char GetScl(void)
|
| Returns the bit value of the SCL pin.
|
| Requires: Nothing
| Returns:  0 on pin low, non-zero on pin high
`--------------------------------------------------------------------------
*/
#if  BIDIRECTIONALSCL
unsigned char GetScl(void)
{
DDRD  &= ~SCLbit;                 // Make SDA an input.
return(PORTD & SCLbit);
}
#endif


/*
.--------------------------------------------------------------------------
| unsigned char I2cRead(unsigned char count);
|
| Reads one or more bytes (from the previously addressed I2C device) and
| places them in the global I2C buffer.
|
| Pins In:  Assumes Both SCL and SDA are low.
| Pins Out: SDA high, SCL low.  (A stop is required after.)
|
| Requires: Nothing
| Returns:  0 on success, non-zero on failure.
|           The bytes read are placed in gI2cBuffer[].
|           gI2cErrors is modified on failure.
`--------------------------------------------------------------------------
*/
unsigned char I2cRead(unsigned char count)
{
unsigned char mask;
unsigned char value;
unsigned char index = 0;

while(count)
   {
   mask  = 0x80;
   value = 0x00;
   while(mask)                    // Do the 8 data bits...
      {
      SclHi();                    // Set SCL and wait for it to go hi.
      I2cDelay();
      if( GetSda() )              // Read the bit, and if high set it in
         value |= mask;           // the returned byte.
      SclLo();                    // Bring SCL low again to complete bit.
      I2cDelay();
      mask >>= 1;
      }

   gI2C.buffer[index] = value;    // Save the read byte.
   index++;
   count--;
   if(count)                      // If not the last byte, ACK the read.
      {
      SdaLo();                    // Bring SDA low for ACK.
      SclHi();                    // Clock high.
      I2cDelay();
      SclLo();                    // Clock low.
      SdaHi();                    // Release SDA.
      I2cDelay();
      }
   }

SdaHi();                          // SDA high for NACK on last byte.
SclHi();                          // Clock high.
I2cDelay();
SclLo();                          // Clock low.
I2cDelay();

return 0;
}



/*
.--------------------------------------------------------------------------
| unsigned char I2cWrite(unsigned char count);
|
| Write the number of bytes from the gI2cBuffer to the I2C port.
|
| Pins In:  Assumes Both SCL and SDA are low.
| Pins Out: SDA indeterminate, SCL low. (A stop is required after.)
|
| Requires: count      - the number of chars to send.
|           gI2cBuffer - must be filled with the data to xmit.
| Returns:  0 on success, 1 on failure
|           gI2cErrors is modified on failure.
`--------------------------------------------------------------------------
*/
unsigned char I2cWrite(unsigned char count)
{
unsigned char mask;
unsigned char value;
unsigned char index = 0;

while(count)
   {
   value = gI2C.buffer[index];
   mask = 0x80;
   while(mask)                    // Do the 8 data bits...
      {
      if(value & mask)            // Set one bit (order is msb to lsb).
         SdaHi();
      else
         SdaLo();
      SclHi();                    // Set SCL and wait for it to go hi.
      I2cDelay();
      SclLo();                    // Bring SCL low again to complete bit.
      I2cDelay();
      mask >>= 1;
      }

   SdaHi();                       // Release SDA, let slave control it.
   SclHi();                       // Set SCL and wait for it to go hi.
   I2cDelay();

   if( GetSda() )                 // Check the acknowledge bit.
      {
      SclLo();                    // No ACK, finish SCK (back low again).
      I2cDelay();                 // Return failure (No ACK).
      gI2cErrors |= I2CERR_NOACK;
      return 1;                      
      }
      
   SclLo();                       // ACK OK, finish SCK (back low again).
   I2cDelay();                    // Return success (ACK OK).
   index++;
   count--;
   }

return 0;
}



/*
.--------------------------------------------------------------------------
| unsigned char I2cWriteByte(unsigned char byte);
|
| Write one to the I2C port (it is the I2C address).
|
| Pins In:  Assumes Both SCL and SDA are low.
| Pins Out: SDA indeterminate, SCL low.
|
| Requires: addr - the address of chars to send.
| Returns:  0 on success, 1 on failure
|           gI2cErrors is modified on failure.
`--------------------------------------------------------------------------
*/
unsigned char I2cWriteByte(unsigned char byte)
{
unsigned char mask = 0x80;
   
while(mask)                    // Do the 8 data bits...
   {
   if(byte & mask)             // Set one bit (order is msb to lsb).
      SdaHi();
   else
      SdaLo();
   SclHi();                    // Set SCL and wait for it to go hi.
   I2cDelay();
   SclLo();                    // Bring SCL low again to complete bit.
   I2cDelay();
   mask >>= 1;
   }

SdaHi();                       // Release SDA, let slave control it.
SclHi();                       // Set SCL and wait for it to go hi.
I2cDelay();

if( GetSda() )                 // Check the acknowledge bit.
   {
   SclLo();                    // No ACK, finish SCK (back low again).
   I2cDelay();                 // Return failure (No ACK).
   gI2cErrors |= I2CERR_NOACK;
   return 1;                      
   }
      
SclLo();                       // ACK OK, finish SCK (back low again).
I2cDelay();                    // Return success (ACK OK).
return 0;
}



/*
.--------------------------------------------------------------------------
| unsigned char I2cTransfer(void);
|
| Transfers the desired # of bytes to/from the intended target. The address,
| operating conditions and data are obtained from the "gI2C" packet struct.
|
| Pins In:  Assumes Both SCL and SDA are high.
| Pins Out: SCL, SDA high if successful.
|
| Requires: gI2C - data structure must be filled in.
|
| Returns:  0 on success, 1 on failure
|           The gI2C structure is NOT modified (except the buffer portion).
|           gI2cErrors is modified and can be read via I2cGetLastError().
`--------------------------------------------------------------------------
*/
unsigned char I2cTransfer(void)
{
unsigned char rtn;
unsigned char addr;

gI2cErrors = 0;

if(gI2C.datalength > I2CBUFSIZE)  // Exit failure on too many bytes requested.
   {
   gI2cErrors |= I2CERR_OVERFLOW;
   return 1;
   }

if((GetSda()==0) || (GetScl()==0))// Both bits should be high on entrance.
   {
   gI2cErrors |= I2CERR_BUS;      // Will require a re-init or h/w fix.
   return 1;
   }

if(gI2C.mode & I2CMODE_READ)      // If it's read make sure b0=1, else b0=0.
   addr = gI2C.address | 1;
else
   addr = gI2C.address & 0xFE;


I2cStart();                       // Send START condition.
if( I2cWriteByte(addr) )          // Send I2c address.
   {
   I2cStop();                     // Address NACK'd, do STOP & exit failure.
   return 1;
   }

/*
| Do Sub-Address 1 if requested.
*/
if(gI2C.mode & I2CMODE_SUBADDR1)
   {
   if( I2cWriteByte(gI2C.subaddr1) ) 
      {
      I2cStop();                  // Byte NACK'd, do STOP & exit failure.
      return 1;
      }
   }

/*
| Do Sub-Address 2 if requested.
*/
if(gI2C.mode & I2CMODE_SUBADDR2)
   {
   if( I2cWriteByte(gI2C.subaddr2) ) 
      {
      I2cStop();                  // Byte NACK'd, do STOP & exit failure.
      return 1;
      }
   }

/*
| Do repeated start if requested.
*/
if(gI2C.mode & I2CMODE_REPSTART)
   {
   SdaHi();                          // Release SDA while clock low.
   I2cDelay();
   SclHi();                          // Release SCL, can now do a START.
   I2cDelay();
   if(gI2C.mode & I2CMODE_REPREAD)   // If repeated start is a read, make 
      addr = gI2C.address | 1;       // sure b0=1, else b0=0.
   else
      addr = gI2C.address & 0xFE;         
   I2cStart();                       // Send START condition.
   if( I2cWriteByte(addr) ) 
      {
      I2cStop();                     // Byte NACK'd, do STOP & exit failure.
      return 1;
      }
   }

/*
| Do read or write (but do not do STOP). If it's a regular read or write, use
| the regular read/write bit to determine direction.  If it's a repeated start,
| use the REP read/write bit.
*/
if(gI2C.mode & I2CMODE_REPSTART)
   {
   if(gI2C.mode & I2CMODE_REPREAD)
      rtn = I2cRead(gI2C.datalength);
   else
      rtn = I2cWrite(gI2C.datalength);
   }
else
   {
   if(gI2C.mode & I2CMODE_READ)
      rtn = I2cRead(gI2C.datalength);
   else
      rtn = I2cWrite(gI2C.datalength);
   }

/*
| Finally, do the STOP.
*/
I2cStop();
if(rtn || gI2cErrors)
   return 1;

return 0;
}




/*
.--------------------------------------------------------------------------
| unsigned char I2cGetLastError(void);
|
| Returns the value of the gI2cErrors variable.  The results refer to only
| the last I2C transaction and are cleared when a new transaction begins.
|
| Requires: Nothing
| Returns:  The gI2cErrors var
`--------------------------------------------------------------------------
*/
unsigned char I2cGetLastError(void)
{
return gI2cErrors;
}


