/**************************************************************************
* I2C.H         Bit-bang master I2C interface in C
*
* This file contains an I2C Driver which implements an I2C bit-bang master
* in C on two Port D pins.
*
* Any transfer requires setting up the:
*   - I2C address,
*   - (optional) sub-addresses,
*   - transfer mode, and
*   - transfer data length
* into the "gI2C" structure.  If the transfer is a write, then the data
* buffer[] portion must be filled in as well.
*
* The transfer then consists of the following transactions:
*
*                          Param to fill in:    Value to set in gI2C.mode:
*  .-------------------.   -----------------   ----------------------------
*  |     I2C START     |
*  >-------------------<
*  |    Chip Address   |     gI2C.address
*  >-------------------<
*  |      R/W bit      |                       I2CMODE_READ or I2CMODE_WRITE
*  >-------------------<
*  | Sub-address (MSB) |     gI2C.subaddr1            I2CMODE_SUBADDR1
*  |     (optional)    |
*  >-------------------<
*  | Sub-address (LSB) |     gI2C.subaddr2            I2CMODE_SUBADDR2
*  |     (optional)    |
*  >-------------------<
*  |   Repeated START  |                              I2CMODE_REPSTART
*  |    Chip Address   |
*  |       R/W bit     |                       I2CMODE_REPREAD or I2CMODE_REPWRITE
*  |     (optional)    |
*  >-------------------<
*  |                   |
*  |         :         |     gI2C.datalength
*  |       data        |
*  |         :         |      gI2C.buffer[]
*  |                   |       (if write)
*  >-------------------<
*  |      I2C STOP     |
*  `-------------------'
*
* Other processor implementations (ie. 8051, x86, etc) are available.
* Bug reports and/or inquiries to: grantb@freenet.edmonton.ab.ca
*
***************************************************************************
* EXAMPLE: Write two bytes to a 24C64 EEPROM.
*
* gI2C.address = 0xA0;            // EEPROM standard address.
* gI2C.subaddr1 = 0;              // Internal address MSB = 00.
* gI2C.subaddr2 = 0;              // Internal address LSB = 00.
* gI2C.buffer[0] = 'H';           // Sample data to write.
* gI2C.buffer[1] = 'i';
* gI2C.datalength = 2;
* // Write the two sub-addresses, then the data.
* gI2C.mode = I2CMODE_WRITE|I2CMODE_SUBADDR1|I2CMODE_SUBADDR2;
* if( I2cTransfer() )
*    printf("\r\nI2C write error: 0x%02X", I2cGetLastError());
*
***************************************************************************
* Version 1.00
* (c) Copyright 2000 Grant Beattie
* This software is the property of Grant Beattie who specifically grants
* the user the right to modify, use and distribute this software provided
* this notice is not removed or altered.  All other rights are reserved.
* NO WARRANTY OF ANY KIND IS IMPLIED.  NO LIABILITY IS ASSUMED FOR
* INCIDENTAL OR CONSEQUENTIAL DAMAGES IN CONNECTION WITH OR ARISING FROM
* THE FURNISHING, PERFORMANCE, OR USE OF THIS SOFTWARE.
**************************************************************************/


// Values from I2cErrors variable returned via I2cGetLastError():
#define  I2CERR_BUS         0x01  // Misc bus error, pins frozen.
#define  I2CERR_NOACK       0x02  // Device failed to acknowlegde.
#define  I2CERR_NOSTOP      0x04  // SDA held when STOP attempted.
#define  I2CERR_OVERFLOW    0x08  // I2C buffer count request too big.


// Values for gI2C.mode variable (the transfer types):
#define  I2CMODE_WRITE      0x00  // R/W* = 0 for address byte.
#define  I2CMODE_READ       0x01  // R/W* = 1 for address byte.
#define  I2CMODE_SUBADDR1   0x02  // Include sub-address value.
#define  I2CMODE_SUBADDR2   0x04  // Include additional sub-address value.
#define  I2CMODE_REPSTART   0x08  // Insert repeated start before data phase.
#define  I2CMODE_REPWRITE   0x00  // R/W* = 0 for repeated start address byte.
#define  I2CMODE_REPREAD    0x10  // R/W* = 1 for repeated start address byte.


#define  I2CBUFSIZE         32    // How much SRAM space allocated to I2C.


typedef struct
{
unsigned char mode;               // Transfer direction and options (see list).
unsigned char address;            // 8-bit I2C address (with b0=0).
unsigned char subaddr1;           // Optional device internal address MSB.
unsigned char subaddr2;           // Optional device internal address LSB.
unsigned char datalength;         // Number of data bytes to read/write.
unsigned char buffer[I2CBUFSIZE]; // Read/write buffer.
}I2CPACKET;


/*
| Global vars (public)
*/
extern I2CPACKET gI2C;


/*
| Prototypes (public)
*/
unsigned char I2cInit(void);
unsigned char I2cTransfer(void);
unsigned char I2cGetLastError(void);

