// Definition of CMemory (68HC11 Memory Operations Class)
//
// Authors: Matt Vandenbush
// Code Review: Matt Vandenbush
// Date: 12/30/96

// Include files
#include "CMemory.h"
#include "assert.h"
#include "ConvToULong.h"
#include "string.h"

CMemory::CMemory(const CTalker* talkerObject)
{
	m_talker = (CTalker*) talkerObject;	// talker object
	m_dwEEPROMDelay = 20;				// Default EEPROM Delay, 20ms
	m_uiEEPROMStart = (UINT)46592;		// Default start location, B600
	m_uiEEPROMEnd = (UINT)47103;		// Default end location, B7FF
}

////////////////////////////////////////////////////////////
// DisplayMemory: reads a specified number of data bytes
//      from a specified start address on the 68HC11.  
//		Data is "returned" since the byte array is
//	    passed by reference.
//
// Arguments:
//		uiStartAddress : Address to start reading at
//		uiByteCount : Number of bytes to read from uiStartAddress 
//		bypReadBuffer : Byte array holding data after completion
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.DisplayMemory(uiStartAddress,
//							 uiByteCount,
//							 bypReadBuffer);
//
// Notes : bypReadBuffer passed by reference since
//		   it is an array of bytes.

UINT CMemory::DisplayMemory(const UINT uiStartAddress,
							const UINT uiByteCount,
							BYTE* bypReadBuffer)
{
	UINT  uiCurrentAddress = 0;		// Holds the address currently being read
	BYTE  byLowByte = 0;			// The low byte of the byte count.  This
									// is used to finish up the non-256 byte
									// reads which are indicated by the upper 
									// byte of byte count
	BYTE  byFullReadReps = 0;		// This is the number of full 256 byte 
									// reads that need to be done starting at.
									// uiStartAddress. It is gotten from the 
									// upper byte of the byte count
	BYTE* bypBufferPointer = NULL;	// A pointer that indexes into the read 
									// buffer passed in by the calling function
    UINT  uiReturn = SUCCESS;		// Return value

	byLowByte = uiByteCount & 0xff;	// Low byte
	byFullReadReps = (uiByteCount & 0xff00) >> 8; // High byte

	uiCurrentAddress = uiStartAddress;
	bypBufferPointer = bypReadBuffer;

	// Do the full 256 byte reads first
	for(BYTE byi = 0; byi < byFullReadReps; byi++)
	{
		if((uiReturn = m_talker->ReadMemory(uiCurrentAddress,// Current Address
								            0,	// Read 256 bytes
								            bypBufferPointer)) != SUCCESS)
			return uiReturn;
		else
		{
			bypBufferPointer += 256;	// Increment buffer pointer
			uiCurrentAddress += 256;	// Increment read address
		}
	}

	// Now do the low byte / partial reads
	// If low byte was zero, means there are no non-256 byte reads
	if(byLowByte != 0)	// '0' means 256 to the talker
	{
		if((uiReturn = m_talker->ReadMemory(uiCurrentAddress,// Current Address
								            byLowByte,	// Number in low byte
								            bypBufferPointer)) != SUCCESS)
			return uiReturn;
	}

	return uiReturn;
}

////////////////////////////////////////////////////////////
// WriteNormalMem: writes a specified number of data bytes
//      to a specified start address on the 68HC11.  This
//		function handles only non-EEPROM writes and is called
//		by WriteMemory on demand for normal memory writes.  
//
// Arguments:
//		uiStartAddress : Address to start writing at
//		uiByteCount : Number of bytes to write from uiStartAddress 
//		bypWriteBuffer : Byte array holding data to be written
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.WriteNormalMem(uiStartAddress,
//                            uiByteCount,
//                            bypWriteBuffer);
//
// Notes : bypWriteBuffer passed by reference since
//		   it is an array of bytes.

UINT CMemory::WriteNormalMem(const UINT uiStartAddress,
			 			     const UINT uiByteCount,
			 			     BYTE* bypWriteBuffer)
{
	UINT  uiCurrentAddress = 0;		// Holds address currently being written
	BYTE  byLowByte = 0;			// The low byte of the byte count.  This
									// is used to finish up the non-256 byte
									// writes which are indicated by the upper 
									// byte of byte count
	BYTE  byFullWriteReps = 0;		// This is number of full 256 byte writes
									// that need to be done starting at 
									// uiStartAddress.  It is gotten from the 
									// upper byte of the byte count.
	BYTE* bypBufferPointer = NULL;	// A pointer that indexes into write buffer
									// passed in by the calling function
    UINT  uiReturn = SUCCESS;		// Return value
	
	byLowByte = uiByteCount & 0xff;	// Low bute
	byFullWriteReps = (uiByteCount & 0xff00) >> 8;	// High byte

	uiCurrentAddress = uiStartAddress;
	bypBufferPointer = bypWriteBuffer;

	// Do the full 256 byte Writes first
	for(BYTE byi = 0; byi < byFullWriteReps; byi++)
	{
		if((uiReturn = m_talker->WriteMemory(uiCurrentAddress,// Curr. Address
								             0,	 // Write 256 bytes
								             bypBufferPointer)) != SUCCESS)
			return uiReturn;
		else
		{
			bypBufferPointer += 256;	// Increment buffer pointer
			uiCurrentAddress += 256;	// Increment write address
		}
	}

	// Now do the low byte / partial writes
	// If low byte was zero, means there are no non-256 byte writes
	if(byLowByte != 0)	// '0' means 256 to the talker
	{
		if((uiReturn = m_talker->WriteMemory(uiCurrentAddress,// Curr. Address
							                 byLowByte,	// Number in low byte
							                 bypBufferPointer)) != SUCCESS)
			return uiReturn;
	}

	return uiReturn;

}

////////////////////////////////////////////////////////////
// SetEEPROMDelay: Sets the private data member m_dwEEPROMDelay to
//		to the specified DWORD value in milliseconds 
//
// Arguments:
//		dwDelay : Value to which to set m_dwEEPROMDelay
//
// Return Value:
//		TRUE if successful, otherwise FALSE
//
// Example Usage:
//		memory.SetEEPROMDelay(dwDelay);
//

BOOL CMemory::SetEEPROMDelay(const DWORD dwDelay)
{
	// Set the member variable
	m_dwEEPROMDelay = dwDelay;
	
	// Read back to verify
	if(m_dwEEPROMDelay != dwDelay)
		return FALSE;
	else
		return TRUE;

}

////////////////////////////////////////////////////////////
// SetEEPROMRange: Sets the private data members m_uiEEPROMStart and
//		m_uiEEPROMEnd to the specified values. 
//
// Arguments:
//		uiStart : Value to which to set m_uiEEPROMStart
//		uiEnd : Value to which to set m_uiEEPROMEnd
//
// Return Value:
//		TRUE if successful, otherwise FALSE
//
// Example Usage:
//		memory.SetEEPROMRange(uiStart, uiEnd);
//

BOOL CMemory::SetEEPROMRange(const UINT uiStart,
							 const UINT uiEnd)
{
	// Set the member variables
	m_uiEEPROMStart = uiStart;
	m_uiEEPROMEnd = uiEnd;

	// Read back to verify
	if((m_uiEEPROMStart != uiStart) || (m_uiEEPROMEnd != uiEnd))
		return FALSE;
	else
		return TRUE;
}

////////////////////////////////////////////////////////////
// FillBlock: writes a single data byte a specified number 
//      of times starting from a specified start address on the 68HC11.  
//
// Arguments:
//		uiStartAddress : Address to start writing at
//		uiByteCount : Number of bytes to write from uiStartAddress 
//		byByte : Byte holding data to be written
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.FillBlock(uiStartAddress,
//                       uiByteCount,
//                       byByte);
//

UINT CMemory::FillBlock(const UINT uiStartAddress,
						const UINT uiByteCount,
						BYTE byByte)
{	
	BYTE* bypWriteBuffer;		// A write buffer created by this function
								// to pass to the WriteMemory member function
	UINT  uiReturn = SUCCESS;	// Return value

	bypWriteBuffer = new BYTE [uiByteCount];
	ASSERT(bypWriteBuffer != NULL);

	for(UINT uiCount = 0; uiCount < uiByteCount; uiCount++)
		bypWriteBuffer[uiCount] = byByte;	    // Fill the array with data

	// Call WriteMemory with the newly allocated write buffer
	uiReturn = WriteMemory(uiStartAddress, 
						   uiByteCount, 
						   bypWriteBuffer);
	
	// Free the pointer
	delete bypWriteBuffer;
	return uiReturn;
}

////////////////////////////////////////////////////////////
// MoveBlock: moves a block of data from a specified start 
//		to a specified destination address on the 68HC11.  
//
// Arguments:
//		uiStartAddress : Address to start reading at
//		uiByteCount : Number of bytes to read from uiStartAddress 
//		uiDestAddress : Address to start writing at
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.MoveBlock(uiStartAddress,
//                       uiByteCount,
//                       byDestAddress);
//

UINT CMemory::MoveBlock(const UINT uiStartAddress,
	 					const UINT uiByteCount,
						const UINT uiDestAddress)
{
	BYTE* bypByteBuffer;		// Byte buffer to transfer data 
	UINT  uiReturn = SUCCESS;	// Return value

	bypByteBuffer = new BYTE [uiByteCount];
	ASSERT(bypByteBuffer != NULL);

	// Read the memory block into the buffer
	uiReturn = DisplayMemory(uiStartAddress, 
							 uiByteCount, 
							 bypByteBuffer);
		
    // If there was not an error
	if(uiReturn == SUCCESS)
	{
	// Write the memory block to the destination
		uiReturn = WriteMemory(uiDestAddress, 
							   uiByteCount, 
							   bypByteBuffer);
	}
		
	// Free the pointer
	delete bypByteBuffer;
	return uiReturn;
}

////////////////////////////////////////////////////////////
// FindFirstByte : Finds a byte in a specified range.
//
// Arguments:
//		uiStartAddress : Address to start reading at
//		uiByteCount : Number of bytes to read from uiStartAddress 
//		byData : Byte holding byte to search for
//		uiLocation : Address at which byte is found
//
// Return Value:
//		SUCCESS if byte found, otherwise error code
//
// Example Usage:
//		memory.FindFirstByte(uiStartAddress,
//                           uiByteCount,
//                           byData
//						     &uiLocation);
//

UINT CMemory::FindFirstByte(const UINT uiStartAddress,
	 					    const UINT uiByteCount,
						    const BYTE byData,
						    UINT &uiLocation)
{
	BYTE* bypByteBuffer = NULL;	// Byte buffer that holds memory to search
	UINT  uiReturn = SUCCESS;   // Return value
	BOOL  bDataNotFound = TRUE;	// Flag indicating whether data was found
	UINT  uiLoopCounter = 0;	// Byte counter to increment search buffer

	bypByteBuffer = new BYTE [uiByteCount];
	ASSERT(bypByteBuffer != NULL);

	// Read the search block into the buffer
	uiReturn = DisplayMemory(uiStartAddress, 
							 uiByteCount, 
							 bypByteBuffer);
			
    // If there was not an error
	if(uiReturn == SUCCESS)
	{
		// Search through the buffer for the search byte
		while((uiLoopCounter < uiByteCount) && (bDataNotFound))
		{
			if(bypByteBuffer[uiLoopCounter] == byData)
			{
				uiLocation = uiStartAddress + uiLoopCounter;
				bDataNotFound = FALSE;
			}
			else
				uiLoopCounter++;
		}
			
		// If after the loop terminates, the bDataNoFound flag is still
		// true, then report that the byte was not found in the range
		if(bDataNotFound)
		{
			uiLocation = 0;
			uiReturn = FIND_BYTE_FAIL;
		}
	}

	// Free the pointer
	delete bypByteBuffer;
	return uiReturn;
}

////////////////////////////////////////////////////////////
// WriteEEPROM: writes a byte to EEPROM
//
// Arguments:
//		uiAddress : Address to write at
//		byData : Data byte to be written
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.WriteEEPROM(uiAddress,
//                         byData);
//

UINT CMemory::WriteEEPROM(const UINT uiAddress,
			 			  BYTE byData)
{
	BYTE byInitialState;		// Used to verify that the write worked
	BYTE byWriteData = 0;		// Dummy to set up registers for programming
	UINT uiReturn = SUCCESS;	// Return value

	// Turn off block protect just in case
	byWriteData = 0x00;
	if((uiReturn = WriteNormalMem(0x1035, 
								  1, 
								  &byWriteData)) != SUCCESS)
		return uiReturn;
	
	// Set EELAT bit (EEPGM = 0)
	byWriteData = 0x02;
	if((uiReturn = WriteNormalMem(0x103b, 
								  1, 
								  &byWriteData)) != SUCCESS)
		return uiReturn;
	
	// Store data to EEPROM address
	// We expect it to fail since the byte is
	// not programmed yet and the echo will be wrong
	// Therefore, don't test response
	WriteNormalMem(uiAddress, 
				   1, 
				   &byData);
	
	// Set EEPGM bit, i.e. turn on byte programming voltage
	byWriteData = 0x03;
	if((uiReturn = WriteNormalMem(0x103b, 
								  1, 
								  &byWriteData)) != SUCCESS)
		return uiReturn;
	
	// Delay for programming
	Sleep(m_dwEEPROMDelay);
	
	// Turn off high voltage and set to read mode
	byWriteData = 0x00;
	if((uiReturn = WriteNormalMem(0x103b, 
								  1, 
								  &byWriteData)) != SUCCESS)
		return uiReturn;
	
	// Verify that the byte got programmed
	if((uiReturn = DisplayMemory(uiAddress, 
								 1, 
								 &byInitialState)) != SUCCESS)
		return uiReturn;
	else
	{
		// If the byte read back is different than what was supposedly
		// written, then report an error
		if(byInitialState != byData)
			return EEPROGRAM_FAIL;
	}

	return SUCCESS;
}

////////////////////////////////////////////////////////////
// WriteMemory: Parses a memory write request into either
//		a normal write or an EEPROM write and reacts accordingly
//
// Arguments:
//		uiStartAddress : Address to start writing at
//		uiByteCount : Number of bytes to write from uiStartAddress 
//		bypWriteBuffer : Byte array holding data to be written
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.WriteMemory(uiStartAddress,
//                         uiByteCount,
//                         bypWriteBuffer);
//
// Notes : bypWriteBuffer passed by reference since
//		   it is an array of bytes.

UINT CMemory::WriteMemory(const UINT uiStartAddress,
			 			  const UINT uiByteCount,
			 			  BYTE* bypWriteBuffer)
{
	UINT uiEEPROMCount = 0; // Counter to do EEPROM writes since they
							// must be done a byte at a time
	UINT uiReturn = SUCCESS;// Return value
	BYTE* bypWriteStart;	// Pointer to index into the Write buffer passed
							// in by the calling function

	if(m_talker->IsEEPROMEnabled())
	{
		// If the address range is strictly in EEPROM
		if((uiStartAddress >= m_uiEEPROMStart) &&
		   ((uiStartAddress + uiByteCount - 1) <= m_uiEEPROMEnd))
		{
			// Erase the EEPROM first
			uiReturn = EraseEEPROM(uiStartAddress, 
								   uiStartAddress + uiByteCount - 1);
			
			while((uiEEPROMCount < uiByteCount) && (uiReturn == SUCCESS))
			{
				if((uiReturn = WriteEEPROM((uiStartAddress+uiEEPROMCount), 
								           bypWriteBuffer[uiEEPROMCount])) 
										   == SUCCESS)
					uiEEPROMCount++;
			}
		}
		
		// If the address range starts in normal mem and ends in EEPROM
		else if((uiStartAddress < m_uiEEPROMStart) &&
				((uiStartAddress + uiByteCount - 1) > m_uiEEPROMStart) &&
				((uiStartAddress + uiByteCount - 1) <= m_uiEEPROMEnd))
		{
			// Write the normal memory first
			bypWriteStart = bypWriteBuffer;
			uiReturn = WriteNormalMem(uiStartAddress,
						             (m_uiEEPROMStart - uiStartAddress),
							         bypWriteStart);

			if(uiReturn == SUCCESS)
			{
				// Reset the buffer pointer to the new location
				bypWriteStart += (m_uiEEPROMStart - uiStartAddress);

				// Now do the EEPROM, must erase it first
				uiReturn = EraseEEPROM(m_uiEEPROMStart, 
									   uiStartAddress + uiByteCount - 1);
				
				// Now write the EEPROM
				while((uiEEPROMCount < (uiStartAddress + uiByteCount - 
					   m_uiEEPROMStart)) && (uiReturn == SUCCESS))
				{
					if((uiReturn = WriteEEPROM((m_uiEEPROMStart+uiEEPROMCount), 
									           bypWriteStart[uiEEPROMCount])) 
											   == SUCCESS)
						uiEEPROMCount++;
				}
			}
		}
		
		// If the address range starts in EEPROM and ends in normal mem
		else if(((uiStartAddress >= m_uiEEPROMStart) && 
			    (uiStartAddress < m_uiEEPROMEnd)) &&
				((uiStartAddress + uiByteCount - 1) > m_uiEEPROMEnd))
		{
			// Start with EEPROM and erase it first
			uiReturn = EraseEEPROM(uiStartAddress, m_uiEEPROMEnd);

			// Write EEPROM first
			bypWriteStart = bypWriteBuffer;
			while((uiEEPROMCount < (m_uiEEPROMEnd - uiStartAddress + 1)) 
				  && (uiReturn == SUCCESS))
			{
				if((uiReturn = WriteEEPROM((uiStartAddress + uiEEPROMCount), 
							               bypWriteStart[uiEEPROMCount])) 
										   == SUCCESS)
					uiEEPROMCount++;
			}

			if(uiReturn == SUCCESS)
			{
				// Now do the normal memory
				// Reset the buffer pointer to the new location
				bypWriteStart += (m_uiEEPROMEnd - uiStartAddress + 1);

				// Now Write the normal memory
				uiReturn = WriteNormalMem((m_uiEEPROMEnd + 1),
								          (uiStartAddress + uiByteCount - 
										  m_uiEEPROMEnd - 1),
								          bypWriteStart);
			}
		}
		
		// If the address range straddles the EEPROM
		else if((uiStartAddress < m_uiEEPROMStart) &&
				((uiStartAddress + uiByteCount - 1) > m_uiEEPROMEnd))
		{
			// Erase the entire EEPROM first
			uiReturn = EraseSpecificEEPROM(m_uiEEPROMStart, BULKERASE);

			// Write the memory before EEPROM first
			if(uiReturn == SUCCESS)
			{
				bypWriteStart = bypWriteBuffer;
				uiReturn = WriteNormalMem(uiStartAddress,
								          (m_uiEEPROMStart - uiStartAddress),
								          bypWriteStart);
			}

			if(uiReturn == SUCCESS)
			{
				// Reset the buffer pointer to the new location
				bypWriteStart += (m_uiEEPROMStart - uiStartAddress);

				// Now write the EEPROM
				while((uiEEPROMCount < (m_uiEEPROMStart - m_uiEEPROMEnd + 1)) 
					  && (uiReturn == SUCCESS))
				{
					if((uiReturn = WriteEEPROM((m_uiEEPROMStart+uiEEPROMCount), 
									           bypWriteStart[uiEEPROMCount])) 
											   == SUCCESS)
						uiEEPROMCount++;
				}
			}

			if(uiReturn == SUCCESS)
			{
				// Reset the buffer pointer to the new location
				bypWriteStart += (m_uiEEPROMStart - m_uiEEPROMEnd + 1);

				// Now finish the normal memory
				uiReturn = WriteNormalMem((m_uiEEPROMEnd + 1),
								          (uiStartAddress + uiByteCount - 
										  m_uiEEPROMEnd - 1),
										  bypWriteStart);
			}
		}
		
		// Otherwise, just write normal mem
		else
		{
			uiReturn = WriteNormalMem(uiStartAddress, 
									  uiByteCount, 
									  bypWriteBuffer);
		}
	}
	else
	{
		// EEPROM was not enabled, so just write normal memory
		uiReturn = WriteNormalMem(uiStartAddress, 
								  uiByteCount, 
								  bypWriteBuffer);
	}
				
	return uiReturn;
}

////////////////////////////////////////////////////////////
// EraseEEPROM: Clears EEPROM in most efficient fashion which
//		means it takes advantage of any row or bulk erasures
//		it can do over a simple byte erase.
//
// Arguments:
//		uiStartAddress : Address to start clear at
//		uiEndAddress : Address to end clear at
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.EraseEEPROM(uiStartAddress,
//                         uiEndAddress);
//

UINT CMemory::EraseEEPROM(UINT uiStartAddress,
			 			  const UINT uiEndAddress)
{
	char szStart[6];			// Used to convert start address to end
								// of row boundary
	UINT uiByteBoundStart = 0;	// Holds the row boundary in the row that
								// that uiStartAddress is in
	UINT uiReturn = SUCCESS;	// Return value

	// Find upper 16 byte boundary for start address so we can 
	// row erase if at all possible
	
	// First, get the value into string.
	uiByteBoundStart = uiStartAddress;

	sprintf(szStart, "$%04x", uiByteBoundStart);
	
	// Now, convert the last characters to f to align to end of row
	szStart[4] = 'f';

	// Convert back to UINT
	uiByteBoundStart = (UINT)ConvToULong(szStart);
	// uiByteBoundStart now holds upper bound on the row that
	// uiStartAddress is in

	// First erase the odd bytes not in a full 16 byte row
	// i.e. erase from uiStartAddress to the row boundary
	while((uiStartAddress <= uiByteBoundStart) && (uiReturn == SUCCESS) &&
		  (uiStartAddress <= uiEndAddress) && 
		  (((uiStartAddress & 0x000f) | 0x0000) != 0))
	{
		uiReturn = EraseSpecificEEPROM(uiStartAddress, BYTEERASE);
		uiStartAddress++;
	}

	// Now erase rows until get past the end address
	while((uiStartAddress + 0x0f <= uiEndAddress) && (uiReturn == SUCCESS))
	{
		uiReturn = EraseSpecificEEPROM(uiStartAddress, ROWERASE);
		// Move to the next row
		uiStartAddress += 0x10;
	}

	// Clean up the rest of the bytes not in a 16 byte row
	// i.e. erase up to uiEndAddress
	while((uiStartAddress <= uiEndAddress) && (uiReturn == SUCCESS))
	{
		uiReturn = EraseSpecificEEPROM(uiStartAddress, BYTEERASE);
		uiStartAddress++;
	}

	return uiReturn;
};

////////////////////////////////////////////////////////////
// EraseSpecificEEPROM: Clears a selected portion of EEPROM
//
// Arguments:
//		uiAddress : Address to clear at
//		bySelector : Selects bulk, row, or byte erase mode
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.EraseSpecificEEPROM(uiAddress, bySelector);
//

UINT CMemory::EraseSpecificEEPROM(const UINT uiAddress, const BYTE bySelector)
{

	UINT uiReturn = SUCCESS;	// Return value
	BYTE byMode = 0;			// Erase mode to use
	BYTE byVoltage = 0;			// Program voltage to use
	BYTE byWriteData = 0;		// Dummy byte to program EEPROM registers

	switch(bySelector)
	{
		// Erase the byte at uiStartAddress
		case BYTEERASE:
			byMode = 0x16;
			byVoltage = 0x17;
			break;

		// Erase the row that uiStartAddress is in
		case ROWERASE:
			byMode = 0x0e;
			byVoltage = 0x0f;
			break;

		// Erase the entire EEPROM
		case BULKERASE:
			byMode = 06;
			byVoltage = 0x07;
			break;

		// Default to bulk in case by some freak of nature one from above is
		// not selected properly
		default:
			byMode = 06;
			byVoltage = 0x07;
	}
	
	// Turn off block protect just in case
	byWriteData = 0x00;
	if((uiReturn = WriteNormalMem(0x1035, 
								  1, 
								  &byWriteData)) != SUCCESS)
	return uiReturn;
	
	// Set to the selected mode from the switch statement above
	byWriteData = byMode;
	if((uiReturn = WriteNormalMem(0x103b, 
								  1, 
								  &byWriteData)) != SUCCESS)
		return uiReturn;
	
	// Write any data to location to be erased
	// We expect it to fail since the byte is
	// not programmed yet and the echo will be wrong
	// Therefore, we don't check the response...just do it
	WriteNormalMem(uiAddress, 
	 			   1, 
				   &byWriteData);
	
	// Turn on the voltage appropriate for the erasure we are doing
	byWriteData = byVoltage;
	if((uiReturn = WriteNormalMem(0x103b, 
								  1, 
								  &byWriteData)) != SUCCESS)
		return uiReturn;
	
	// Delay for programming
	Sleep(m_dwEEPROMDelay);
	
	// Turn off high voltage and set to read mode
	byWriteData = 0x00;
	if((uiReturn = WriteNormalMem(0x103b, 
								  1, 
								  &byWriteData)) != SUCCESS)
		return uiReturn;

	return uiReturn;
}