// Header file for CMemory (68HC11 Memory Operation Class)
//
// Author: Matt Vandenbush
// Code Review: Matt Vandenbush 
// Date : 12/30/96

// Constants for EEPROM erasure selection
#define BYTEERASE 0
#define ROWERASE 1
#define BULKERASE 2

// Include files
#include "CTalker.h"
#include "Errcodes.h"

class CMemory
{
private:
	CTalker*	m_talker;
	DWORD		m_dwEEPROMDelay;
	UINT		m_uiEEPROMStart;
	UINT		m_uiEEPROMEnd;

public:

	CMemory(const CTalker* talkerObject);
	
////////////////////////////////////////////////////////////
// DisplayMemory: reads a specified number of data bytes
//      from a specified start address on the 68HC11.  
//		Data is "returned" since the byte array is
//	    passed by reference.
//
// Arguments:
//		uiStartAddress : Address to start reading at
//		uiByteCount : Number of bytes to read from uiStartAddress 
//		bypReadBuffer : Byte array holding data after completion
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.DispalyMemory(uiStartAddress,
//							 uiByteCount,
//							 &bypReadBuffer);
//

	UINT DisplayMemory(const UINT uiStartAddress,
					   const UINT uiByteCount,
					   BYTE* bypReadBuffer);

////////////////////////////////////////////////////////////
// WriteNormalMem: writes a specified number of data bytes
//      to a specified start address on the 68HC11.  
//
// Arguments:
//		uiStartAddress : Address to start writing at
//		uiByteCount : Number of bytes to write from uiStartAddress 
//		bypWriteBuffer : Byte array holding data to be written
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.WriteNormalMem(uiStartAddress,
//                            uiByteCount,
//                            &bypWriteBuffer);
//
// Notes : bypWriteBuffer passed by reference since
//		   it is an array of bytes.

	UINT WriteNormalMem(const UINT uiStartAddress,
			 		    const UINT uiByteCount,
			 		    BYTE* bypWriteBuffer);

////////////////////////////////////////////////////////////
// SetEEPROMDelay: Sets the private data member m_dwEEPROMDelay to
//		to the specified DWORD value in milliseconds. 
//
// Arguments:
//		dwDelay : Value to which to set m_dwEEPROMDelay
//
// Return Value:
//		TRUE if successful, otherwise FALSE
//
// Example Usage:
//		memory.SetEEPROMDelay(dwDelay);
//

	BOOL SetEEPROMDelay(const DWORD dwDelay);

////////////////////////////////////////////////////////////
// SetEEPROMRange: Sets the private data members m_uiEEPROMStart and
//		m_uiEEPROMEnd to the specified values. 
//
// Arguments:
//		uiStart : Value to which to set m_uiEEPROMStart
//		uiEnd : Value to which to set m_uiEEPROMEnd
//
// Return Value:
//		TRUE if successful, otherwise FALSE
//
// Example Usage:
//		memory.SetEEPROMRange(uiStart, uiEnd);
//

	BOOL SetEEPROMRange(const UINT uiStart,
						const UINT uiEnd);

////////////////////////////////////////////////////////////
// FillBlock: writes a single data byte a specified number 
//      of times starting from a specified start address on the 68HC11.  
//
// Arguments:
//		uiStartAddress : Address to start writing at
//		uiByteCount : Number of bytes to write from uiStartAddress 
//		byByte : Byte holding data to be written
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.FillBlock(uiStartAddress,
//                       uiByteCount,
//                       byByte);
//

	UINT FillBlock(const UINT uiStartAddress,
	 			   const UINT uiByteCount,
				   BYTE byByte);

////////////////////////////////////////////////////////////
// MoveBlock: moves a block of data from a specified start 
//		to a specified destination address on the 68HC11.  
//
// Arguments:
//		uiStartAddress : Address to start reading at
//		uiByteCount : Number of bytes to read from uiStartAddress 
//		uiDestAddress : Address to start writing at
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.MoveBlock(uiStartAddress,
//                       uiByteCount,
//                       byDestAddress);
//

	UINT MoveBlock(const UINT uiStartAddress,
	 			   const UINT uiByteCount,
				   const UINT uiDestAddress);

////////////////////////////////////////////////////////////
// FindFirstByte : Finds a byte in a specified range.
//
// Arguments:
//		uiStartAddress : Address to start reading at
//		uiByteCount : Number of bytes to read from uiStartAddress 
//		byData : Byte holding byte to search for
//		uiLocation : Address at which byte is found
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.FindFirstByte(uiStartAddress,
//                           uiByteCount,
//                           byData
//						     &uiLocation);
//

	UINT FindFirstByte(const UINT uiStartAddress,
					   const UINT uiByteCount,
					   const BYTE byData,
					   UINT &uiLocation);

////////////////////////////////////////////////////////////
// WriteEEPROM: writes a byte to EEPROM
//
// Arguments:
//		uiAddress : Address to write at
//		byData : Data byte to be written
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.WriteEEPROM(uiAddress,
//                         byData);
//

	UINT WriteEEPROM(const UINT uiAddress,
			 		 BYTE byData);

////////////////////////////////////////////////////////////
// WriteMemory: Parses a memory write request into either
//		a normal write or an EEPROM write and reacts accordingly
//
// Arguments:
//		uiStartAddress : Address to start writing at
//		uiByteCount : Number of bytes to write from uiStartAddress 
//		bypWriteBuffer : Byte array holding data to be written
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.WriteMemory(uiStartAddress,
//                         uiByteCount,
//                         bypWriteBuffer);
//
// Notes : bypWriteBuffer passed by reference since
//		   it is an array of bytes.

	UINT WriteMemory(const UINT uiStartAddress,
			 		 const UINT uiByteCount,
			 		 BYTE* bypWriteBuffer);

////////////////////////////////////////////////////////////
// EraseEEPROM: Clears EEPROM in most efficient fashion
//
// Arguments:
//		uiStartAddress : Address to start clear at
//		uiEndAddress : Address to end clear at
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.EraseEEPROM(uiStartAddress,
//                         uiEndAddress);
//

	UINT EraseEEPROM(UINT uiStartAddress,
			 		 const UINT uiEndAddress);

////////////////////////////////////////////////////////////
// EraseSpecificEEPROM: Clears a selected portion of EEPROM
//
// Arguments:
//		uiAddress : Address to clear at
//		bySelector : Selects bulk, row, or byte erase mode
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		memory.EraseSpecificEEPROM(uiAddress, bySelector);
//

	UINT EraseSpecificEEPROM(const UINT uiAddress, const BYTE bySelector);

};