// Definition of CProgram (68HC11 User Program Class)
//
// Authors: Jason Gaedtke, Matt Vandenbush
// Code Review: Jason Gaedtke
// Date: 12/7/96

// include files
#include "CProgram.h"
#include "CMemory.h"
#include "assert.h"

extern CMemory memory;

// constructor definition
CProgram::CProgram(const CTalker* talkerObject)
{
    // initialize pointer to talker object
    m_talker = (CTalker*) talkerObject; 
    
    // clear running flag
    m_fRunning = FALSE;

    // initialize breakpoint table and tracepoint structures
    for(int i=0; i<MAXBREAKPOINTS; i++)
    {
        m_breakpointTable[i].uiAddress = 0;
        m_breakpointTable[i].byOpCode = 0;
    }
    m_tracepoint1.uiAddress = 0;
    m_tracepoint1.byOpCode = 0;
    m_tracepoint2.uiAddress = 0;
    m_tracepoint2.byOpCode = 0;
}

// destructor definition
CProgram::~CProgram()
{
    // close communications port
    m_talker->ClosePort();
}

////////////////////////////////////////////////////////////
// Init: initializes specified talker
//
// Arguments:
//      cpComPort : communications port
//      byTalkerType : talker type (RAMTALKER, ROMTALKER)
//      cpTalkerFile : talker file
//      byOperatingMode : 68HC11 mode of operation (BOOTSTRAP, SPECIALTEST) 
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.Init("COM2", 
//                   RAMTALKER, 
//                   "c:\talke.s19"
//                   SPECIALTEST);
//
// Note: COMMUNICATIONS STRING MUST NOT CONTAIN SPACES

UINT CProgram::Init(const char* cpComPort, 
                    const BYTE byTalkerType,
                    const char* cpTalkerFile,
                    const BYTE byOperatingMode,
                    const BOOL bEnableEEPROM)
{
    UINT uiResult = SUCCESS;
    UINT uiStackPointer = 0;
    BYTE byCCR = 0; 
    BYTE byACCB = 0;
    BYTE byACCA = 0;
    UINT uiIX = 0;
    UINT uiIY = 0;
    UINT uiProgramCounter = 0;
    UINT uiHPRIO = 0x103c;
    BYTE byHPRIOValue = 0;
    BYTE byData;
    BYTE byConfirmConfig;
	BYTE byWriteData;

    // initialize talker communications
    if(byTalkerType == RAMTALKER)
    {
        if((uiResult = m_talker->LoadRAMTalker(cpComPort, cpTalkerFile)) != SUCCESS)
            return uiResult;
    }
    else if(byTalkerType == ROMTALKER)
    {
        if((uiResult = m_talker->LoadROMTalker(cpComPort, cpTalkerFile)) != SUCCESS)
            return uiResult;
    }
    else
        return INVALID_TALKER_TYPE;

    // Clear the BPROT register
	// Have to do it run away to make sure we can write to CONFIG
	byWriteData = 0x00;
    if((uiResult = memory.WriteMemory(0x1035, 
                                      1, 
                                      &byWriteData)) != SUCCESS)
		return uiResult;

	// read registers (to determine location of idle loop)
    if((uiResult = m_talker->ReadRegisters(uiStackPointer,
                                           byCCR,
                                           byACCB,
                                           byACCA,
                                           uiIX,
                                           uiIY,
                                           uiProgramCounter)) != SUCCESS)
        return uiResult;
    
    // store talker idle address
    m_uiTalkerIdle = uiProgramCounter;
    m_uiProgramExecution = uiProgramCounter;
    
    // clear X bit in CCR (enable nonmaskable interrupts)\
	// SEBERN 4/8/97: Are we sure this is a good idea?
    byCCR = byCCR;// & 0xbf;
    if((uiResult = m_talker->WriteRegisters(uiStackPointer,
                                            byCCR,
                                            byACCB,
                                            byACCA,
                                            uiIX,
                                            uiIY,
                                            uiProgramCounter)) != SUCCESS)
        return uiResult;

    // test EEPROM flag
    if(bEnableEEPROM != m_talker->IsEEPROMEnabled())
    {
        if(m_talker->IsEEPROMEnabled())
        {
            byData = 0x0c;  // disable EEPROM
        }
        else
        {
            byData = 0x0d;  // enable EEPROM
        }

        // erase config register
        memory.EraseSpecificEEPROM(0x103f, BYTEERASE);

        // write config register
        memory.WriteEEPROM(0x103f, byData);

        // reset talker
        if((uiResult = m_talker->Reset()) != SUCCESS)
            return uiResult;

        // Clear the BPROT register for later writes to EEPROM
		byWriteData = 0x00;
		if((uiResult = memory.WriteMemory(0x1035, 
										  1, 
                                          &byWriteData)) != SUCCESS)
		return uiResult;

		// read and confirm config register value
        if((uiResult = memory.DisplayMemory(0x103f, 
                                            1, 
                                            &byConfirmConfig)) != SUCCESS)
        {
            return uiResult;
        }
        else
        {
            // test new config value
            if(byConfirmConfig != byData)
            {
                return REGISTER_WRITE_FAIL;
            }
            else
            {
                // update EEPROM flag in talker class
                m_talker->SetEEPROMFlag(bEnableEEPROM);
            }
        }
    }

    // set specified operating mode (BOOTSTRAP, SPECIALTEST)
    if(byOperatingMode == SPECIALTEST)
    {
        byHPRIOValue = 0xe5;
        if((uiResult = m_talker->WriteMemory(uiHPRIO,
                                             1,         
                                             &byHPRIOValue)) != SUCCESS)
            return uiResult;
    }

    AfxMessageBox("Talker initialized successfully.");

  return uiResult;
}

////////////////////////////////////////////////////////////
// LoadSRecordFile: opens specified S19 file and writes
//      program to 68HC11 memory
//
// Arguments:
//      cpFileName : user S19 file
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.LoadSRecordFile("c:\user.s19");
//
// Note: SPECIFIED FILE MUST BE A COMPILED S19 FILE

UINT CProgram::LoadSRecordFile(const char* cpFileName)
{
    UINT uiResult = SUCCESS;
    CFile userFile;                 // user file object
    UINT uiFileLength = 0;          // user file size
    BYTE bypBuffer[MAXS19SIZE];     // byte array for buffering user file
    CString strBuffer;              // CString buffer for user file
    int iIndex;                     // index into S19 file CString buffer
    BYTE byCounter = 0;             // index into current line of S19 file          
    BYTE byLineBytes = 0;           // counter: number of data bytes on line
    UINT uiAddress = 0;             // memory location for current line
    CString strASCII;               // ASCII data (two characters)
    BYTE bypWrite[60];              // byte array for writing one line (of S19)
    BOOL fFirstLine = TRUE;         // flags first line of S19 file
    
    // open file
    if(!(userFile.Open(cpFileName, 
                       CFile::modeRead | CFile::shareCompat)))
        return OPEN_S19_FAIL;
    m_strProgramFile = cpFileName;

    // read file length
    uiFileLength = userFile.GetLength();
    if(uiFileLength > MAXS19SIZE)
       return INVALID_S19; 
    
    // read file into byte array (max bytes = MAXS19SIZE)
    if(userFile.Read(bypBuffer, uiFileLength) != uiFileLength)
        return READ_S19_FAIL;

    // close file
    userFile.Abort();   

    // convert byte array to CString object
    strBuffer = bypBuffer;

    // process S19 file
    while ((iIndex = strBuffer.Find('S')) != -1)    // more valid data
    {
        // overwrite current 'S' character
        strBuffer.SetAt(iIndex, '0');

        // test S19 format ('1' or '9')
        if((strBuffer[iIndex+1] != '1') && (strBuffer[iIndex+1] != '9'))
            return INVALID_S19;

        // read number of data bytes (on line)
        strASCII = strBuffer[iIndex+2];     // high char of line byte count
        strASCII += strBuffer[iIndex+3];    // low char of line byte count
        byLineBytes = (BYTE) strtoul(strASCII, NULL, 16);
        
        // read memory address of data bytes (on 68HC11)
        strASCII = strBuffer[iIndex+4];     // high char of high byte
        strASCII += strBuffer[iIndex+5];    // low char of high byte
        strASCII += strBuffer[iIndex+6];    // high char of low byte
        strASCII += strBuffer[iIndex+7];    // low char of low byte
        uiAddress = (UINT) strtoul(strASCII, NULL, 16);

        // process each data byte (two character pair)
        byLineBytes -= 3;   // ignore address bytes (2) & checksum (1)
		// SEBERN 4/8/97 Buffer not big enough before.
		// Increased size and added test.
		assert (byLineBytes <= sizeof bypWrite);
        iIndex += 8;        // advance pointer to first data char
        for(byCounter=0; byCounter<byLineBytes; byCounter++)
        {
            // read data byte (two chars)
            strASCII = strBuffer[iIndex+2*byCounter];
            strASCII += strBuffer[iIndex+2*byCounter+1];
            bypWrite[byCounter] = (BYTE) strtoul(strASCII, NULL, 16);
        }

        // set program execution address (best guess)
        if(fFirstLine)
        {
            m_uiProgramExecution = uiAddress;
            fFirstLine = FALSE;
        }
        
        // write S19 line to 68HC11 memory
        if(byLineBytes != 0)    // skip empty lines
            if((uiResult = m_talker->WriteMemory(uiAddress,
                                                 byLineBytes,
                                                 bypWrite)) != SUCCESS)
                return uiResult;
    }
    
    return uiResult;
}

////////////////////////////////////////////////////////////
// VerifySRecordFile: opens specified S19 file and verifies
//      program in 68HC11 memory
//
// Arguments:
//      cpFileName : user S19 file
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.VerifySRecordFile("c:\user.s19");
//
// Note: SPECIFIED FILE MUST BE A COMPILED S19 FILE

UINT CProgram::VerifySRecordFile(const char* cpFileName)
{
    UINT uiResult = SUCCESS;
    CFile userFile;                 // user file object
    UINT uiFileLength = 0;          // user file size
    BYTE bypBuffer[MAXS19SIZE];     // byte array for user file
    CString strBuffer;              // CString buffer for user file
    int iIndex;                     // index into S19 file CString buffer
    BYTE byCounter = 0;             // index into current line of S19 file          
    BYTE byLineBytes = 0;           // counter: number of data bytes on line
    UINT uiAddress = 0;             // memory location for current line
    CString strASCII;               // ASCII data (two characters)
    BYTE bypFile[40];               // byte array for reading one line (of S19)
    BYTE bypMemory[40];             // byte array for reading 68HC11 memory

    // open file
    if(!(userFile.Open(cpFileName, 
                       CFile::modeRead | CFile::shareCompat)))
        return OPEN_S19_FAIL;

    // read file length
    uiFileLength = userFile.GetLength();
    if(uiFileLength > MAXS19SIZE)
        return INVALID_S19;
    
    // read file into byte array (max bytes = MAXS19SIZE)
    if(userFile.Read(bypBuffer, uiFileLength) != uiFileLength)
        return READ_S19_FAIL;

    // close file object
    userFile.Abort();

    // convert byte array to CString object
    strBuffer = bypBuffer;

    // process S19 file
    while ((iIndex = strBuffer.Find('S')) != -1)    // more valid data
    {
        // overwrite current 'S' character
        strBuffer.SetAt(iIndex, '0');

        // test S19 format ('1' or '9')
        if((strBuffer[iIndex+1] != '1') && (strBuffer[iIndex+1] != '9'))
            return INVALID_S19;

        // read number of data bytes (on line)
        strASCII = strBuffer[iIndex+2];     // high char of line byte count
        strASCII += strBuffer[iIndex+3];    // low char of line byte count
        byLineBytes = (BYTE) strtoul(strASCII, NULL, 16);
        
        // read memory address of data bytes (on 68HC11)
        strASCII = strBuffer[iIndex+4];     // high char of high byte
        strASCII += strBuffer[iIndex+5];    // low char of high byte
        strASCII += strBuffer[iIndex+6];    // high char of low byte
        strASCII += strBuffer[iIndex+7];    // low char of low byte
        uiAddress = (UINT) strtoul(strASCII, NULL, 16);

        // process each data byte (two character pair)
        byLineBytes -= 3;   // ignore address bytes (2) & checksum (1)
        iIndex += 8;        // advance pointer to first data char
        for(byCounter=0; byCounter<byLineBytes; byCounter++)
        {
            // read data byte (two chars)
            strASCII = strBuffer[iIndex+2*byCounter];
            strASCII += strBuffer[iIndex+2*byCounter+1];
            bypFile[byCounter] = (BYTE) strtoul(strASCII, NULL, 16);
        }

        // read 68HC11 memory

        if(byLineBytes != 0)    // skip empty lines
            if((uiResult = m_talker->ReadMemory(uiAddress,
                                                byLineBytes,
                                                bypMemory)) != SUCCESS)
                return uiResult;

        // compare S19 data and 68HC11 memory
        for(byCounter=0; byCounter<byLineBytes; byCounter++)
            if(bypFile[byCounter] != bypMemory[byCounter])
                return INVALID_S19_MEMORY;
    }
    
    return uiResult;
}

////////////////////////////////////////////////////////////
// ExecuteProgram:  executes from specified location
//
// Arguments:
//      uiStartAddress : starting address of user program
//		fBreakpoints : flag enabling and disabling user breakpoints
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.ExecuteProgram(uiStartAddress);

UINT CProgram::ExecuteProgram(const UINT uiStartAddress,
							  const BOOL fBreakpoints)
{
    UINT uiResult = SUCCESS;
    UINT uiStackPointer = 0;
    BYTE byCCR = 0; 
    BYTE byACCB = 0;
    BYTE byACCA = 0;
    UINT uiIX = 0;
    UINT uiIY = 0;
    UINT uiProgramCounter = 0;

    // update program execution address
    m_uiProgramExecution = uiStartAddress;
    
    // place breakpoint SWI opcodes in 68HC11 memory
    if(fBreakpoints)
		if((uiResult = PlaceBreakpoints()) != SUCCESS)
			return uiResult;
    
    // read current 68HC11 status
    if((uiResult = m_talker->ReadRegisters(uiStackPointer,
                                           byCCR,
                                           byACCB,
                                           byACCA,
                                           uiIX,
                                           uiIY,
                                           uiProgramCounter)) != SUCCESS)
        return uiResult;
    
    // replace program counter with specified program address
    uiProgramCounter = uiStartAddress;
    if((uiResult = m_talker->WriteRegisters(uiStackPointer,
                                            byCCR,
                                            byACCB,
                                            byACCA,
                                            uiIX,
                                            uiIY,
                                            uiProgramCounter)) != SUCCESS)
        return uiResult;

    return uiResult;
}

////////////////////////////////////////////////////////////
// ExecuteSubroutine:  executes subroutine and returns 
//      control to the talker idle loop
//
// Arguments:
//      uiSubAddress : starting address of user subroutine
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.ExecuteSubroutine(uiSubAddress);

UINT CProgram::ExecuteSubroutine(const UINT uiSubAddress)
{
    UINT uiResult = SUCCESS;
    UINT uiStackPointer = 0;
    BYTE byCCR = 0; 
    BYTE byACCB = 0;
    BYTE byACCA = 0;
    UINT uiIX = 0;
    UINT uiIY = 0;
    UINT uiProgramCounter = 0;
    BYTE bypReturnAddress[2];

    // read current 68HC11 status
    if((uiResult = m_talker->ReadRegisters(uiStackPointer,
                                           byCCR,
                                           byACCB,
                                           byACCA,
                                           uiIX,
                                           uiIY,
                                           uiProgramCounter)) != SUCCESS)
        return uiResult;
    
    // move stack frame down two bytes (to make room for RTS return address)
    uiStackPointer -= 2;
    if((uiResult = m_talker->WriteRegisters(uiStackPointer,
                                            byCCR,
                                            byACCB,
                                            byACCA,
                                            uiIX,
                                            uiIY,
                                            uiProgramCounter)) != SUCCESS)
        return uiResult;            
    
    // write return address to stack (above SCI frame)
    uiStackPointer += 9;
    bypReturnAddress[0] = (m_uiTalkerIdle & 0xff00) >> 8;
    bypReturnAddress[1] = m_uiTalkerIdle & 0xff;
    if((uiResult = m_talker->WriteMemory(uiStackPointer,    // address to write
                                         2,                 // bytes to write
                                         bypReturnAddress)) != SUCCESS)
        return uiResult;
    
    // replace program counter with specified program address
    uiStackPointer -= 9;    // restore stack previous pointer
    uiProgramCounter = uiSubAddress;
    if((uiResult = m_talker->WriteRegisters(uiStackPointer,
                                            byCCR,
                                            byACCB,
                                            byACCA,
                                            uiIX,
                                            uiIY,
                                            uiProgramCounter)) != SUCCESS)
        return uiResult;

    return uiResult;
}

////////////////////////////////////////////////////////////
// StopExecution:  stops execution of current program or
//      subroutine and stores address for future reference;
//      control then returns to the talker idle loop
//
// Arguments:
//      none
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.StopExecution();

UINT CProgram::StopExecution()
{
    UINT uiResult = SUCCESS;
    UINT uiStackPointer = 0;
    BYTE byCCR = 0; 
    BYTE byACCB = 0;
    BYTE byACCA = 0;
    UINT uiIX = 0;
    UINT uiIY = 0;
    UINT uiProgramCounter = 0;

    // read registers
    if((uiResult = m_talker->ReadRegisters(uiStackPointer,
                                           byCCR,
                                           byACCB,
                                           byACCA,
                                           uiIX,
                                           uiIY,
                                           uiProgramCounter)) != SUCCESS)
        return uiResult;            
    
    // save current program counter for later reference
    m_uiProgramExecution = uiProgramCounter;

    // replace program counter with talker idle loop
    uiProgramCounter = m_uiTalkerIdle;
    if((uiResult = m_talker->WriteRegisters(uiStackPointer,
                                            byCCR,
                                            byACCB,
                                            byACCA,
                                            uiIX,
                                            uiIY,
                                            uiProgramCounter)) != SUCCESS)
        return uiResult;

    // replace user (breakpoint) opcodes in 68HC11 memory
    if((uiResult = ClearBreakpoints()) != SUCCESS)
        return uiResult;
    
    // clear running flag
	m_fRunning = FALSE;
	
	return uiResult;
}

////////////////////////////////////////////////////////////
// SetBreakpoint: places a new breakpoint entry for the specified
//      address in the breakpoint table
//
// Arguments:
//      uiAddress : memory address for new breakpoint
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.SetBreakpoint(uiAddress);
//
// Note: REDUNDANT BREAKPOINTS ARE NOT ALLOWED

UINT CProgram::SetBreakpoint(const UINT uiAddress)
{
    UINT uiResult = SUCCESS;
    BYTE bySWI = 0x3f;
    BYTE byOpCode = 0;
    CString cpMnemonic;
    CString cpAddressMode;
    BYTE byByteCount = 0;
    BOOL fBranchFlag = FALSE;
    int i = 0;

    // search breakpoint table for requested address
    while((i<MAXBREAKPOINTS) && (m_breakpointTable[i].uiAddress != uiAddress))
        i++;

    // return if breakpoint is found
    if(m_breakpointTable[i].uiAddress == uiAddress)
        return BREAKPOINT_SET;  // error: breakpoint match in table

    // search breakpoint table for empty slot
    i=0;
    while((i<MAXBREAKPOINTS) && (m_breakpointTable[i].uiAddress != 0))
        i++;
    if(i == MAXBREAKPOINTS)
        return BREAKPOINT_LIMIT;    // error: table is full

    // read opcode at specified memory address
    if((uiResult = m_talker->ReadMemory(uiAddress,
                                        1,
                                        &byOpCode)) != SUCCESS)
        return uiResult;

    // test for valid opcode (an attempt to prevent crashes)
    if(!(DisassembleOpCode(byOpCode,
                           cpMnemonic,
                           cpAddressMode,
                           byByteCount,
                           fBranchFlag)))
        return INVALID_OPCODE;      // error: invalid opcode at address

    // update breakpoint table
    m_breakpointTable[i].uiAddress = uiAddress;
    m_breakpointTable[i].byOpCode = byOpCode;

    return uiResult;
}

////////////////////////////////////////////////////////////
// RemoveBreakpoint: clears the entry corresponding to the specified
//      address in the breakpoint table
//
// Arguments:
//      uiAddress : memory address of breakpoint to remove
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.RemoveBreakpoint(uiAddress);

UINT CProgram::RemoveBreakpoint(const UINT uiAddress)
{
    UINT uiResult = SUCCESS;
    int i = 0;

    // search breakpoint table for specified breakpoint

   while((i<MAXBREAKPOINTS) && (m_breakpointTable[i].uiAddress != uiAddress))
        i++;
    if(i == MAXBREAKPOINTS)
        return BREAKPOINT_NOT_FOUND;    // error: breakpoint not in table

    // update breakpoint table
    m_breakpointTable[i].uiAddress = 0;
    m_breakpointTable[i].byOpCode = 0;

    return uiResult;
}

////////////////////////////////////////////////////////////
// PlaceBreakpoints: replaces user opcodes with SWI opcode
//      for each breakpoint entry in table
//
// Arguments:
//      none
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.PlaceBreakpoints();
//
// Note: BREAKPOINTS ARE NOT PLACED IN MEMORY UNTIL THIS FUNCTION
//      IS CALLED (PRIOR TO EXECUTION)

UINT CProgram::PlaceBreakpoints()
{
    UINT uiResult = SUCCESS;
    BYTE bySWI = 0x3f;      // SWI opcode
    int i = 0;

    // search breakpoint table for valid breakpoints
    while((i<MAXBREAKPOINTS) && (uiResult == SUCCESS))
    {
        // place breakpoint opcode in memory
        if(m_breakpointTable[i].uiAddress != 0)
            uiResult = m_talker->WriteMemory(m_breakpointTable[i].uiAddress,
                                             1, 
                                             &bySWI);   // SWI opcode
        i++;
    }

    // set running flag (to enable OnIdle processing)
    if(uiResult == SUCCESS)
        m_fRunning = TRUE;

    return uiResult;
}

////////////////////////////////////////////////////////////
// ClearBreakpoints: replaces SWI opcodes with user opcode
//      for each breakpoint table entry
//
// Arguments:
//      none
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.ClearBreakpoints();
//
// Note: THIS ROUTINE WORKS IN CONJUCTION WITH PlaceBreakpoints TO
//      MANAGE SWI OPCODES IN 68HC11 MEMORY

UINT CProgram::ClearBreakpoints()
{
    UINT uiResult = SUCCESS;
    int i = 0;

    // clear running flag (to disable OnIdle processing)
    m_fRunning = FALSE;

    // search breakpoint table for valid breakpoints
    while((i<MAXBREAKPOINTS) && (uiResult == SUCCESS))
    {   
        // replace user opcode in memory (remove SWI opcode)
        if(m_breakpointTable[i].uiAddress != 0)
            uiResult = m_talker->WriteMemory(m_breakpointTable[i].uiAddress,
                                             1,         
                                             &m_breakpointTable[i].byOpCode);
        i++;
    }
    
    return uiResult;
}

////////////////////////////////////////////////////////////
// ListBreakpoints: returns a list of memory addresses   
//      corresponding with current breakpoints
//
// Arguments:
//      uiBreakpoints[MAXBREAKPOINTS] : array of unsigned integers 
//          containing breakpointed memory addresses
//
// Return Value:
//      number of breakpoint entries in table
//
// Example Usage:
//      program.ListBreakpoints(uiBreakpoints[MAXBREAKPOINTS]);

BYTE CProgram::ListBreakpoints(UINT* uiBreakpoints)
{
    int i;
    BYTE byCount = 0;

    // process breakpoint table, updating UINT array
    for(i=0; i<MAXBREAKPOINTS; i++)
    {
        uiBreakpoints[i] = m_breakpointTable[i].uiAddress;
        if (uiBreakpoints[i] != 0)
            byCount++;
    }

    return byCount;
}

////////////////////////////////////////////////////////////
// TraceInstruction: places tracepoint(s) after instruction
//      at specified address and then executes instruction;
//      tracepoint(s) is/are removed prior to returning control
//      to user
//
// Arguments:
//      uiAddress : address of instruction to trace
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.TraceInstruction(uiAddress);

UINT CProgram::TraceInstruction(const UINT uiAddress)
{
    UINT uiResult = SUCCESS;
    UINT uiTracepoint = 0;      // memory address for tracepoint
    char cRelative = 0;         // signed char used for relative branches
    BYTE bypByteArray[6];       // storage for six memory bytes (max. required)
    BYTE bypReturnAddress[2];   // storage for return address (RTS opcode)
    BYTE byOffset = 0;          // offset of opcode in memory array
    BYTE byPage = PAGE1;        // current instruction page (1,2,3,4)
    BOOL fSetStandard = TRUE;   // flag for standard tracepoint (after instruction)
    CString cpMnemonic;         // storage for instruction mnemonic
    CString cpAddressMode;      // storage for instruction addressing mode
    BYTE byByteCount = 0;       // number of bytes in current instruction
    BOOL fBranchFlag = FALSE;   // flag noting special branching instructionS
    UINT uiProgramCounter = 0;
    UINT uiStackPointer = 0;
    BYTE byCCR = 0;
    BYTE byACCA = 0;
    BYTE byACCB = 0;
    UINT uiIX = 0;
    UINT uiIY = 0;

    // read and buffer user instruction (from 68HC11 memory) 
    if((uiResult = m_talker->ReadMemory(uiAddress,      
                                        6,  
                                        bypByteArray)) != SUCCESS)
        return uiResult;

    // test for prefix bytes (used to change instruction pages)
    if(bypByteArray[0] == 0x18)
    {
        byPage = PAGE2;
        byOffset++;
    }
    else if(bypByteArray[0] == 0x1a)
    {
        byPage = PAGE3;
        byOffset++;
    }
    else if(bypByteArray[0] == 0xcd)
    {
        byPage = PAGE4;
        byOffset++;
    }

    // disassemble opcode
    if(!(DisassembleOpCode(bypByteArray[byOffset],
                           cpMnemonic,
                           cpAddressMode,
                           byByteCount,
                           fBranchFlag)))
        return INVALID_OPCODE;      // error: invalid opcode at address

    // process special branching instructions (noted during disassembly)
    if(fBranchFlag)
    {
        // process branch if set & branch if clear
        if((cpMnemonic == "BRSET") || 
           (cpMnemonic == "BRCLR"))
        {
            // process direct addressing mode
            if(cpAddressMode == "DIR")
                if((uiResult = SetTracepoint(bypByteArray[3])) != SUCCESS)
                    return uiResult;
            
            // process indexed addressing mode
            else if(cpAddressMode == "IND")
            {
                // read registers
                if((uiResult = m_talker->ReadRegisters(uiStackPointer,
                                                       byCCR,
                                                       byACCB,
                                                       byACCA,
                                                       uiIX,
                                                       uiIY,
                                                       uiProgramCounter)) != SUCCESS)
                    return uiResult;

                // IX index register
                if(byPage == PAGE1)
                {
                    // calculate and place tracepoint
                    uiTracepoint = uiIX + bypByteArray[3];
                    if((uiResult = SetTracepoint(uiTracepoint)) != SUCCESS)
                        return uiResult;
                }
                // IY index register
                else if(byPage == PAGE2)
                {
                    // calculate and place tracepoint
                    uiTracepoint = uiIY + bypByteArray[4];
                    if((uiResult = SetTracepoint(uiTracepoint)) != SUCCESS)
                        return uiResult;
                }
            }
        }
        // process jump & branch always
        else if((cpMnemonic == "JMP") ||
                (cpMnemonic == "BRA") ||
                (cpMnemonic == "JSR") ||
                (cpMnemonic == "BSR"))
        {
            // clear normal breakpoint flag (instruction always branches)
            fSetStandard = FALSE;

            // process direct addressing mode
            if(cpAddressMode == "DIR")
            {
                if((uiResult = SetTracepoint(bypByteArray[3])) != SUCCESS)
                    return uiResult;
            }

            // process extended addressing mode
            else if(cpAddressMode == "EXT")
            {
                // calculate and place tracepoint
                uiTracepoint = bypByteArray[1] << 8;
                uiTracepoint += bypByteArray[2];
                if((uiResult = SetTracepoint(uiTracepoint)) != SUCCESS)
                    return uiResult;
            }

            // process indexed addressing mode
            else if(cpAddressMode == "IND")
            {
                // read registers
                if((uiResult = m_talker->ReadRegisters(uiStackPointer,
                                                       byCCR,
                                                       byACCB,
                                                       byACCA,
                                                       uiIX,
                                                       uiIY,
                                                       uiProgramCounter)) != SUCCESS)
                    return uiResult;

                // IX index register
                if(byPage == PAGE1)
                {
                    // calculate and place tracepoint
                    uiTracepoint = uiIX + bypByteArray[1];
                    if((uiResult = SetTracepoint(uiTracepoint)) != SUCCESS)
                        return uiResult;
                }
                // IY index register
                else if(byPage == PAGE2)
                {
                    // calculate and place tracepoint
                    uiTracepoint = uiIY + bypByteArray[3];
                    if((uiResult = SetTracepoint(uiTracepoint)) != SUCCESS)
                        return uiResult;
                }
            }

            // process relative addressing mode
            else
            {
                // calculate and place tracepoint
                cRelative = (signed char) bypByteArray[1];
                uiTracepoint = uiAddress +      // base address
                               byByteCount +    // instruction byte count (2)
                               cRelative;       // relative offset
                if((uiResult = SetTracepoint(uiTracepoint)) != SUCCESS)
                    return uiResult;
            }   
        }

        // process return from subroutine (RTS)
        else if(cpMnemonic == "RTS")
        {
            // clear normal breakpoint flag (instruction always branches)
            fSetStandard = FALSE;

            // read registers
            if((uiResult = m_talker->ReadRegisters(uiStackPointer,
                                                   byCCR,
                                                   byACCB,
                                                   byACCA,
                                                   uiIX,
                                                   uiIY,
                                                   uiProgramCounter)) != SUCCESS)
                return uiResult;

            // read memory (RTS return address)
            if((uiResult = m_talker->ReadMemory(uiStackPointer + 9,
                                                2,
                                                bypReturnAddress)) != SUCCESS)
                return uiResult;

            // calculate and place tracepoint
            uiTracepoint = bypReturnAddress[0] << 8;
            uiTracepoint += bypReturnAddress[1];
            if((uiResult = SetTracepoint(uiTracepoint)) != SUCCESS)
                return uiResult;
        }
        
        // process general relative branch instructions (assorted)
        else
        {
            // calculate and place tracepoint
            cRelative = (signed char) bypByteArray[1];
            uiTracepoint = uiAddress +      // base address
                           byByteCount +    // instruction byte count (2)
                           cRelative;       // relative offset
            if((uiResult = SetTracepoint(uiTracepoint)) != SUCCESS)
                return uiResult;
        }
    }

    // place standard tracepoint (after current instruction)
    if(fSetStandard)
    {
        // calculate and set tracepoint
        uiTracepoint = uiAddress +      // base address 
                       byOffset +       // prefix bytes
                       byByteCount;     // instruction byte count
        if((uiResult = SetTracepoint(uiTracepoint)) != SUCCESS)
            return uiResult;
    }

    // execute instruction at specified address
    if((uiResult = ExecuteProgram(uiAddress, FALSE)) != SUCCESS)

            return uiResult;



    // test for SWI signal

    if(!(m_talker->CheckSWI()))     // try again ...

    {

        // delay for execution (constant defined in header file)

        Sleep(EXECUTEDELAY);

        if(!(m_talker->CheckSWI()))

        {

            AfxMessageBox("68HC11 failed to return control to WinBug11.");

            return TRACE_FAILED;

        }

    }



    // acknowledge SWI signal and receive control registers

    if(m_talker->ProcessSWI(uiProgramCounter,

                            uiStackPointer,

                            byCCR,

                            byACCB,

                            byACCA,

                            uiIX,

                            uiIY) != SUCCESS)

    {

        AfxMessageBox("SWI confirmation failed.");

        return TRACE_FAILED;

    }



    // update execution addresss

    m_uiProgramExecution = uiProgramCounter;



    // remove tracepoints from 68HC11 memory (restore opcodes)

    if(RemoveTracepoints() != SUCCESS)

    {

        AfxMessageBox("Error: unable to remove all tracepoints.");

        return TRACE_FAILED;

    }



    return uiResult;

}



////////////////////////////////////////////////////////////

// SetTracepoint: places a new tracepoint at specified address

//      and initializes a tracepoint structure

//

// Arguments:

//      uiAddress : memory address for new tracepoint

//

// Return Value:

//      SUCCESS if successful, otherwise error code

//

// Example Usage:

//      program.SetTracepoint(uiAddress);

//

// Note: THIS IS A PRIVATE UTILITY FUNCTION CALLED BY TraceInstruction



UINT CProgram::SetTracepoint(const UINT uiAddress)

{

    UINT uiResult = SUCCESS;

    BYTE byByte = 0;

    BYTE bySWI = 0x3f;



    // read user opcode

    if((uiResult = m_talker->ReadMemory(uiAddress,      

                                        1,  

                                        &byByte)) != SUCCESS)

        return uiResult;



    // place SWI opcode

    if((uiResult = m_talker->WriteMemory(uiAddress,     

                                         1, 

                                         &bySWI)) != SUCCESS)

        return uiResult;



    // update tracepoint structure

    if(m_tracepoint1.uiAddress == 0)

    {

        m_tracepoint1.uiAddress = uiAddress;

        m_tracepoint1.byOpCode = byByte;

    }

    // m_tracepoint1 already set; set m_tracepoint2

    else    

    {

        m_tracepoint2.uiAddress = uiAddress;

        m_tracepoint2.byOpCode = byByte;

    }



    return uiResult;

}



////////////////////////////////////////////////////////////

// RemoveTracepoints: replaces original user opcodes and 

//      clears tracepoint structures

//

// Arguments:

//      none

//

// Return Value:

//      SUCCESS if successful, otherwise error code

//

// Example Usage:

//      program.RemoveTracepoints();

//

// Note: THIS IS A PRIVATE UTILITY FUNCTION CALLED BY TraceInstruction



UINT CProgram::RemoveTracepoints()

{

    UINT uiResult = SUCCESS;



    // replace user opcode(s)

    if(m_tracepoint1.uiAddress != 0)

        if((uiResult = m_talker->WriteMemory(m_tracepoint1.uiAddress,       

                                             1, 

                                             &m_tracepoint1.byOpCode)) != SUCCESS)

            return uiResult;

    

    if(m_tracepoint2.uiAddress != 0)

        if((uiResult = m_talker->WriteMemory(m_tracepoint2.uiAddress,       

                                             1, 

                                             &m_tracepoint2.byOpCode)) != SUCCESS)

            return uiResult;



    // reset tracepoint structures

    m_tracepoint1.uiAddress = 0;

    m_tracepoint1.byOpCode = 0;

    m_tracepoint2.uiAddress = 0;

    m_tracepoint2.byOpCode = 0;



    return uiResult;

}



////////////////////////////////////////////////////////////

// DisassembleOpCode:  processes specified 68HC11 opcode

//      and returns a mnemonic, addressing mode, byte count,

//      prefix flag and branch flag; (if the specified opcode

//      is invalid the routine returns FALSE)

//

// Arguments:

//      byOpCode : single byte opcode to disassemble

//      cpMnemonic : opcode mnemonic

//      cpAddressMode : opcode addressing mode

//      byByteCount : number of bytes in instruction

//      fBranchFlag : set if opcode is special branch instruction

//

// Return Value:

//      TRUE if opcode is valid, otherwise FALSE

//

// Example Usage:

//      program.DisassembleOpCode(byOpcode, 

//                                cpMnemonic,

//                                cpAddressMode,

//                                byByteCount,

//                                fBranchFlag);



BOOL CProgram::DisassembleOpCode(const BYTE byOpCode,

                                 CString& cpMnemonic,

                                 CString& cpAddressMode,

                                 BYTE& byByteCount,

                                 BOOL& fBranchFlag)

{

    // set default for branch flag

    fBranchFlag = FALSE;



    // switch based on opcode

    switch (byOpCode) 

    {

        case 0x00 : 

            cpMnemonic = "TEST";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x01 : 

            cpMnemonic = "NOP";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x02 : 

            cpMnemonic = "IDIV";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x03 : 

            cpMnemonic = "FDIV";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x04 : 

            cpMnemonic = "LSRD";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x05 : 

            cpMnemonic = "ASLD/LSLD";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x06 : 

            cpMnemonic = "TAP";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x07 : 

            cpMnemonic = "TPA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x08 : 

            cpMnemonic = "INX/INY";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x09 : 

            cpMnemonic = "DEX/DEY";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x0A : 

            cpMnemonic = "CLV";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x0B : 

            cpMnemonic = "SEV";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x0C : 

            cpMnemonic = "CLC";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x0D : 

            cpMnemonic = "SEC";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x0E : 

            cpMnemonic = "CLI";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x0F : 

            cpMnemonic = "SEI";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x10 : 

            cpMnemonic = "SBA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x11 : 

            cpMnemonic = "CBA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x12 : 

            cpMnemonic = "BRSET";

            cpAddressMode = "DIR";

            byByteCount = 4;

            fBranchFlag = TRUE;

            break; 

        case 0x13 : 

            cpMnemonic = "BRCLR";

            cpAddressMode = "DIR";

            byByteCount = 4;

            fBranchFlag = TRUE;

            break; 

        case 0x14 : 

            cpMnemonic = "BSET";

            cpAddressMode = "DIR";

            byByteCount = 3;

            break; 

        case 0x15 : 

            cpMnemonic = "BCLR";

            cpAddressMode = "DIR";

            byByteCount = 3;

            break; 

        case 0x16 : 

            cpMnemonic = "TAB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x17 : 

            cpMnemonic = "TBA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x18 : 

            cpMnemonic = "PAGE2";

            cpAddressMode = "CTRL";

            byByteCount = 1;

            break; 

        case 0x19 : 

            cpMnemonic = "DAA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x1A : 

            cpMnemonic = "PAGE3";

            cpAddressMode = "CTRL";

            byByteCount = 1;

            break; 

        case 0x1B : 

            cpMnemonic = "ABA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x1C : 

            cpMnemonic = "BSET";

            cpAddressMode = "IND";

            byByteCount = 3;

            break; 

        case 0x1D : 

            cpMnemonic = "BCLR";

            cpAddressMode = "IND";

            byByteCount = 3;

            break; 

        case 0x1E : 

            cpMnemonic = "BRSET";

            cpAddressMode = "IND";

            byByteCount = 4;

            fBranchFlag = TRUE;

            break; 

        case 0x1F : 

            cpMnemonic = "BRCLR";

            cpAddressMode = "IND";

            byByteCount = 4;

            fBranchFlag = TRUE;

            break;

        case 0x20 : 

            cpMnemonic = "BRA";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x21 : 

            cpMnemonic = "BRN";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x22 : 

            cpMnemonic = "BHI";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x23 : 

            cpMnemonic = "BLS";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x24 : 

            cpMnemonic = "BCC/BHS";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x25 : 

            cpMnemonic = "BCS/BLO";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x26 : 

            cpMnemonic = "BNE";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x27 : 

            cpMnemonic = "BEQ";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x28 : 

            cpMnemonic = "BVC";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x29 : 

            cpMnemonic = "BVS";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x2A : 

            cpMnemonic = "BPL";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x2B : 

            cpMnemonic = "BMI";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x2C : 

            cpMnemonic = "BGE";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x2D : 

            cpMnemonic = "BLT";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x2E : 

            cpMnemonic = "BGT";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x2F : 

            cpMnemonic = "BLE";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x30 : 

            cpMnemonic = "TSX/TSY";

            cpAddressMode = "INH";

            byByteCount = 1;

            fBranchFlag = FALSE;

            break; 

        case 0x31 : 

            cpMnemonic = "INS";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x32 : 

            cpMnemonic = "PULA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x33 : 

            cpMnemonic = "PULB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x34 : 

            cpMnemonic = "DES";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x35 : 

            cpMnemonic = "TXS/TYS";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x36 : 

            cpMnemonic = "PSHA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x37 : 

            cpMnemonic = "PSHB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x38 : 

            cpMnemonic = "PULX/PULY";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x39 : 

            cpMnemonic = "RTS";

            cpAddressMode = "INH";

            byByteCount = 1;

            fBranchFlag = TRUE;

            break; 

        case 0x3A : 

            cpMnemonic = "ABX/ABY";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x3B : 

            cpMnemonic = "RTI";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x3C : 

            cpMnemonic = "PSHX/PSHY";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x3D : 

            cpMnemonic = "MUL";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x3E : 

            cpMnemonic = "WAI";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x3F : 

            cpMnemonic = "SWI";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x40 : 

            cpMnemonic = "NEGA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x43 : 

            cpMnemonic = "COMA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x44 : 

            cpMnemonic = "LSRA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x46 : 

            cpMnemonic = "RORA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x47 : 

            cpMnemonic = "ASRA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x48 : 

            cpMnemonic = "ASLA/LSLA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x49 : 

            cpMnemonic = "ROLA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x4A : 

            cpMnemonic = "DECA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x4C : 

            cpMnemonic = "INCA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x4D : 

            cpMnemonic = "TSTA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x4F : 

            cpMnemonic = "CLRA";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x50 : 

            cpMnemonic = "NEGB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break;

        case 0x53 : 

            cpMnemonic = "COMB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x54 : 

            cpMnemonic = "LSRB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x56 : 

            cpMnemonic = "RORB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x57 : 

            cpMnemonic = "ASRB/ASLB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x58 : 

            cpMnemonic = "LSLB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x59 : 

            cpMnemonic = "ROLB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x5A : 

            cpMnemonic = "DECB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x5C : 

            cpMnemonic = "INCB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x5D : 

            cpMnemonic = "TSTB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x5F : 

            cpMnemonic = "CLRB";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x60 : 

            cpMnemonic = "NEG";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x63 : 

            cpMnemonic = "COM";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x64 : 

            cpMnemonic = "LSR";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x66 : 

            cpMnemonic = "ROR";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x67 : 

            cpMnemonic = "ASR";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x68 : 

            cpMnemonic = "ASL/LSL";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x69 : 

            cpMnemonic = "ROL";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x6A : 

            cpMnemonic = "DEC";

            cpAddressMode = "IND";

            byByteCount = 2;

            fBranchFlag = FALSE;

            break; 

        case 0x6C : 

            cpMnemonic = "INC";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x6D : 

            cpMnemonic = "TST";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x6E : 

            cpMnemonic = "JMP";

            cpAddressMode = "IND";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x6F : 

            cpMnemonic = "CLR";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0x70 : 

            cpMnemonic = "NEG";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x73 : 

            cpMnemonic = "COM";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x74 : 

            cpMnemonic = "LSR";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x76 : 

            cpMnemonic = "ROR";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x77 : 

            cpMnemonic = "ASR";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x78 : 

            cpMnemonic = "ASL/LSL";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x79 : 

            cpMnemonic = "ROL";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x7A : 

            cpMnemonic = "DEC";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x7C : 

            cpMnemonic = "INC";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x7D : 

            cpMnemonic = "TST";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x7E : 

            cpMnemonic = "JMP";

            cpAddressMode = "EXT";

            byByteCount = 3;

            fBranchFlag = TRUE;

            break; 

        case 0x7F : 

            cpMnemonic = "CLR";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0x80 : 

            cpMnemonic = "SUBA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x81 : 

            cpMnemonic = "CMPA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x82 : 

            cpMnemonic = "SBCA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x83 : 

            cpMnemonic = "SUBD";

            cpAddressMode = "IMM";

            byByteCount = 3;

            break; 

        case 0x84 : 

            cpMnemonic = "ANDA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x85 : 

            cpMnemonic = "BITA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x86 : 

            cpMnemonic = "LDAA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x88 : 

            cpMnemonic = "EORA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x89 : 

            cpMnemonic = "ADCA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x8A : 

            cpMnemonic = "ORAA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x8B : 

            cpMnemonic = "ADDA";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0x8C : 

            cpMnemonic = "CPX/CPY";

            cpAddressMode = "IMM";

            byByteCount = 3;

            break; 

        case 0x8D : 

            cpMnemonic = "BSR";

            cpAddressMode = "REL";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x8E : 

            cpMnemonic = "LDS";

            cpAddressMode = "IMM";

            byByteCount = 3;

            break; 

        case 0x8F : 

            cpMnemonic = "XGDX/XGDY";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0x90 : 

            cpMnemonic = "SUBA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x91 : 

            cpMnemonic = "CMPA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x92 : 

            cpMnemonic = "SBCA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x93 : 

            cpMnemonic = "SUBD";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x94 : 

            cpMnemonic = "ANDA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x95 : 

            cpMnemonic = "BITA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x96 : 

            cpMnemonic = "LDAA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x97 : 

            cpMnemonic = "STAA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x98 : 

            cpMnemonic = "EORA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x99 : 

            cpMnemonic = "ADCA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x9A : 

            cpMnemonic = "ORAA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x9B : 

            cpMnemonic = "ADDA";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x9C : 

            cpMnemonic = "CPX/CPY";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x9D : 

            cpMnemonic = "JSR";

            cpAddressMode = "DIR";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0x9E : 

            cpMnemonic = "LDS";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0x9F : 

            cpMnemonic = "STS";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xA0 : 

            cpMnemonic = "SUBA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xA1 : 

            cpMnemonic = "CMPA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xA2 : 

            cpMnemonic = "SBCA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xA3 : 

            cpMnemonic = "SUBD";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xA4 : 

            cpMnemonic = "ANDA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xA5 : 

            cpMnemonic = "BITA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xA6 : 

            cpMnemonic = "LDAA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xA7 : 

            cpMnemonic = "STAA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xA8 : 

            cpMnemonic = "EORA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xA9 : 

            cpMnemonic = "ADCA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xAA : 

            cpMnemonic = "ORAA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xAB : 

            cpMnemonic = "ADDA";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xAC : 

            cpMnemonic = "CPX/CPY";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xAD : 

            cpMnemonic = "JSR";

            cpAddressMode = "IND";

            byByteCount = 2;

            fBranchFlag = TRUE;

            break; 

        case 0xAE : 

            cpMnemonic = "LDS";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xAF : 

            cpMnemonic = "STS";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xB0 : 

            cpMnemonic = "SUBA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xB1 : 

            cpMnemonic = "CMPA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xB2 : 

            cpMnemonic = "SBCA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xB3 : 

            cpMnemonic = "SUBD";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xB4 : 

            cpMnemonic = "ANDA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xB5 : 

            cpMnemonic = "BITA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xB6 : 

            cpMnemonic = "LDAA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xB7 : 

            cpMnemonic = "STAA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xB8 : 

            cpMnemonic = "EORA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xB9 : 

            cpMnemonic = "ADCA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xBA : 

            cpMnemonic = "ORAA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xBB : 

            cpMnemonic = "ADDA";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xBC : 

            cpMnemonic = "CPX/CPY";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xBD : 

            cpMnemonic = "JSR";

            cpAddressMode = "EXT";

            byByteCount = 3;

            fBranchFlag = TRUE;

            break; 

        case 0xBE : 

            cpMnemonic = "LDS";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xBF : 

            cpMnemonic = "STS";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xC0 : 

            cpMnemonic = "SUBB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xC1 : 

            cpMnemonic = "CMPB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xC2 : 

            cpMnemonic = "SBCB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xC3 : 

            cpMnemonic = "ADDD";

            cpAddressMode = "IMM";

            byByteCount = 3;

            break; 

        case 0xC4 : 

            cpMnemonic = "ANDB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xC5 : 

            cpMnemonic = "BITB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xC6 : 

            cpMnemonic = "LDAB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xC8 : 

            cpMnemonic = "EORB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xC9 : 

            cpMnemonic = "ADCB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xCA : 

            cpMnemonic = "ORAB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xCB : 

            cpMnemonic = "ADDB";

            cpAddressMode = "IMM";

            byByteCount = 2;

            break; 

        case 0xCC : 

            cpMnemonic = "LDD";

            cpAddressMode = "IMM";

            byByteCount = 3;

            break; 

        case 0xCD : 

            cpMnemonic = "PAGE4";

            cpAddressMode = "CRTL";

            byByteCount = 1;

            break; 

        case 0xCE : 

            cpMnemonic = "LDX/LDY";

            cpAddressMode = "IMM";

            byByteCount = 3;

            break; 

        case 0xCF : 

            cpMnemonic = "STOP";

            cpAddressMode = "INH";

            byByteCount = 1;

            break; 

        case 0xD0 : 

            cpMnemonic = "SUBB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xD1 : 

            cpMnemonic = "CMPB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xD2 : 

            cpMnemonic = "SBCB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xD3 : 

            cpMnemonic = "ADDD";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xD4 : 

            cpMnemonic = "ANDB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xD5 : 

            cpMnemonic = "BITB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xD6 : 

            cpMnemonic = "LDAB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xD7 : 

            cpMnemonic = "STAB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xD8 : 

            cpMnemonic = "EORB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xD9 : 

            cpMnemonic = "ADCB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xDA : 

            cpMnemonic = "ORAB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xDB : 

            cpMnemonic = "ADDB";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xDC : 

            cpMnemonic = "LDD";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xDD : 

            cpMnemonic = "STD";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xDE : 

            cpMnemonic = "LDX/LDY";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xDF : 

            cpMnemonic = "STX/STY";

            cpAddressMode = "DIR";

            byByteCount = 2;

            break; 

        case 0xE0 : 

            cpMnemonic = "SUBB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xE1 : 

            cpMnemonic = "CMPB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xE2 : 

            cpMnemonic = "SBCB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xE3 : 

            cpMnemonic = "ADDD";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xE4 : 

            cpMnemonic = "ANDB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xE5 : 

            cpMnemonic = "BITB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xE6 : 

            cpMnemonic = "LDAB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xE7 : 

            cpMnemonic = "STAB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xE8 : 

            cpMnemonic = "EORB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xE9 : 

            cpMnemonic = "ADCB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xEA : 

            cpMnemonic = "ORAB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xEB : 

            cpMnemonic = "ADDB";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xEC : 

            cpMnemonic = "LDD";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xED : 

            cpMnemonic = "STD";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xEE : 

            cpMnemonic = "LDX/LDY";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xEF : 

            cpMnemonic = "STX/STY";

            cpAddressMode = "IND";

            byByteCount = 2;

            break; 

        case 0xF0 : 

            cpMnemonic = "SUBB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xF1 : 

            cpMnemonic = "CMPB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xF2 : 

            cpMnemonic = "SBCB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xF3 : 

            cpMnemonic = "ADDD";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xF4 : 

            cpMnemonic = "ANDB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xF5 : 

            cpMnemonic = "BITB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xF6 : 

            cpMnemonic = "LDAB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xF7 : 

            cpMnemonic = "STAB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xF8 : 

            cpMnemonic = "EORB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xF9 : 

            cpMnemonic = "ADCB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xFA : 

            cpMnemonic = "ORAB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xFB : 

            cpMnemonic = "ADDB";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xFC : 

            cpMnemonic = "LDD";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xFD : 

            cpMnemonic = "STD";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xFE : 

            cpMnemonic = "LDX/LDY";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        case 0xFF : 

            cpMnemonic = "STX/STY";

            cpAddressMode = "EXT";

            byByteCount = 3;

            break; 

        default :

            cpMnemonic = "INVALID"; 

            cpAddressMode = "INVALID";

            byByteCount = 0;

            return FALSE;

    }



    return TRUE;

}

