// Header file for CProgram (68HC11 User Program Class)
//
// Authors: Jason Gaedtke, Matt Vandenbush
// Code Review: Jason Gaedtke
// Date: 12/7/96

// include files
#include "CTalker.h"
#include "Errcodes.h"
#include "stdafx.h"
#include <afx.h>

// talker constants
#define RAMTALKER	0
#define ROMTALKER	1

// 68HC11 operating mode constants
#define BOOTSTRAP	0
#define SPECIALTEST	1

// S19 size
#define MAXS19SIZE  16384				// (16K)

// breakpoint constants
#define MAXBREAKPOINTS	10
#define EXECUTEDELAY	20				// milliseconds
#define PAGE1	1
#define PAGE2	2
#define PAGE3	3
#define PAGE4	4

// definition of breakpoint structure
typedef struct {
	UINT	uiAddress;
	BYTE	byOpCode;
} breakpointStruct;

class CProgram
{
public:
	UINT				m_uiProgramExecution;	// address for executing program
private:
	breakpointStruct	m_breakpointTable[MAXBREAKPOINTS];	// table for breakpoint info
	breakpointStruct	m_tracepoint1;			// breakpoint structure for first tracepoint
	breakpointStruct	m_tracepoint2;			// breakpoint structure for second tracepoint
	CString				m_strProgramFile;		// user program file (S19)
	CTalker*			m_talker;				// talker object
	UINT				m_uiTalkerIdle;			// address of talker idle loop

////////////////////////////////////////////////////////////
// PlaceBreakpoints: replaces user opcodes with SWI opcode
//		for each breakpoint table entry
//
// Arguments:
//		none
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		program.PlaceBreakpoints();

	UINT PlaceBreakpoints();

////////////////////////////////////////////////////////////
// SetTracepoint: places a new tracepoint at specified address
//      and initializes a tracepoint structure
//
// Arguments:
//      uiAddress : memory address for new tracepoint
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.SetTracepoint(uiAddress);
//
// Note: THIS IS A PRIVATE UTILITY FUNCTION CALLED BY TraceInstruction

	UINT SetTracepoint(const UINT uiAddress);

////////////////////////////////////////////////////////////
// RemoveTracepoints: replaces original user opcodes and 
//      clears tracepoint structures
//
// Arguments:
//      none
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.RemoveTracepoints();
//
// Note: THIS IS A PRIVATE UTILITY FUNCTION CALLED BY TraceInstruction

	UINT RemoveTracepoints();

////////////////////////////////////////////////////////////
// DisassembleOpCode:  processes specified 68HC11 opcode
//      and returns a mnemonic, addressing mode, byte count,
//      prefix flag and branch flag; (if the specified opcode
//      is invalid the routine returns FALSE)
//
// Arguments:
//      byOpCode : single byte opcode to disassemble
//      cpMnemonic : opcode mnemonic
//      cpAddressMode : opcode addressing mode
//      byByteCount : number of bytes in instruction
//      fBranchFlag : set if opcode is special branch instruction
//
// Return Value:
//      TRUE if opcode is valid, otherwise FALSE
//
// Example Usage:
//      program.DisassembleOpCode(byOpcode, 
//                                cpMnemonic,
//                                cpAddressMode,
//                                byByteCount,
//                                fBranchFlag);

	BOOL DisassembleOpCode(const BYTE byOpCode,
		                   CString& cpMnemonic,
						   CString& cpAddressMode,
						   BYTE& byByteCount,
						   BOOL& fBranchFlag);

public:
	BOOL				m_fRunning;				// Boolean flag set when code is running

	CProgram(const CTalker* talkerObject);
	~CProgram();

////////////////////////////////////////////////////////////
// Init: initializes specified talker
//
// Arguments:
//      cpComPort : communications port
//      byTalkerType : talker type (RAMTALKER, ROMTALKER)
//      cpTalkerFile : talker file
//      byOperatingMode : 68HC11 mode of operation (BOOTSTRAP, SPECIALTEST) 
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.Init("COM2", 
//                   RAMTALKER, 
//                   "c:\talke.s19"
//                   SPECIALTEST);
//
// Note: COMMUNICATIONS STRING MUST NOT CONTAIN SPACES

	UINT Init(const char* cpComPort, 
		      const BYTE byTalkerType,
			  const char* cpTalkerFile,
			  const BYTE byOperatingMode,
			  const BOOL bEnableEEPROM);

////////////////////////////////////////////////////////////
// GetProgramName: updates the sole argument to contain the
//		name of the current user program file
//
// Arguments:
//		strProgramName : CString object which will contain
//			name of user program file
//
// Return Value:
//		none
//
// Example Usage:
//		program.GetProgramName(strProgramName);

	void GetProgramName(CString& strProgramName)
	{
		strProgramName = m_strProgramFile;
	};

////////////////////////////////////////////////////////////
// LoadSRecordFile: opens specified S19 file and writes
//		user program to 68HC11 memory
//
// Arguments:
//		cpFileName : user file
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		program.LoadSRecordFile("c:\user.s19");
//
// Note: SPECIFIED FILE MUST BE A COMPILED S19 FILE

	UINT LoadSRecordFile(const char* cpFileName);

////////////////////////////////////////////////////////////
// VerifySRecordFile: opens specified S19 file and verifies
//      program in 68HC11 memory
//
// Arguments:
//      cpFileName : user S19 file
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.VerifySRecordFile("c:\user.s19");
//
// Note: SPECIFIED FILE MUST BE A COMPILED S19 FILE

	UINT VerifySRecordFile(const char* cpFileName);

////////////////////////////////////////////////////////////
// ExecuteProgram:  executes from specified location
//
// Arguments:
//      uiStartAddress : starting address of user program
//		fBreakpoints : flag enabling and disabling user breakpoints
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.ExecuteProgram(uiStartAddress);

	UINT ExecuteProgram(const UINT uiStartAddress,
						const BOOL fBreakpoints);

////////////////////////////////////////////////////////////
// ExecuteSubroutine:  executes subroutine and returns 
//      control to the talker idle loop
//
// Arguments:
//      uiSubAddress : starting address of user subroutine
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.ExecuteSubroutine(uiSubAddress);

	UINT ExecuteSubroutine(const UINT uiStartAddress);

////////////////////////////////////////////////////////////
// StopExecution:  stops execution of current program or
//      subroutine and stores address for future reference;
//      control then returns to the talker idle loop
//
// Arguments:
//      none
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.StopExecution();

	UINT StopExecution();

////////////////////////////////////////////////////////////
// SetBreakpoint: places a new breakpoint entry for the specified
//      address in the breakpoint table
//
// Arguments:
//      uiAddress : memory address for new breakpoint
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.SetBreakpoint(uiAddress);
//
// Note: REDUNDANT BREAKPOINTS ARE NOT ALLOWED

	UINT SetBreakpoint(const UINT uiAddress);

////////////////////////////////////////////////////////////
// RemoveBreakpoint: clears the entry corresponding to the specified
//      address in the breakpoint table
//
// Arguments:
//      uiAddress : memory address of breakpoint to remove
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.RemoveBreakpoint(uiAddress);

	UINT RemoveBreakpoint(const UINT uiAddress);

////////////////////////////////////////////////////////////
// ListBreakpoints: returns a list of memory addresses   
//      corresponding with current breakpoints
//
// Arguments:
//      uiBreakpoints[MAXBREAKPOINTS] : array of unsigned integers 
//          containing breakpointed memory addresses
//
// Return Value:
//      number of breakpoint entries in table
//
// Example Usage:
//      program.ListBreakpoints(uiBreakpoints[MAXBREAKPOINTS]);

	BYTE ListBreakpoints(UINT* uiBreakpoints);

////////////////////////////////////////////////////////////
// ClearBreakpoints: replaces SWI opcodes with user opcode
//      for each breakpoint table entry
//
// Arguments:
//      none
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.ClearBreakpoints();
//
// Note: THIS ROUTINE WORKS IN CONJUCTION WITH PlaceBreakpoints TO
//      MANAGE SWI OPCODES IN 68HC11 MEMORY

	UINT ClearBreakpoints();

////////////////////////////////////////////////////////////
// TraceInstruction: places tracepoint(s) after instruction
//      at specified address and then executes instruction;
//      tracepoint(s) is/are removed prior to returning control
//      to user
//
// Arguments:
//      uiAddress : address of instruction to trace
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      program.TraceInstruction(uiAddress);

	UINT TraceInstruction(const UINT uiAddress);

};
