// CSerial Serial Communications Class
//
// Designed for Win32 based software development
//
// (c) Copyright 1996 Phil VanHelden
// Permission is granted to redistribute this
// class as long as this message and copyright
// information is kept with the file and is
// left intact.
//
// Version 1.0 : 12/9/96
// Version 1.1 : 12/14/96
//
////////////////////////////////////////////////////////////////
// Revisions
//
// 12/9/96 Matt Vandenbush - Changed error return values for the
//                           member functions to a constant declared
//                           in the header, COMERROR, which is set
//                           to 1.
// 12/14/96 Phil VanHelden - Renamed private member variables to
//				have the prefix m_, enclosed main() in
//				#ifdef / #endif blocks
//
////////////////////////////////////////////////////////////////

#include "CSerial.h"

CSerial::CSerial()
{
    m_hComPort = NULL;        // comm port handle
    m_fActive = FALSE;        // comm port flag
	m_byByteSize = 0;
	m_byParity = 0;
	m_byStopBits = 0;
	m_szComPort[0] = '\0';
	m_uiBaud = 0;
	m_uiDelay = 20;
}

CSerial::~CSerial()
{
    if(m_fActive)             // close port if open
        Close();
}

////////////////////////////////////////////////////////////
// Open:  opens the communications port
//
// Arguments:
//      cpComString : communications paramaters
//          format: "COMport:baud,parity,length,stop"
//
// Return Value:
//      0 if successful, otherwise an error structure
//
// Example Usage:
//      serial.Open("COM2:9600,n,8,1");
//
// Note: STRING MUST NOT CONTAIN SPACES

DWORD CSerial::Open(const char* pszComString)
{
    DCB dcb;                // data control block
    DWORD dwError;          // error code
    BOOL fSuccess;          // status flag
    COMMTIMEOUTS commtimeouts;
    char szCom[COMMPORTSIZE];          // comm port name (COM1)
    char* cpComParse;       // copy of commands to parse
    char* cpComTemp;        // pointer into parse string
	BYTE* bypClearCommBuff = NULL;
	UINT  uiReadLength = 0;

    // parse command string: comm port
    strncpy(szCom, pszComString, 4);
    szCom[4] = '\0';
    wsprintf(m_szComPort, "%s:", szCom);

    // parse command string: baud rate
    cpComParse = strchr(pszComString, ':');
	if(!cpComParse)
		return PARSEERROR;
    cpComParse++;
    cpComTemp = strchr(cpComParse, ',');
	if(!cpComTemp)
		return PARSEERROR;
    *cpComTemp = '\0';
    m_uiBaud = atoi(cpComParse);

    // parse command string: parity type
    cpComParse = cpComTemp+1;
    cpComTemp = strchr(cpComParse, ',');
	if(!cpComTemp)
		return PARSEERROR;
    *cpComTemp = '\0';
    switch(*cpComParse)
	{
	case 'n':
	case 'N':
        m_byParity = NOPARITY;
		break;
	case 'e':
	case 'E':
        m_byParity = EVENPARITY;
		break;
    case 'o':
    case 'O':
		m_byParity = ODDPARITY;
		break;
    default:
        return COMERROR;            // invalid parity type
    }

    // parse command string: word size
    cpComParse = cpComTemp+1;
    cpComTemp = strchr(cpComParse, ',');
	if(!cpComTemp)
		return PARSEERROR;
    *cpComTemp = '\0';
    m_byByteSize = atoi(cpComParse);

    // parse command string: stop bits
    cpComParse = cpComTemp+1;
    if(!strcmp(cpComParse, "1"))
        m_byStopBits = ONESTOPBIT;
    else if(!strcmp(cpComParse, "1.5"))
        m_byStopBits = ONE5STOPBITS;
    else if(!strcmp(cpComParse, "2"))
        m_byStopBits = TWOSTOPBITS;
    else
        return COMERROR;            // invalid stop bit code

    // open port
    m_hComPort = CreateFile(m_szComPort, GENERIC_READ | GENERIC_WRITE,
                          0, NULL, OPEN_EXISTING, 0, NULL);
    if(m_hComPort == INVALID_HANDLE_VALUE)
    {
        dwError = GetLastError();
        return dwError;
    }

    // get current comm state
    fSuccess = GetCommState(m_hComPort, &dcb);
    if(!fSuccess)
    {
        dwError = GetLastError();
        CloseHandle(m_hComPort);
        return dwError;
    }

    // initialize data control parameters
    dcb.BaudRate = m_uiBaud;
    dcb.ByteSize = m_byByteSize;
    dcb.Parity = m_byParity;
    dcb.StopBits = m_byStopBits;
    fSuccess = SetCommState(m_hComPort, &dcb);
    if(!fSuccess)
    {
        dwError = GetLastError();
        CloseHandle(m_hComPort);
        return dwError;
    }

    // set timeout parameters
    fSuccess = GetCommTimeouts(m_hComPort, &commtimeouts);
    if(!fSuccess)
    {
        dwError = GetLastError();
        CloseHandle(m_hComPort);
        return dwError;
    }
    commtimeouts.ReadIntervalTimeout=100;
    fSuccess = SetCommTimeouts(m_hComPort, &commtimeouts);
    if(!fSuccess)
    {
        dwError = GetLastError();
        CloseHandle(m_hComPort);
        return dwError;
    }

    m_fActive = TRUE;
    
	// Clear out the input buffer before starting
	PurgeComm(m_hComPort, PURGE_TXABORT | PURGE_RXABORT |
						  PURGE_TXCLEAR | PURGE_RXCLEAR);

	return 0L;
}

////////////////////////////////////////////////////////////
// IsPortOpen:  Boolean test: Is communications port open?
//
// Arguments:
//      none
//
// Return value:
//      TRUE if open, else FALSE
//
// Example usage:
//      serial.IsPortOpen();

BOOL CSerial::IsPortOpen()
{
    return m_fActive;
}

////////////////////////////////////////////////////////////
// IsDataWaiting:  Boolean test: Is data waiting to be read?
//
// Arguments:
//      none
//
// Return value:
//      TRUE if data is available, else FALSE
//
// Example usage:
//      serial.IsDataWaiting();

BOOL CSerial::IsDataWaiting()
{
    DWORD dwErrorMask;
    COMSTAT comstat;

    if(!m_fActive)            // return False if port is not open
        return FALSE;   

    // read number of bytes available
    ClearCommError(m_hComPort, &dwErrorMask, &comstat);
    if(dwErrorMask)
        return FALSE; // return False on comm error
    if(comstat.cbInQue)
        return TRUE;        // return True if data available
    else
        return FALSE;       // else return False
}

////////////////////////////////////////////////////////////
// GetReadLength:  returns number of bytes available on the 
//      communications port
//
// Arguments:
//      none
//
// Return value:
//      number of bytes available in input queue
//
// Example usage:
//      serial.GetReadLength();

UINT CSerial::GetReadLength()
{
    DWORD dwErrorMask;
    COMSTAT comstat;

    if(!m_fActive)            // return error if port is not open
        return COMERROR;

    // read number of bytes available
    ClearCommError(m_hComPort, &dwErrorMask, &comstat);
    if(dwErrorMask)
        return FALSE;			// Return FALSE;
    return (UINT) comstat.cbInQue;  // return byte count
}

////////////////////////////////////////////////////////////
// Read:  reads specified number of bytes from the communications
//      port
//
// Arguments:
//      bpData : pointer to BYTE array to receive data
//      uiMax : maximum number of bytes to read
//
// Return value:
//      number of bytes read
//
// Example usage:
//      serial.Read(&buffer, maxBytes);

UINT CSerial::Read(const BYTE* pbyData, const UINT uiCount)
{
    DWORD dwErrorMask;
    COMSTAT comstat;
    DWORD dwNumberToRead;
    UINT uiMaxBytes = uiCount;
    UINT uiByteCount = 0;

    if(!m_fActive)     
        return 0;           // return 0 if port is not open

    // read number of bytes available
    ClearCommError(m_hComPort, &dwErrorMask, &comstat);
    if(dwErrorMask)
        return 0;           // return 0 on error

    // set read limit
    if(uiMaxBytes > MAXRXSIZE)
        uiMaxBytes = MAXRXSIZE;
    
    // set actual read count
    if(comstat.cbInQue > uiMaxBytes)
        dwNumberToRead = uiMaxBytes;
    else
        dwNumberToRead = comstat.cbInQue;

    if(dwNumberToRead == 0)
    {
		Sleep(m_uiDelay);  // If no reply, wait
		
		ClearCommError(m_hComPort, &dwErrorMask, &comstat);
		if(dwErrorMask)
			return 0; // return False on comm error
		
		// set actual read count
		if(comstat.cbInQue > uiMaxBytes)
			dwNumberToRead = uiMaxBytes;
		else
			dwNumberToRead = comstat.cbInQue;

		if(dwNumberToRead == 0)
			return 0;
	}

    if(!ReadFile(m_hComPort, (BYTE*) pbyData, dwNumberToRead,
                (ULONG*) &uiByteCount, NULL))
    {
        return 0;           // return 0 on error
    }

    return uiByteCount;
}

////////////////////////////////////////////////////////////
// Write:  writes specified number of bytes to the communications
//      port
//
// Arguments:
//    bpData : pointer to BYTE array of data
//    uiCount : number of bytes to write
//
// Return value:
//    0 if successful, otherwise an error structure
//
// Example usage:
//    serial.Write(&bpData, uiCount);

DWORD CSerial::Write(const BYTE* pbyData, const UINT uiCount)
{
    DWORD dwErrorMask;
    COMSTAT comstat;
    DWORD dwActualWrite;
    DWORD dwLength;
    UINT uiCounter = uiCount;
    BYTE* pbyDataToWrite = (BYTE*) pbyData;

    if(!m_fActive)            // return error if port is not open
        return COMERROR;

    while (uiCounter)
    {
        ClearCommError(m_hComPort, &dwErrorMask, &comstat);
        if(dwErrorMask)
            return COMERROR;     // comm error 
        if(MAXTXSIZE < uiCount)
            dwLength = MAXTXSIZE;
        else
            dwLength = uiCount;
        
        WriteFile(m_hComPort, pbyDataToWrite,
                  dwLength, &dwActualWrite, NULL);
        uiCounter -= dwActualWrite;
        pbyDataToWrite += dwActualWrite;
    }
    return 0L;
}

////////////////////////////////////////////////////////////
// Close:  closes the communications port
//
// Arguments:
//      none
//
// Return value:
//      none
//
// Example usage:
//      serial.Close();

void CSerial::Close()
{
    if(m_fActive)                     // close port if open
        CloseHandle(m_hComPort);
    m_fActive = FALSE;    
    m_hComPort = NULL;
}

#ifdef __TEST

// Test Program
int main()
{
    CSerial serial;
    BYTE bypBuffer[256];
    UINT uiLength = 512;

    // open port
    cout << "Initializing..." << endl;
    if (serial.Open("com2:9600,n,8,1") != 0)
        return COMERROR;
    cout << "Port opened successfully..." << endl;

    // write data
    bypBuffer[0] = 'A';
    bypBuffer[1] = 'T';
    bypBuffer[2] = 'Z';
    bypBuffer[3] = 13;
    serial.Write(bypBuffer, 4);
    cout << "Wrote:  ATZ\\n" << endl;

    // read response
    bypBuffer[0] = '\0';
    bypBuffer[1] = '\0';
    bypBuffer[2] = '\0';
    bypBuffer[3] = '\0';
    Sleep(2000);            // brief delay
    cout << "Bytes available: " << serial.GetReadLength() << endl;
    while(serial.IsDataWaiting())   // (will crash if byte count > 256)
    {
        uiLength = serial.Read(bypBuffer, serial.GetReadLength());
        bypBuffer[uiLength] = '\0';
        cout << "Received: " << bypBuffer << endl;
        cout << "Length: " << uiLength << endl;
    }
    serial.Close();
    cout << "Port closed..." << endl;

    return 0;
}

#endif
