// Definition of CTalker (68HC11 Talker Abstraction Class)
//
// Authors: Jason Gaedtke, Matt Vandenbush
// Code Review: Jason Gaedtke
// Date: 12/7/96

// include files
#include "CTalker.h"

// constructor definition
CTalker::CTalker(const CSerial* comObject)
{
    // initialize pointer to communictions object
    m_serialPort = (CSerial*) comObject;
}

////////////////////////////////////////////////////////////
// OpenPort:  calls CSerial method to open communications port
//
// Arguments:
//		cpComString : communications paramaters
//			format: "COMport:baud,parity,length,stop"
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		talker.OpenPort("COM2:9600,n,8,1");
//
// Note: STRING MUST NOT CONTAIN SPACES

UINT CTalker::OpenPort(const char* cpComString)
{
	// open comm port
	if(m_serialPort->Open(cpComString) != 0)
		return OPEN_PORT_FAIL;      // error: failed to open comm port
	else
		return SUCCESS;
}

////////////////////////////////////////////////////////////
// LoadRAMTalker: processes talker file and passes byte array to
//      CSerial.Write() to initialize talker
//
// Arguments:
//      cpComPort : communications port
//      cpTalkerFile : talker file
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.LoadRAMTalker("COM1",
//                           "c:\talke.boo");
//
// Note: SPECIFIED TALKER MUST BE A COMPILED BINARY (*.boo) FILE

UINT CTalker::LoadRAMTalker(const char* cpComPort,
                            const char* cpTalkerFile)
{
    UINT uiResult = SUCCESS;        // return status
    CFile talkerFile;               // talker file object
    UINT uiFileLength;              // talker file size
    BYTE bypBuffer[257];            // byte array for talker file
    UINT uiConfig = 0x103f;         // address of config register
    BYTE byConfigValue = 0;         // value of config register
    CString strComString;           // CString object for comm params

    // close port if open, close it (allows user to reboot talker)
    if(m_serialPort->IsPortOpen())
        ClosePort();
    
    // open port (at 1200 baud)
    strComString = cpComPort;
    strComString += ":1200,n,8,1";
    if((uiResult = OpenPort(strComString)) != SUCCESS)
        return uiResult;        // error: failed to open comm port

    // open talker file
    if(!(talkerFile.Open(cpTalkerFile, CFile::modeRead | CFile::shareCompat)))
        return OPEN_TALKER_FAIL;    // error: failed to open talker file
    m_strTalkerFile = cpTalkerFile;

    // read length of talker file
    uiFileLength = talkerFile.GetLength();
    if(uiFileLength != 257)
        return INVALID_TALKER;      // error: invalid talker file
    
    // read talker file into byte array
    if(talkerFile.Read(bypBuffer, uiFileLength) != uiFileLength)
        return READ_TALKER_FAIL;    // error: failed to read talker file
    talkerFile.Abort();             // close file object

    // write data array to serial port
    if(m_serialPort->Write(bypBuffer, 257) != 0)
        return COMM_ERROR;      // error: comm error

    // close port
    ClosePort();

    // reopen port (at 9600 baud)
    strComString = cpComPort;
    strComString += ":9600,n,8,1";
    if((uiResult = OpenPort(strComString)) != SUCCESS)
        return uiResult;        // error: failed to open comm port

    // read config register (to determine EEPROM status)
    if((uiResult = ReadMemory(uiConfig, 1, &byConfigValue)) != SUCCESS)
        return uiResult;        // error: memory read error
    
    // test EEPROM status and set flag
    if(byConfigValue & 0x01)
        m_fEEPROM = TRUE;
    else
        m_fEEPROM = FALSE;

    // talker is now downloaded and initialized
    return uiResult;
}

////////////////////////////////////////////////////////////
// LoadROMTalker: initializes ROM talker
//
// Arguments:
//      cpComPort : communications port 
//      cpMapFile : map file for ROM talker
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.LoadROMTalker("COM1",
//                           "c:\talke.map");
//
// Note: EVB2 BOARD SWITCH MUST BE IN "BUFFALO" POSITION;
//      THIS FUNCTION HAS NOT BEEN TESTED!

UINT CTalker::LoadROMTalker(const char* cpComPort,
                            const char* cpMapFile)
{
    UINT uiResult = SUCCESS;        // return status
    CFile mapFile;                  // map file object
    BYTE byByte = 0;                // command byte to load ROM talker
    UINT uiConfig = 0x1037;         // address of config register
    BYTE byConfigValue = 0;         // value of config register
    CString strComString;           // CString object for comm params

    // open port (at 1200 baud)
    strComString = cpComPort;
    strComString += ":1200,n,8,1";
    if((uiResult = OpenPort(strComString)) != SUCCESS)
        return uiResult;        // error: failed to open comm port

    // verify that map file exists
    if(!mapFile.Open(cpMapFile, CFile::modeRead | CFile::shareCompat))
        return READ_MAP_FAIL;   // error: failed to read map file
    m_strTalkerFile = cpMapFile;
    mapFile.Abort();            // close file

    // write single byte (0) to serial port; signals ROMTALKER to bootloader 
    if(m_serialPort->Write(&byByte, 1) != 0)
        return COMM_ERROR;      // error: comm error

    // close port
    ClosePort();

    // reopen port (at 9600 baud)
    strComString = cpComPort;
    strComString += ":9600,n,8,1";
    if((uiResult = OpenPort(strComString)) != SUCCESS)
        return uiResult;        // error: failed to open comm port
    
    // read config register (to determine EEPROM status)
    if((uiResult = ReadMemory(uiConfig, 1, &byConfigValue)) != SUCCESS)
        return uiResult;        // error: memory read failed
    
    // test EEPROM status
    if(byConfigValue & 0x01)
        m_fEEPROM = TRUE;
    else
        m_fEEPROM = FALSE;

    // ROM talker is now initialized
    return uiResult;
}

////////////////////////////////////////////////////////////
// ReadRegisters: reads control registers on 68HC11 
//      and returns a copy to caller (since the
//      values are passed by reference)
//
// Arguments:
//      uiStackPointer : stack pointer
//      byCCR : condition control register
//      byACCB : accumulator B
//      byACCA : accumulator A
//      uiIX : IX register
//      uiIY : IY register
//      uiProgramCounter : program counter
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.ReadRegisters(uiStackPointer,
//                           byCCR,
//                           byACCB,
//                           byACCA,
//                           uiIX,
//                           uiIY,
//                           uiProgramCounter);

UINT CTalker::ReadRegisters(UINT &uiStackPointer,
                            BYTE &byCCR,
                            BYTE &byACCB,
                            BYTE &byACCA,
                            UINT &uiIX,
                            UINT &uiIY,
                            UINT &uiProgramCounter)
{
    BYTE byData;

    // test comm port
    if(!(m_serialPort->IsPortOpen()))
        return PORT_NOT_OPEN;       // error: comm port not open

    // send register read command
    byData = 0x81;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;          // error: comm error
    
    // read and test echo confirmation
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    if(byData != 0x7e)
        return TALKER_ECHO_ERROR;   // talker error
    
    // receive stack pointer (two bytes)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiStackPointer = byData;
    uiStackPointer = uiStackPointer << 8;
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiStackPointer += byData;

    // receive condition control register
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    byCCR = byData;

    // receive accumulator B
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    byACCB = byData;

    // receive accumulator A
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    byACCA = byData;

    // receive IX register (two bytes)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiIX = byData;
    uiIX = uiIX << 8;
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiIX += byData;

    // receive IY register (two bytes)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiIY = byData;
    uiIY = uiIY << 8;
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiIY += byData;

    // receive program counter (two bytes)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiProgramCounter = byData;
    uiProgramCounter = uiProgramCounter << 8;
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiProgramCounter += byData;

    return SUCCESS;
}

////////////////////////////////////////////////////////////
// WriteRegisters: updates control registers on 68HC11 
//      to reflect argument values
//
// Arguments:
//      uiStackPointer : stack pointer
//      byCCR : condition control register
//      byACCB : accumulator B
//      byACCA : accumulator A
//      uiIX : IX register
//      uiIY : IY register
//      uiProgramCounter : program counter
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.WriteRegisters(uiStackPointer,
//                            byCCR,
//                            byACCB,
//                            byACCA,
//                            uiIX,
//                            uiIY,
//                            uiProgramCounter);

UINT CTalker::WriteRegisters(const UINT uiStackPointer,
                             const BYTE byCCR,
                             const BYTE byACCB,
                             const BYTE byACCA,
                             const UINT uiIX,
                             const UINT uiIY,
                             const UINT uiProgramCounter)
{
    BYTE byData;

    // test comm port
    if(!(m_serialPort->IsPortOpen()))
        return PORT_NOT_OPEN;   // error: comm port not open
    
    // send register write command
    byData = 0xc1;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error

    // read and test echo confirmation
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    if(byData != 0x3e)
        return REGISTER_WRITE_FAIL;

    // send stack pointer (high byte)
    byData = (uiStackPointer & 0xff00) >> 8;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error

    // send stack pointer (low byte)
    byData = uiStackPointer & 0xff;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error

    // send condition control register
    byData = byCCR;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
    
    // send accumulator B
    byData = byACCB;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
            
    // send accumulator A
    byData = byACCA;
    if(m_serialPort->Write(&byData, 1) != 0)
        return FALSE;       // comm error
    
    // send IX register (high byte)
    byData = (uiIX & 0xff00) >> 8;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
    
    // send IX register (low byte)
    byData = uiIX & 0xff;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
    
    // send IY register (high byte)
    byData = (uiIY & 0xff00) >> 8;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
    
    // send IY register (low byte)
    byData = uiIY & 0xff;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
    
    // send program counter (high byte)
    byData = (uiProgramCounter & 0xff00) >> 8;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
    
    // send program counter (low byte)
    byData = uiProgramCounter & 0xff;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
    
    return SUCCESS;
}

////////////////////////////////////////////////////////////
// ReadMemory: reads a specified number of data bytes
//      from a specified start address on the 68HC11;  
//      data is returned since the byte array is
//      passed by reference
//
// Arguments:
//      uiStartAddress : Address to start reading at
//      byByteCount : Number of bytes to read from uiStartAddress 
//      bypReadBuffer : Byte array holding data after completion
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.ReadMemory(uiStartAddress,
//                        byByteCount,
//                        bypReadBuffer);

UINT CTalker::ReadMemory(const UINT uiStartAddress,
                         const BYTE byByteCount,
                         BYTE* bypReadBuffer)
{   
    BYTE byData = 0;
    UINT uiBytesToRead = 0;
    UINT uiReturn = SUCCESS;
    UINT uiByteReturn = 0;
	
    // test comm port
    if(!(m_serialPort->IsPortOpen()))
        return PORT_NOT_OPEN;       // comm error

    // send memory read command
    byData = 0x01;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error

    // read and test echo confirmation
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    if(byData != 0xfe)
        return TALKER_ECHO_ERROR;       // talker error

    // send byte count
    byData = byByteCount;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error

    // send start address (high byte) 
    byData = (uiStartAddress & 0xff00) >> 8;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
    
    // send start address (low byte)
    byData = uiStartAddress & 0xff;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error
    
    // note: zero in byByteCount reads 256 bytes
    if(byByteCount == 0) 
        uiBytesToRead = 256;
    else
        uiBytesToRead = byByteCount;

    // fill data buffer
    UINT uiTemp = 0;
	while((uiByteReturn < uiBytesToRead) && (uiReturn == SUCCESS))
    {
        if((uiTemp = m_serialPort->Read(bypReadBuffer + uiByteReturn, 
                                        uiBytesToRead - uiByteReturn)) == 0)
            uiReturn = COMM_ERROR;
		else
			uiByteReturn += uiTemp;
    }

    return uiReturn;
}

////////////////////////////////////////////////////////////
// WriteMemory: writes specified number of data bytes
//      to specified start address on the 68HC11  
//
// Arguments:
//      uiStartAddress : Address to start writing at
//      byByteCount : Number of bytes to to write from uiStartAddress 
//      bypWriteBuffer : Byte array holding data to be written
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.WriteMemory(uiStartAddress,
//                         byByteCount,
//                         bypWriteBuffer);

UINT CTalker::WriteMemory(const UINT uiStartAddress,
                          const BYTE byByteCount,
                          const BYTE* bypWriteBuffer)
{   
    BYTE  byData = 0;
    BYTE* bypConfirmBuffer = NULL;
    UINT  uiBytesToWrite = 0;
    UINT  uiReturn = SUCCESS;

    // test comm port
    if(!(m_serialPort->IsPortOpen()))
        return PORT_NOT_OPEN;       // comm error

    // send memory write command
    byData = 0x41;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      // comm error

    // read and test echo confirmation
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    if(byData != 0xbe)
        return TALKER_ECHO_ERROR;       // talker error
        
    // send byte count
    byData = byByteCount;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;  

    // send start address (high byte) 
    byData = (uiStartAddress & 0xff00) >> 8;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      
    
    // send start address (low byte)
    byData = uiStartAddress & 0xff;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;      
    
    // if byte count is zero, then write 256 bytes
    if(byByteCount == 0)
        uiBytesToWrite = 256;
    else
        uiBytesToWrite = byByteCount;
    
    // write block
    if(m_serialPort->Write(bypWriteBuffer, uiBytesToWrite) != 0)
        return COMM_ERROR;      

    bypConfirmBuffer = new BYTE [uiBytesToWrite];
    ASSERT(bypConfirmBuffer);

    uiReturn = ReadMemory(uiStartAddress, byByteCount, bypConfirmBuffer);

    // test read/write buffers (should be identical)
    UINT uiLoopCount = 0;
    while((uiReturn == SUCCESS) && (uiLoopCount < uiBytesToWrite))
    {
        if(bypWriteBuffer[uiLoopCount] != bypConfirmBuffer[uiLoopCount])
            uiReturn = TALKER_ECHO_ERROR;
        else
            uiLoopCount++;
    }

    return uiReturn;
}

////////////////////////////////////////////////////////////
// CheckSWI: returns a Boolean value indicating whether a 
//      SWI signal has been received on the serial port
//
// Arguments:
//      none
//
// Return Value:
//      TRUE if SWI received, otherwise FALSE
//
// Example Usage:
//      talker.CheckSWI();

BOOL CTalker::CheckSWI()
{
    BYTE byData = 0;

    // test for new data
    if(m_serialPort->IsDataWaiting())
    {
        // read single byte
        if(m_serialPort->Read(&byData, 1) != 1)
            return FALSE;

        // test for SWI signal
        if(byData == 0x4a)
            return TRUE;
    }
    
    return FALSE;
}

////////////////////////////////////////////////////////////
// ProcessSWI: sends confirmation of SWI signal and receives
//      state of MCU registers
//
// Arguments:
//      uiReturnAddress : location of SWI
//      uiStackPointer : stack pointer
//      byCCR : condition control register
//      byACCB : accumulator B
//      btACCA : accumulator A
//      uiIX : IX register
//      uiIY : IY register
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.ProcessSWI(&uiReturnAddress,
//                        &uiStackPointer,
//                        &byCCR,
//                        &byACCB,
//                        &byACCA,
//                        &uiIX,
//                        &uiIY);

UINT CTalker::ProcessSWI(UINT& uiReturnAddress,
                         UINT& uiStackPointer,
                         BYTE& byCCR,
                         BYTE& byACCB,
                         BYTE& byACCA,
                         UINT& uiIX,
                         UINT& uiIY)
{
    BYTE byData;

    // send SWI confirmation
    byData = 0xb5;
    if(m_serialPort->Write(&byData, 1) != 0)
        return COMM_ERROR;
    
    // receive SWI confirmation ack.
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    
    // receive SWI address (high byte)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;

    uiReturnAddress = byData;
    uiReturnAddress = uiReturnAddress << 8;
    
    // receive SWI address (low byte)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;

    // decrement PC address to actual opcode
    byData--;   
    uiReturnAddress += byData;
    
    // receive stack pointer (high byte)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiStackPointer = byData;
    uiStackPointer = uiStackPointer << 8;
        
    // receive stack pointer (low byte)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiStackPointer += byData;

    // receive condition control register
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    byCCR = byData;

    // receive accumulator B
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    byACCB = byData;

    // receive accumulator A
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    byACCA = byData;

    // receive IX register (high byte)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiIX = byData;
    uiIX = uiIX << 8;

    // receive IX register (low byte)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiIX += byData;

    // receive IY register (high byte)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiIY = byData;
    uiIY = uiIY << 8;
    
    // receive IY register (low byte)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    uiIY += byData;

    // receive program counter (and discard)
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;
    if(m_serialPort->Read(&byData, 1) == 0)
        return TALKER_TIMEOUT;

    return SUCCESS;
}

////////////////////////////////////////////////////////////
// Reset: generates a software reset of the 68HC11
//
// Arguments:
//      none
//
// Return Value:
//      TRUE if successful, otherwise FALSE
//
// Example Usage:
//      talker.Reset

UINT CTalker::Reset()
{
    BYTE byData;
    UINT uiReturn = SUCCESS;

    // set clock monitor reset vector to zero
    byData = 0x7e;                          // jump opcode
    if((uiReturn = WriteMemory(0x00fd,      // low byte of clock monitor reset vector
                               1,           
                               &byData)) != SUCCESS)
        return uiReturn;
    byData = 0x00;                          // reset to $0000
    if((uiReturn = WriteMemory(0x00fe,      // high byte of clock monitor reset vector
                               1,           
                               &byData)) != SUCCESS)
        return uiReturn;
    if((uiReturn = WriteMemory(0x00ff,      // high byte of clock monitor reset vector
                               1,       
                               &byData)) != SUCCESS)
        return uiReturn;

    // enable clock monitor
    byData = 0x08;
    if((uiReturn = WriteMemory(0x1039,      // OPTION register
                               1,           
                               &byData)) != SUCCESS)
        return uiReturn;

    // force clock monitor reset by setting the FCM bit of the TEST1 register
    // (we expect a failure after this command because we will no longer
    // have communication with the MCU if it resets as it should);
    // if it returns, something went wrong
    byData = 0x04;
    if(WriteMemory(0x103e,  // TEST1 register location
                   1,   
                   &byData) == SUCCESS)
        return RESET_FAIL;

    return SUCCESS;
}

////////////////////////////////////////////////////////////
// ClosePort: calls CSerial method to close communications port
//
// Arguments:
//      none
//
// Return Value:
//      none
//
// Example Usage:
//      talker.ClosePort();

void CTalker::ClosePort()
{
	m_serialPort->Close();
}

