// Header file for CTalker (68HC11 Talker Abstraction Class)
//
// Authors: Jason Gaedtke, Matt Vandenbush
// Code Review: Jason Gaedtke
// Date: 12/7/96


#ifndef CTALKER_H
#define CTALKER_H

// include files
#include <afx.h>
#include "CSerial.h"
#include "Errcodes.h"

#define MAXTALKERSIZE 513			// maximum size of talker file

class CTalker
{
private:
	CSerial*	m_serialPort;		// communications object
	CString		m_strTalkerFile;	// talker file (*.boo)
	BOOL		m_fEEPROM;			// EEPROM flag (TRUE = enabled)

public:
	CTalker(const CSerial* comObject);

////////////////////////////////////////////////////////////
// OpenPort:  calls CSerial method to open communications port
//
// Arguments:
//		cpComString : communications paramaters
//			format: "COMport:baud,parity,length,stop"
//
// Return Value:
//		SUCCESS if successful, otherwise error code
//
// Example Usage:
//		talker.OpenPort("COM2:9600,n,8,1");
//
// Note: STRING MUST NOT CONTAIN SPACES

	UINT OpenPort(const char* cpComString);

////////////////////////////////////////////////////////////
// LoadRAMTalker: processes talker file and passes byte array to
//      CSerial.Write() to initialize talker
//
// Arguments:
//      cpComPort : communications port
//      cpTalkerFile : talker file
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.LoadRAMTalker("COM1",
//                           "c:\talke.boo");
//
// Note: SPECIFIED TALKER MUST BE A COMPILED BINARY (*.boo) FILE

	UINT LoadRAMTalker(const char* cpComPort,
					   const char* cpTalkerFile);

////////////////////////////////////////////////////////////
// LoadROMTalker: initializes ROM talker
//
// Arguments:
//      cpComPort : communications port 
//      cpMapFile : map file for ROM talker
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.LoadROMTalker("COM1",
//                           "c:\talke.map");
//
// Note: EVB2 BOARD SWITCH MUST BE IN "BUFFALO" POSITION;
//      THIS FUNCTION HAS NOT BEEN TESTED!

	UINT LoadROMTalker(const char* cpComPort,
					   const char* cpMapFile);

////////////////////////////////////////////////////////////
// GetTalkerName: updates the sole argument to contain the
//		name of the current talker file
//
// Arguments:
//		strTalkerName : CString object which will contain
//			name of talker file
//
// Return Value:
//		TRUE if successful, otherwise FALSE
//
// Example Usage:
//		talker.GetTalkerName(strTalkerName);

	void GetTalkerName(CString& strTalkerName)
	{
		strTalkerName = m_strTalkerFile;
	};

////////////////////////////////////////////////////////////
// SetEEPROMFlag: Sets status of m_fEEPROM
//
// Arguments:
//		bEEflag : the value that the flag is to be set to
//
// Return Value:
//		none
//
// Example Usage:
//		talker.SetEEPROMFlag(bEEflag);

	void SetEEPROMFlag(const BOOL bEEflag)
	{
		m_fEEPROM = bEEflag;
	};

////////////////////////////////////////////////////////////
// IsEEPROMEnabled: Boolean test: returns status of m_fEEPROM
//
// Arguments:
//		none
//
// Return Value:
//		TRUE if successful, otherwise FALSE
//
// Example Usage:
//		talker.IsEEPROMEnabled();

	BOOL IsEEPROMEnabled()
	{
		return m_fEEPROM;
	};

////////////////////////////////////////////////////////////
// ReadRegisters: reads control registers on 68HC11 
//      and returns a copy to caller (since the
//      values are passed by reference)
//
// Arguments:
//      uiStackPointer : stack pointer
//      byCCR : condition control register
//      byACCB : accumulator B
//      byACCA : accumulator A
//      uiIX : IX register
//      uiIY : IY register
//      uiProgramCounter : program counter
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.ReadRegisters(uiStackPointer,
//                           byCCR,
//                           byACCB,
//                           byACCA,
//                           uiIX,
//                           uiIY,
//                           uiProgramCounter);

	UINT ReadRegisters(UINT &uiStackPointer,
					   BYTE &byCCR,
					   BYTE &byACCB,
					   BYTE &byACCA,
					   UINT &uiIX,
					   UINT &uiIY,
					   UINT &uiProgramCounter);

////////////////////////////////////////////////////////////
// WriteRegisters: updates control registers on 68HC11 
//      to reflect argument values
//
// Arguments:
//      uiStackPointer : stack pointer
//      byCCR : condition control register
//      byACCB : accumulator B
//      byACCA : accumulator A
//      uiIX : IX register
//      uiIY : IY register
//      uiProgramCounter : program counter
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.WriteRegisters(uiStackPointer,
//                            byCCR,
//                            byACCB,
//                            byACCA,
//                            uiIX,
//                            uiIY,
//                            uiProgramCounter);

	UINT WriteRegisters(const UINT uiStackPointer,
					    const BYTE byCCR,
					    const BYTE byACCB,
					    const BYTE byACCA,
					    const UINT uiIX,
					    const UINT uiIY,
					    const UINT uiProgramCounter);

////////////////////////////////////////////////////////////
// ReadMemory: reads a specified number of data bytes
//      from a specified start address on the 68HC11;  
//      data is returned since the byte array is
//      passed by reference
//
// Arguments:
//      uiStartAddress : Address to start reading at
//      byByteCount : Number of bytes to read from uiStartAddress 
//      bypReadBuffer : Byte array holding data after completion
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.ReadMemory(uiStartAddress,
//                        byByteCount,
//                        bypReadBuffer);

	UINT ReadMemory(const UINT uiStartAddress,
					const BYTE byByteCount,
					BYTE* bypReadBuffer);

////////////////////////////////////////////////////////////
// WriteMemory: writes specified number of data bytes
//      to specified start address on the 68HC11  
//
// Arguments:
//      uiStartAddress : Address to start writing at
//      byByteCount : Number of bytes to to write from uiStartAddress 
//      bypWriteBuffer : Byte array holding data to be written
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.WriteMemory(uiStartAddress,
//                         byByteCount,
//                         bypWriteBuffer);

	UINT WriteMemory(const UINT uiStartAddress,
			 		 const BYTE byByteCount,
			 		 const BYTE* bypWriteBuffer);

////////////////////////////////////////////////////////////
// CheckSWI: returns a Boolean value indicating whether a 
//      SWI signal has been received on the serial port
//
// Arguments:
//      none
//
// Return Value:
//      TRUE if SWI received, otherwise FALSE
//
// Example Usage:
//      talker.CheckSWI();

	BOOL CheckSWI();

////////////////////////////////////////////////////////////
// ProcessSWI: sends confirmation of SWI signal and receives
//      state of MCU registers
//
// Arguments:
//      uiReturnAddress : location of SWI
//      uiStackPointer : stack pointer
//      byCCR : condition control register
//      byACCB : accumulator B
//      btACCA : accumulator A
//      uiIX : IX register
//      uiIY : IY register
//
// Return Value:
//      SUCCESS if successful, otherwise error code
//
// Example Usage:
//      talker.ProcessSWI(&uiReturnAddress,
//                        &uiStackPointer,
//                        &byCCR,
//                        &byACCB,
//                        &byACCA,
//                        &uiIX,
//                        &uiIY);

	UINT ProcessSWI(UINT& uiReturnAddress,
					UINT& uiStackPointer,
					BYTE& byCCR,
					BYTE& byACCB,
					BYTE& byACCA,
					UINT& uiIX,
					UINT& uiIY);
	
////////////////////////////////////////////////////////////
// Reset: generates a software reset of the 68HC11
//
// Arguments:
//      none
//
// Return Value:
//      TRUE if successful, otherwise FALSE
//
// Example Usage:
//      talker.Reset

	UINT Reset();

////////////////////////////////////////////////////////////
// ClosePort: calls CSerial method to close communications port
//
// Arguments:
//      none
//
// Return Value:
//      none
//
// Example Usage:
//      talker.ClosePort();

	void ClosePort();

};

#endif
