/*	Private Header file for the FAT16 file system routines	*/

/*	Description of the FAT16 as it pertains to the MMCARD
 *
 *	Sectors are of 512 bytes as standard for the MMCARD
 *	Sector 0 is the boot sector, this contains
 *	Offset Description 										Size 
 *	00h 	Jump Code + NOP 								3 Bytes 
 *	03h 	OEM Name 										8 Bytes 
 *	0Bh 	Bytes Per Sector 								1 Word 
 *	0Dh 	Sectors Per Cluster 							1 Byte 
 *	0Eh 	Reserved Sectors 								1 Word 
 *	10h 	Number of Copies of FAT 						1 Byte 
 *	11h 	Maximum Root Directory Entries 					1 Word 
 *	13h 	Number of Sectors in Partition Smaller than 32MB 1 Word 
 *	15h 	Media Descriptor (F8h for Hard Disks) 			1 Byte 
 *	16h 	Sectors Per FAT 								1 Word 
 *	18h 	Sectors Per Track 								1 Word 
 *	1Ah 	Number of Heads 								1 Word 
 *	1Ch 	Number of Hidden Sectors in Partition 			1 Double Word 
 *	20h 	Number of Sectors in Partition 					1 Double Word 
 *	24h 	Logical Drive Number of Partition 				1 Word 
 *	26h 	Extended Signature (29h) 						1 Byte 
 *	27h 	Serial Number of Partition 						1 Double Word 
 *	2Bh 	Volume Name of Partition 						11 Bytes 
 *	36h 	FAT Name (FAT16) 								8 Bytes 
 *	3Eh 	Executable Code 								448 Bytes 
 *	1FEh	Executable Marker (55h AAh) 					2 Bytes 
 *	
 *	So define the offsets to the parts we need	*/
#define FAT16_BytesPerSector 	0x000b
#define FAT16_SectorsPerCluster 0x000d
#define FAT16_NumCopiesOfFAT 	0x0010
#define FAT16_MaxRootDirEntries 0x0011
#define FAT16_NumbOfSectorSmall 0x0013
#define FAT16_SectorPerFat	 	0x0016
#define FAT16_HiddenSectors	 	0x001c
#define FAT16_NumbOfSectorBig	0x0020
#define FAT16_VolumeNameStart	0x002b
#define FAT16_FileSystemName	0x0036

/*	The format of the traditional DOS 8.3 Directory entry is as follows:
 *	Offset  Length    	Value  
 *	0  	8 bytes    	Name  
 *	8  	3 bytes    	Extension  
 *	11 	1 byte    Attribute (00ARSHDV) 
 *	 		 0: unused bit 
 *	 		 A: archive bit, 
 *			 R: read-only bit 
 *			 S: system bit 
 *			 D: directory bit 
 *			 V: volume bit 
 *	Other spare bytes here, used in FAT32 etc... 
 *	22  	word    	Time  
 *	24  	word    	Date  
 *	26  	word    	Cluster (desc. below)  
 *	28  	dword   	File Size  
 *	
 *	Note: WORD = 2 bytes, DWORD = 4 bytes
 *	
 *	So define the offsets to the parts we need	*/
	
 
#define FAT16_FILENAME	0x0000
#define FAT16_FILE_EXT	0x0008
#define FAT16_ATTRIBUTE	0x000b
#define FAT16_START_CLUSTER	0x001a
#define FAT16_FILE_BYTE_SIZE	0x001c

#define DIR_ENTRY_SIZE	32
/*	On Storage device initialization, read the boot block
 *	From this data, load the FAT16Boot structure
 *	From this calculate the position of 
 *			root directory start
 *			root directory size
 *			cluster 2 start
 *			cluster size
 *
 *	On a call to open a file
 *	Search the directory space for the filename and fileext
 *	If found
 *	From the Directory entry, load the File Structure with
 *		StartCluster
 *		FileByteSize
 *		Data from the first sector of first cluster
 *		Next cluster from the FAT table.
 */

typedef struct PartitionEntryStruct{
	uint8	BootDesc;
	uint8	FirstPartitionSector0;
	uint8	FirstPartitionSector1;
	uint8	FirstPartitionSector2;
	uint8	FileSystemDescriptor;
	uint8	LastPartSect0;
	uint8	LastPartSect1;
	uint8	LastPartSect2;
	uint32	FirstSectPos;
	uint32	NumSectInPartition;
};

struct BootDataType{
				uint16	BytesPerSector;
				uint16	SectorsPerCluster;
				uint16 	NumCopiesOfFAT;
				uint16 	MaxRootDirEntries;
				uint16	NumbOfSectorSmall;
				uint16	SectorPerFat;
				uint32	NumbOfSectorBig;
				uint8	VolumeName[12];
				uint32 	RootDirStart;	/*	Memory address of start of Dir*/
				uint16	RootDirSize;
				uint32	Cluster2Start;	/* Memory address of Cluster 2 (First Data)*/
				uint16	ClusterSize;
				uint8	FileSystemName[9];
				uint32	HiddenSectors;
			};

struct FileDataType{
				uint8	FileName[8];
				uint8	FileExt[3];
				uint8	Attribute;
				uint8	ResvNT;
				uint8	Time100th;
				uint16	Time;
				uint16	Date;
				uint16	LastDate;
				uint16	ResvFat32;
				uint16	LastWriteTime;
				uint16	LastWriteDate;
				uint16	StartCluster;
				uint32	FileByteSize;		/*	End of Dir entry		*/
				uint32	StartAddress;		/*	Calculated			*/
 				uint8	LocalSector[513];	/*	Buffer with the sector		*/
				uint16	SectorInCluster;	/*	Which Sector of the cluster	*/
				uint32	StartAddressOfLoaded;	/*	The address of the loaded sector*/
				uint16	ClusterLoaded;		/*	Currently used cluster		*/
				uint16	Ptr;			/*	Current file pointer		*/
				uint16	Flags;	
				uint16	Mode;
			};
#define MODIFIED	0x0001u	/*	Used to indicate that the sector loaded has been modified */ 
#define OPENED_OK	0x0002u	/*	Used to flag that the file was opened OK				*/
#define INDEX_OK	0x0004u	/*	Used to flag that the IndexOfEndData is valid			*/

typedef struct FAT16Time{
	unsigned Sec : 5;
	unsigned Min : 6;
	unsigned Hour: 5;
};
typedef struct FAT16Date{
	unsigned Day   : 5;
	unsigned Month : 4;
	unsigned Year  : 7;
};




/*	Odd Functions for the FAT16 file system Code	*/
uint16	BootSectorFound(void);
void	LoadBootInfo(void);
uint16	SaveFileData(void);
uint16	LittleCharToInt(uint8 *LittleEndianInt);
uint32	LittleCharToLong(uint8 *LittleEndianLong);
void	LoadFileName(uint8 *FileName);
void	GetFileName(uint8 *FileName);
uint16	FileOpenOK(void);
uint16	LoadFileAt(uint32 *FilePtr);
uint16	StoreFileDirEntry(void);


