/*	Functions to manage the FAT tables		*/

#include <hardwareIF.h>
#include "..\MMCard\MMCard.h"

#include "FAT16.h"
#include "FAT16P.h"
#include "FAT16Man.h"
#include "FAT16DirMan.h"
#include <string.h>

extern struct BootDataType Boot;
extern struct FileDataType File;

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to find the cluster, and the address of the sector in that cluster 
 *	of the offset specified, of the presently open file.
 *	File.LocalSector is used to load FAT table sectors.
 *	ErrorCodes
 *		END_OF_FILE
 *		LOST_CLUSTER
 *		BAD_CLUSTER
 */
uint16	FindFileCluster(uint32 DataOffset, uint32 *SectorAddress, uint16 *ClusterNum)
{
uint16	ErrorCode = NoError;
uint16	ClusterCount;
uint16	ClustersFound = 0;
uint16	ThisClusterNum;
//uint16	ClusterNum;
uint32	AddressOfCluster;
uint16	ClusterSize;
	ClusterSize = Boot.BytesPerSector * Boot.SectorsPerCluster;
/*	Test DataOffset is not silly	*/
	if(DataOffset > File.FileByteSize){/*Check that the DataOffset is within the filesize limitation*/
   		ErrorCode = END_OF_FILE;
		}

/*	Make a special case for the beginning of the file, i.e. if the DataOffset = 0 or less than the Boot.BytesPerSector	*/
	if(!ErrorCode){
		if(DataOffset < Boot.BytesPerSector){
			*ClusterNum = File.StartCluster;
			*SectorAddress = File.StartAddress;
			ErrorCode = 0xfe;
		}
	}
	
/*	Make a special case for an empty or small file. only one cluster but maybe in any of the sectors of that cluster	*/
	if(!ErrorCode){
		if(File.FileByteSize < (Boot.BytesPerSector * Boot.SectorsPerCluster)){
			*SectorAddress = File.StartAddress +((DataOffset / Boot.BytesPerSector) * Boot.BytesPerSector); 
			File.SectorInCluster = (File.FileByteSize % (Boot.SectorsPerCluster * Boot.BytesPerSector)) / Boot.BytesPerSector;	/*	Calc the sector within the cluster	*/
			ErrorCode = 0xfe;			/*	This is for the any sector of the cluster	*/
			}
		}
/*	Make a special case for a get end of file sector, if the File.FileByteSize == DataOffset then look instead for the last sector	*/
	if(!ErrorCode){
		if(File.FileByteSize == DataOffset){
			ErrorCode = FindLastCluster(ClusterNum);

			if(*ClusterNum == File.ClusterLoaded){
				ErrorCode = NoError;
			}
			AddressOfCluster = Boot.Cluster2Start +	((Boot.SectorsPerCluster * ((uint32)(*ClusterNum) - 2)) * (uint32)Boot.BytesPerSector);	/* Find the Cluster boundary	*/
   			*SectorAddress = AddressOfCluster + (((DataOffset % ClusterSize) / Boot.BytesPerSector) * Boot.BytesPerSector);			/* Find the sector boundary	*/
			/*		= Start of cluster + (((Remainder with cluster)/sectorsize) * sectorsize) <-this / gets rid of the fractional part	*/
			File.SectorInCluster = (File.FileByteSize % (Boot.SectorsPerCluster * Boot.BytesPerSector)) / Boot.BytesPerSector;	/*	Calc the sector within the cluster	*/
			ErrorCode = 0xfe;
			}
		}

/*	Go find the general case	*/
   	if(!ErrorCode){		/*	Find the number of clusters to search for in the FAT trail	*/
   		ClusterCount = (DataOffset+1)/(Boot.BytesPerSector * Boot.SectorsPerCluster);	/* This is calc from the Number of bytes into the file		*/
   		ThisClusterNum = File.StartCluster;					/*	Start looking at the beginning				*/
   		do{									/*	Plod through the FAT table until the cluster is found	*/
   			*ClusterNum = GetNextClusterNumber(ThisClusterNum);
   			switch(*ClusterNum){
   				case 0xfff8:case 0xfff9:case 0xfffa:case 0xfffb:case 0xfffc:case 0xfffd:case 0xfffe:case 0xffff:{
   					ErrorCode = END_OF_FILE; 	/*	end of file found, then if this is the last to be looked for...		*/
   					ClustersFound++;							/*	Count it to be sure	*/
   					*ClusterNum = ThisClusterNum;			/*	If we have the right number then		*/
   					break;						/*	This is the required cluster and the previous number belongs to it.	*/
   				} 
   				case 0x0000:ErrorCode = LOST_CLUSTER; break;		/*	If NextClusterNum >= 0x0000 LOST_CLUSTER found		*/
   				case 0xfff7:ErrorCode = BAD_CLUSTER; break;		/*	If NextClusterNum = 0xfff7 BAD_CLUSTER found		*/
   				default :{
   					ClustersFound++;				/*	If the desired cluster number has been found		*/ 
   					if(ClustersFound == ClusterCount) {		/*	Then the previous Number, and not current number points	*/ 
   					;
					}
   					else{						/*	Otherwise keep following the chain			*/
   						ThisClusterNum = *ClusterNum;		/*	Increment ClustersFound, and point to the next		*/
   					}
   				}
   			}
   		}while((ClustersFound < ClusterCount) && (!ErrorCode));			/*	check if found required cluster					*/
    			/*	If all OK. Find the address of the sector that contains the data required	*/
   		if(((ErrorCode == END_OF_FILE) && (ClustersFound == ClusterCount)) ||	(!ErrorCode)){
			AddressOfCluster = Boot.Cluster2Start +	((Boot.SectorsPerCluster * ((uint32)(*ClusterNum) - 2)) * (uint32)Boot.BytesPerSector);	/* Find the Cluster boundary	*/
			File.SectorInCluster = (File.FileByteSize % (Boot.SectorsPerCluster * Boot.BytesPerSector)) / Boot.BytesPerSector;	/* Calc the sector within the cluster	*/
   			*SectorAddress = AddressOfCluster + (((DataOffset % ClusterSize) / Boot.BytesPerSector) * Boot.BytesPerSector);		/* Find the sector boundary	*/
   			ErrorCode = NoError;
   		}
   	}
	if(ErrorCode == 0xfe){
		ErrorCode = NoError;
	}
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to return the cluster number of the next in the chain.
 *	Uses the specified cluster number to find the next cluster 
 *	File.LocalSector is used to load FAT table sectors.
 *
 *	ErrorCodes are inherent in the Clusternumber
 *		END_OF_FILE	0xfff8 -> 0xffff
 *		LOST_CLUSTER	0x0000 -> 0x0002
 *		BAD_CLUSTER	0xfff7
 */
uint16	GetNextClusterNumber(uint16 ThisClusterNum)
{
uint16	ErrorCode = NoError;
uint16	ClusterNum = 0x0000;
uint32	Fat1EntryAddress;
uint32	Fat1SectorAddr;

		Fat1EntryAddress = ((Boot.HiddenSectors + 1) * Boot.BytesPerSector) + (ThisClusterNum * 2);	/*	Get the File startcluster number, 		*/
		Fat1SectorAddr = (Fat1EntryAddress/Boot.BytesPerSector) * Boot.BytesPerSector;										/*	calculate the FatTable entry position	*/
		ErrorCode = MMCardReadSector(Fat1SectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);	/*	Get the fat table sector*/
		if(!ErrorCode){				/*	Cluster numbers are in little endian format.*/
			ClusterNum =  LittleCharToInt(&File.LocalSector[Fat1EntryAddress % Boot.BytesPerSector]);
		}
	return ClusterNum;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to Find the last Cluster, of the open file 
 *	returns errorcodes
 *		LoastCluster
 *		BAD_CLUSTER
 *		END_OF_FILE
 *	If EndOFFile returned, then the LastCluster is the number required 
 */
uint16	FindLastCluster(uint16 *LastClusterNumber)
{
uint16	ErrorCode = NoError;
uint16	ThisClusterNum;
uint16	ClusterNum;

	ThisClusterNum = File.StartCluster;
	do{
		ClusterNum = GetNextClusterNumber(ThisClusterNum);
		switch(ClusterNum)
			{
			case 0xfff8:case 0xfff9:case 0xfffa:case 0xfffb:case 0xfffc:case 0xfffd:case 0xfffe:case 0xffff:
				{													/*	If NextClusterNum >= 0xfff8 end of file found	*/
				ErrorCode = END_OF_FILE; 
				*LastClusterNumber = ThisClusterNum;
				break;
				}
			case 0x0000:	ErrorCode = LOST_CLUSTER; break;		/*	If NextClusterNum >= 0x0000 LOST_CLUSTER found	*/
			case 0xfff7:	ErrorCode = BAD_CLUSTER; break;		/*	If NextClusterNum = 0xfff7 BAD_CLUSTER found		*/
			default :		ThisClusterNum = ClusterNum;			/*	Increment ClustersFound, 						*/
			}
		}while(!ErrorCode);										/*	check if found required cluster					*/
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to Find the Next available Cluster in the FAT table 
 *	returns ClusterNum, if Media full, then returns 0xffff
 *	SHOULD! Also clears the data in that cluster
 */
uint16	FindFreeCluster(void)
{
uint16	ErrorCode = NoError;
uint16	i = 0;
uint16	SectorCount = 0;
uint32	SectorAddr;
uint16	FreeFound = 0;
uint16	FreeClusterNum = 0xffff;

/*	Speed improvement, look for next free cluster, starting at present loaded cluster	*/
/*	Find the SectorCount to start with	*/
/*	This assumes that there is an open file, however, that's probably a reasonable assumption*/
	SectorCount = ( File.ClusterLoaded * 2) / Boot.BytesPerSector;
/*	The fat table starts at ((Boot.HiddenSectors + 1) * Boot.BytesPerSector)	*/
	do{
		SectorAddr = ((SectorCount + Boot.HiddenSectors + 1) * Boot.BytesPerSector);/*Find the address of the loaded cluster.*/
		/*	Load it and Look for a free cluster */
		ErrorCode = MMCardReadSector(SectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);	/*	Get the fat table sector*/
		if(!ErrorCode){
			i = 0;
			while((!FreeFound) && (i < Boot.BytesPerSector)){	/*	Search until a free cluster is found or untill the end of the sector	*/
				if((!File.LocalSector[i]) && (!File.LocalSector[i+1])){
					FreeFound = 1;
					FreeClusterNum = (SectorCount * (Boot.BytesPerSector/2)) + (i/2); /* Calc the cluster number	*/ 
    				}
				else{
					i++; i++;
    				} 
    			}
			SectorCount++;	
		}
		if(ErrorCode){
			i = ErrorCode;
		}
	}while((!ErrorCode) && (!FreeFound) && (SectorCount < Boot.SectorPerFat));
	return FreeClusterNum;		
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to Write the Cluster Data to both the FAT tables
 *	Reverses order of Bytes as should be Little-Endian
 *	Returns ErrorCodes
 *		WriteFail etc....
 */
uint16	WriteToFAT1and2(uint16	ClusterNum, uint16 ClusterData)
{
uint16	ErrorCode = NoError;
uint32	EntryAddr;
uint32	SectorAddr;
uint16	EntryIndex;

/*	Find the sector with the FAT entry required to be changed	*/
/*	The fat table starts at ((Boot.HiddenSectors + 1) * Boot.BytesPerSector)	*/
	EntryAddr = ((uint32)(Boot.HiddenSectors + 1) * (uint32)Boot.BytesPerSector) + (uint32)(ClusterNum * 2);
	EntryIndex = EntryAddr % Boot.BytesPerSector;
	SectorAddr = EntryAddr - EntryIndex;
/*	Load it and change the entry in the File.LocalSector, then save the sector	*/
	ErrorCode = MMCardReadSector(SectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);	/*	Get the fat table sector*/
	if(!ErrorCode)
		{																		/*	Cluster numbers are in little endian format.*/
		File.LocalSector[EntryIndex] = (uint8)(ClusterData);
		File.LocalSector[EntryIndex +1] = (uint8)(ClusterData>>8);
		ErrorCode = MMCardWriteSector(SectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);	
		}
	if(!ErrorCode)
		{
		if(Boot.NumCopiesOfFAT >= 2)										/*	Check the copies of the FAT and if a second is used	*/
			{																/*	Get the corresponding Sector of the FAT2 table 		*/
			SectorAddr += ((uint32)Boot.SectorPerFat * (uint32)Boot.BytesPerSector);		/*	(if used) and save the same sector to it			*/
			ErrorCode = MMCardWriteSector(SectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);	
			}
		}
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to find the next available cluster, 
 *	and allocating it to the open file
 *	And returning ErrorCodes
 *	!!!!!	This uses the file buffer so that gets wiped	!!!!!
 */ 
uint16	AllocNewCluster(void)
{
uint16	ErrorCode = NoError;
uint16	LocalErrorCode = NoError;
uint16	LastClusterNumber;
uint16	NextFreeCluster;
uint16	i;
uint32	SectAddr = 0;

	/*Find the next available cluster					*/
	NextFreeCluster = FindFreeCluster();
	if(NextFreeCluster == 0xffff){
		ErrorCode = MEDIA_FULL;
		}

/*	With the following errorcodes, if a routine errors, 
 *	then there is a debug catch, but the rest of the code keeps going.
 */
	if(!ErrorCode){
		ErrorCode = WriteToFAT1and2(NextFreeCluster, 0xffff);
		if(ErrorCode){
			LocalErrorCode = ErrorCode;
			ErrorCode = NoError;
		}
		ErrorCode = FindLastCluster(&LastClusterNumber);
		if(ErrorCode){
			if(ErrorCode != 0x6000){
				LocalErrorCode = ErrorCode + 1;
				ErrorCode = NoError;
			}
		}
		ErrorCode = WriteToFAT1and2(LastClusterNumber, NextFreeCluster);
		if(ErrorCode){
			LocalErrorCode = ErrorCode + 2;
			ErrorCode = NoError;
		}
	}
	
	/*	Calc the address of the Next Free Cluster	*/
	SectAddr = Boot.Cluster2Start +	((Boot.SectorsPerCluster * ((uint32)(NextFreeCluster - 2))) * (uint32)Boot.BytesPerSector);
/*	Clear the File buffer and write it to the Next Free Cluster, repeating for all the sectors in the cluster	*/
	File.SectorInCluster = 0; 
	do{
		SectAddr += (File.SectorInCluster * 2);		/*	Calc the sector address		*/ 
		for(i=0;i<Boot.BytesPerSector;i++){		/*	clear the local sector		*/
			File.LocalSector[i] = 0;
			}
		ErrorCode = MMCardWriteSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);
		File.SectorInCluster++;				/*	Next sector in the cluster	*/
	}while(File.SectorInCluster < Boot.SectorsPerCluster);
	File.SectorInCluster = 0;				/*	Reset the counter to the start of the cluster	*/ 
	if(!ErrorCode && LocalErrorCode){
		ErrorCode = LocalErrorCode;
	}
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to Follow the file FAT trail and deallocate clusters 
 *	by marking then 0x0000
 *	This is part of the erase or delete function.
 */ 
uint16	DeAllocClusters(void)
{
uint16	ErrorCode = NoError;
uint16	NextCluster;
uint16	ClusterNum;

	ClusterNum = File.StartCluster;
	ErrorCode = WriteToFAT1and2(ClusterNum, 0x0000);
	do{
		NextCluster =	GetNextClusterNumber(ClusterNum);
		ErrorCode = WriteToFAT1and2(ClusterNum, 0x0000);
		ClusterNum = NextCluster;
		}while(NextCluster < 0xfff8);

	return ErrorCode;
}

