/*	Access to the card is only to read and write full sectors
 *	So only 3 functions Init, Read, Write
 *	Plus some Status read/modify functions.
 *
 *	Macros to be define by the hadrwareIF.h and associated files
 *	CardPresent -> a moethod of detecting the card is inserted
 */

#include "hardwareIF.h"
#include <string.h>
#include "MMCard.h"
#include "MMCardP.h"


/*	A structure for the Card data and flags etc.	*/
struct MMCARDType Card;

/*****************************************************************************
 *	Function to Initialise the data structures and hardware associated
 *	With the MMCard interface.
 *	This must be specific to the hardware inplementation.
 *	Determined by the macros..INITMMCardPwrCtl and INITMMCardDetect
 */
void	MMCardInitSystem(void)
{
	INITCardPwrCtlPin;
	INITCardDetect;
	
	InitMMCardSPI();
	memcpy(Card.CID,"CIDCIDCIDCIDCID",16);
	memcpy(Card.CSD,"CSDCSDCSDCSDCSD",16);
	Card.Flags = 0;
	Card.IsReady = 0;
}

/*****************************************************************************
 *	Function to Control the power to the MMCard, 
 *	this is required before communicating with the card.
 *	Could also be used for the other 7 outputs of the chip.
 */
uint16	MMCardControl(uint16 Cmd)
{
uint16	State = 0;

	switch(Cmd){
		case 1 :	MMCardPwrON; break;
		case 0 :	MMCardPwrOFF; break;
		default : 	break;
		}
	MMCardPwrQuy(State);
	return State;
}

/*****************************************************************************
 *	Function to report state of card switch
 */
uint16	MMCardIn(void)
 {
 uint16 result = 0;
 
 	if(CardPresent){
		result = 1;
		}
	else{
		result = 0;
		}
	return result;
}
 
/*****************************************************************************
 *	Function to check the presence of a card,  
 *	A few possibilities.
 *	if MMCardPresent and Card.Flags says !Present
 *		Then card has just been inserted, 
 *		wait/check debounce and then power up and intiialize
 *		if successful then return 1/yes
 *	if MMCardPresent amd Card.Flags says Present
 *		then return 1/yes
 *	if !MMCardPresent amd Card.Flags says !Present
 *		then return 0/no
 *	if !MMCardPresent and Card.Flags says Present
 *		Then card has just been removed, 
 *		power down and clear data structures
 *		then return 0/no
  */
uint16	MMCardInserted(void)
{
uint16	Result = 0;
uint16	i;
uint16	ErrorCode = NoError;

	if(CardPresent){
		if(Card.Flags & Present){ 	/*	If it WAS present	*/
			Result = 1;
			}
		else{ 				/*	If is WAS NOT present	*/
			}
			i = 2;
    		MMCardDelay(i);			/*	Wait debounce period	*/
    		if(CardPresent){		/*	If still there		*/
    			(void)MMCardControl(1);	/*	Power On and wait	*/
				i = 20;
    			MMCardDelay(i);
    			ErrorCode = MMCardInit();/*	Initialise the card data*/
    			if(!ErrorCode){
    				Result = 1;
    				}
    			else{
    				Result = 0;
    				}
    			}
    		}
	else {					/*	If MMCardPresent = NO	*/
		if(Card.Flags & Present){ 	/*	If is WAS present	*/
			MMCard_Clear();		/*	Clear the flags		*/
			(void)MMCardControl(0);	/*	Power off		*/
			}
		else{ 				/*	If it WAS NOT present	*/
			Result = 0;
			}
		}

	return Result;
}

/*****************************************************************************
 *	Function to Initialize the MMCard, 
 *	this is required before communicating with the cards.
 */
uint16	MMCardInit(void)
{
uint16 ErrorCode = NoError;

	if(CardPresent){
		ErrorCode = MMCard_InitComs();				/*	Init the comms to the card	*/
		if(!ErrorCode){
			ErrorCode = MMCardReadCardData(SEND_CSD);	/*	Get the Card Specific Data	*/
			if(!ErrorCode){
				ErrorCode = MMCardReadCardData(SEND_CID);/*	Get the Card ID	*/
				}
			}
		if(!ErrorCode){
			Card.IsReady = 1;
			Card.Flags BitOn CardOK;
			Card.Flags BitOn Present;
			}
		else{
			MMCard_Clear();					/*	Clear the flags			*/
			}
		}
	else{
		MMCard_Clear();						/*	Clear the flags			*/
		ErrorCode = MMCardNotPresent;
		}
	return ErrorCode;
}

/*****************************************************************************
 *	Function to load the MMCARD Memory data into the specified buffer
 *	
 */
uint16	MMCardReadSector(uint32 Address, uint16 SectorSize, uint8 *Buffer)
{
uint16 ErrorCode = NoError;

	if(Card.IsReady){
		ErrorCode =	MMCardReadCardMem(Address, SectorSize, Buffer);
	}
	else{
		ErrorCode = MMCardNotReady;
	}
	return ErrorCode;
}


/*****************************************************************************
 *
 *
 */
uint16	MMCardWriteSector(uint32 Address, uint16 SectorSize, uint8 *Buffer)
{
uint16 ErrorCode = NoError;

	if(Card.IsReady){
		ErrorCode =	MMCardWriteCardMem(Address, SectorSize, Buffer);
		}
	else{
		ErrorCode = MMCardNotReady;
		}
	return ErrorCode;
}

/*****************************************************************************
 *
 *
 */
void	MMCard_Clear(void)
{
	Card.Flags = 0;
	Card.IsReady = 0;
}
/*****************************************************************************
 *
 *
 */
uint16	MMCard_Ready(void)
{
	return (Card.Flags & CardOK);
}

/*****************************************************************************
 *
 *
 */
uint16		MMCard_Flags(uint16 Command)
{
uint16	Result = 0;

	switch(Command)
		{
		case	ClearFlags:	Card.Flags = 0; Card.IsReady = 0; break;
		case	CardReady	:	Result = Card.IsReady; break;
		}
	return Result;
}

/******************************************************************
******************************************************************
******************************************************************
******************************************************************/
