// filename ******** PWM.C ************** 
// Example main program for Pulse Width Modulated outputs 
// Technological Arts MC9S12C32 board  
// PT3, PT1 and PT0 are PWM outputs
// PT3 is a 166ms wave, variable duty cycle 
// PT1 and PT0 are outputs to LEDs (count of 250 = 666us period)
// Duty8 varies from 0 to 250 
//     __             ________________             __
// PT1   \___________/  250-Duty8     \___________/
//        ___________                  ___________
// PT0 __/  Duty8		 \________________/ 	Duty8		\___

// PAD6 and PAD7 are inputs from switches
// if PAD7 button pushed,  duty cycle increases
// if PAD6 button pushed,  duty cycle decreses
// if both are pushed, duty cycles are cleared

//  This example accompanies the books
//   "Embedded Microcomputer Systems: Real Time Interfacing", Brooks-Cole, copyright (c) 2000,
//   "Introduction to Embedded Microcomputer Systems: 
//    Motorola 6811 and 6812 Simulation", Brooks-Cole, copyright (c) 2002,
//    Jonathan W. Valvano 5/27/04

// You may use, edit, run or distribute this file 
//    as long as the above copyright notices remain 
#include <hidef.h>      /* common defines and macros */
#include <mc9s12c32.h>     /* derivative information */
#include "pll.h"


#pragma LINK_INFO DERIVATIVE "mc9s12c32"

// ---------PWM_Init---------
// enable PWM channels 3,1,0
// inputs: none
// outputs: none
// concatenates 2,3
void PWM_Init(void){
  MODRR = 0x0B;    // PTT3,PTT1,PTT0 associated with PWM
  PWME = 0x0B;	   // enable channels 0,1,3
  // bits 5-0, PWME5-0, Pulse Width Channel 5-0 Enable
  PWMPOL = 0x05;   // PTT2,PPT0 high then low, PTT1 low then high
  // bits 5-0, PPOL5-0, Pulse Width Channel 5-0 Polarity
  //  1 = high at beginning of period, then low when duty count reached.
  //  0 = low at beginning of period, then high when duty count reached.
  PWMCLK = 0x08;   // clock sources, clock A for 0,1, clock SB for 3
  // bit 5, PCLK5, Pulse Width Channel 5 Clock Select
  //   1 = Clock SA is the clock source for PWM channel 5
  //   0 = Clock A is the clock source for PWM channel 5
  // bit 4, PCLK4, Pulse Width Channel 4 Clock Select
  //   1 = Clock SA is the clock source for PWM channel 4
  //   0 = Clock A is the clock source for PWM channel 4
  // bit 3, PCLK3, Pulse Width Channel 3 Clock Select
  //   1 = Clock SB is the clock source for PWM channel 3  
  //   0 = Clock B is the clock source for PWM channel 3
  // bit 2, PCLK2, Pulse Width Channel 2 Clock Select
  //   1 = Clock SB is the clock source for PWM channel 2
  //   0 = Clock B is the clock source for PWM channel 2
  // bit 1, PCLK1, Pulse Width Channel 1 Clock Select
  //   1 = Clock SA is the clock source for PWM channel 1
  //   0 = Clock A is the clock source for PWM channel 1
  // bit 0, PCLK0, Pulse Width Channel 0 Clock Select
  //   1 = Clock SA is the clock source for PWM channel 0
  //   0 = Clock A is the clock source for PWM channel 0
  PWMPRCLK = 0x76; // B is bus clock/128, A is 24MHz/64 = 375kHz
  // bits 6-4, PCKB2-PCKB0, n is 0 to 7, Prescaler Select for Clock B
  //           clock B is bus clock/(2**n)
  // bits 2-0, PCKA2-PCKA0, m is 0 to 7, Prescaler Select for Clock A
  //           clock A is bus clock/(2**m)
  PWMCAE = 0;      // left aligned mode
  // bits 5-0, CAE5-0, Center Aligned Output Mode on Channel 5-0
  //   1 = operates in Center Aligned Output Mode.
  //   0 = operates in Left Aligned Output Mode.
  PWMCTL = 0x20;   // Concatenate 2+3, runs in wait/freeze mode 
  // bit 6, CON45, Concatenate channels 4 and 5
  //   1 = Channels 4 and 5 are concatenated to create one 16-bit channel 
  //       Channel 4 becomes high byte and channel 5 becomes low byte 
  //       Channel 5 output pin is used as output for 16-bit PWM
  //       Channel 5 clock select control-bit determines clock source 
  //       Channel 5 polarity bit determines polarity
  //       Channel 5 enable	bit enables  output and 
  //       Channel 5 center aligned enable bit determines  output mode.
  //   0 = Channels 4 and 5 are separate 8-bit PWMs.
  // bit 5, CON23, Concatenate channels 2 and 3
  //   1 = Channels 2 and 3 are concatenated to create one 16-bit channel 
  //       Channel 2 becomes high byte and channel 3 becomes low byte 
  //       Channel 3 output pin is used as output for 16-bit PWM
  //       Channel 3 clock select control-bit determines clock source 
  //       Channel 3 polarity bit determines polarity 
  //       Channel 3 enable	bit enables  output and 
  //       Channel 3 center aligned enable bit determines output mode
  //   0 = Channels 2 and 3 are separate 8-bit PWMs.
  // bit 4, CON01, Concatenate channels 0 and 1
  //   1 = Channels 0 and 1 are concatenated to create one 16-bit channel 
  //       Channel 0 becomes high byte and channel 1 becomes low byte 
  //       Channel 1 output pin is used as output for 16-bit PWM
  //       Channel 1 clock select control-bit determines clock source 
  //       Channel 1 polarity bit determines polarity 
  //       Channel 1 enable	bit enables  output and 
  //       Channel 1 center aligned enable bit determines output mode
  //   0 = Channels 0 and 1 are separate 8-bit PWMs.
  // bit 3, PSWAI, PWM Stops in Wait Mode
  // bit 2, PFRZ, PWM Counters Stop in Freeze Mode
  PWMSCLA = 0;     // SA prescaled A by 512
  // Clock SA = Clock A / (2 * PWMSCLA)
  PWMSCLB = 3;     // SB prescaled B by 6	 = 24MHz/128/6 = 31.25kHz
  // Clock SB = Clock B / (2 * PWMSCLB)
  PWMPER0 = 250;   // period0
  PWMDTY0 = 0;     // duty cycle0, off
  PWMPER1 = 250;   // period1
  PWMDTY1 = 0;     // duty cycle1, off
  PWMPER23 = 31250; // period3
  PWMDTY23 = 0;     // duty cycle23, off
}
//---------Timer_Init------------------
// Activate TCNT at 1.5MHz, counts every 667ns
// inputs: none
// outputs: none
void Timer_Init(void){
  TSCR1 = 0x80;   // Enable TCNT, 24MHz boot mode, 4MHz in run mode
  TSCR2 = 0x04;   // divide by 16 TCNT prescale, TCNT at 667nsec
/* Bottom three bits of TSCR2 (PR2,PR1,PR0) determine TCNT period
    divide  fast(24MHz)       slow (4MHz)
000   1     42ns  TOF  2.73ms  250ns TOF 16.384ms
001   2     84ns  TOF  5.46ms  500ns TOF 32.768ms 
010   4    167ns  TOF  10.9ms    1us TOF 65.536ms   
011   8    333ns  TOF  21.8ms 	 2us TOF 131.072ms
100  16    667ns  TOF  43.7ms 	 4us TOF 262.144ns
101  32   1.33us  TOF  87.4ms		 8us TOF 524.288ms
110  64   2.67us  TOF 174.8ms   16us TOF 1.048576s
111 128   5.33us  TOF 349.5ms   32us TOF 2.097152s */ 

  PACTL = 0;      // timer prescale used for TCNT
}

//---------Timer_Wait------------------
// fixed time delay
// inputs: time to wait in 667ns cycles
// outputs: none
void Timer_Wait(unsigned short time){ short TimeLeft,EndT;
  for(;time>0;time--){
    EndT = TCNT+1500;       // Time (667ns cycles) to wait 
    do{
      TimeLeft = (short)TCNT-EndT;
    }
    while((TimeLeft<0)||(TimeLeft>100));   // wait 
  }
}

unsigned char Duty8;	 // 0 to 250
unsigned short Duty16; // 0 to 10000
unsigned char input,lastInput;
void main(void) {
  PLL_Init();      // running at 24MHz
  Timer_Init();    // enable TCNT at 666ns
  DDRT |= 0x0B;
  DDRAD = 0x00;   // PortAD are inputs, PortT is output
  ATDDIEN = 0xFF; // enable AD digital input drivers  
  EnableInterrupts;
  Duty8 = 125;      // initialize
  Duty16 = 15000;  
  PWM_Init();
  lastInput = PTAD>>6;
  for(;;) {
    input = PTAD>>6;       // Input=0,1,2,or 3
    if((input != lastInput)&&(input != 3)){
      if(input == 1){
        if(Duty8<250){
          Duty8 += 25;     // increase by 10%
        }
        if(Duty16<31250){
          Duty16 += 3125;  // increase by 10%
        }
      } else if(input == 2){
        if(Duty8>0){
          Duty8 -= 25;     // decrease by 10%
        }
        if(Duty16>0){
          Duty16 -= 3125;  // decrease by 10%
        }
      } else{		// both
        Duty8 = 0;   // initialize
        Duty16 = 0;  
      }
      
      PWMDTY0 = Duty8;     // new duty cycle0
      PWMDTY1 = Duty8;     // new duty cycle1
      PWMDTY23 = Duty16;   // duty cycle23
    }
    lastInput = input;
    Timer_Wait(15);   // wait 10ms to debounce switches
  } 
}
