/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Filename: FAT16.c 
 *
 *	Common Purpose:
 *
 *	Functions:	
 *
 *	Notes:	This is a "more" standard interface to the fat16 system
 *		Due to memory space limitations, only one file open at a time
 *		This is specifically for the MMCard interface, but parts could be used for
 *		other media.
 *
 *		Having looked around it seems that it's OK to roll your own FILE interface, 
 *		as everyone is doing it!
 *	
 *		Limitations of this file system as at
 *		8/2/04.
 *		One file open at any one time.
 *		All files in Root Directory only, no sub directories
 *		Media formated with FAT16 using "File Allocation Unit" (Cluster) = 512 bytes = 1 sector
 *		Partitioned Media, only first partition used.
 *		No Format command.
 *		8.3 file names, no long filenames.
 */
#include <hardwareIF.h>
#include "..\MMCard\MMCard.h"
#include "FAT16.h"
#include "FAT16P.h"
#include "FAT16Man.h"
#include "FAT16DirMan.h"
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
struct BootDataType Boot;
struct FileDataType File;
/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function: Setup the File System Data structures for the device
 *
 *	Input:      none
 *
 *	Return:	ErrorCodes; Passed errors and MEDIA_NOT_FORMATED
 *
 *	Notes:	Function to get sector 0, the boot sector, and from this
 *		check that it is a FAT16 file system, if not exit with error
 *		get File system data, 
 *		Check the first sector, if it is a partition table then, 
 *		go find the Boot sector of the active partition
 */
uint16	MountDevice(void)
{
uint16 	ErrorCode = NoError;
uint8 	*Sector;
uint32	SectAddr;
#define PARTITION_0 0x01be
#define PARTITION_1 0x01ce
#define PARTITION_2 0x01de
#define PARTITION_3 0x01ee
#define START_SECT(P) ((uint32)Sector[P + 8] * 0x00000200u)	/*	Calculation of the BootSector sector number	*/
//struct PartitionEntryStruct *PE;

/*	Check if the Device is ready	*/
//  if(DeviceStatus(IsReady))
  if(MMCard_Ready()){
	SectAddr = 0x00ul;
/*	GET	FIRST SECTOR	*/
	Sector = File.LocalSector;
	ErrorCode = MMCardReadSector(0ul, 512u, Sector);
	if(!ErrorCode){
		if (!BootSectorFound()){/*	If the file system is not FAT16, then this would be a MBR. Check all the Partition tables, */
			if((Sector[PARTITION_0] == 0x00) || (Sector[PARTITION_0] == 0x80)){/* and go get the first partition Bootsector	*/
				SectAddr = LittleCharToLong(&Sector[PARTITION_0 + 8]);
				SectAddr *= 0x00000200u;
			}
			else if((Sector[PARTITION_1] == 0x00) || (Sector[PARTITION_1] == 0x80)){
				SectAddr = LittleCharToLong(&Sector[PARTITION_1 + 8]);
				SectAddr *= 0x00000200u;
			}
			else if((Sector[PARTITION_2] == 0x00) || (Sector[PARTITION_2] == 0x80)){ 
				SectAddr = LittleCharToLong(&Sector[PARTITION_2 + 8]);
				SectAddr *= 0x00000200u;
			}
			else if((Sector[PARTITION_3] == 0x00) || (Sector[PARTITION_3] == 0x80)){ 
				SectAddr = LittleCharToLong(&Sector[PARTITION_3 + 8]);
				SectAddr *= 0x00000200u;
			}
			else {
				ErrorCode = MEDIA_NOT_FORMATED;
			}
		}
		ErrorCode = MMCardReadSector(SectAddr, 512u, Sector);
		if(!ErrorCode){
			if (BootSectorFound()){
				LoadBootInfo();
			}
			else {
				ErrorCode = MEDIA_NOT_FORMATED;
			}
		}
	}
	else{
		ErrorCode = DEVICE_NOT_READY;
		File.Flags = 0;
	}
  }
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function: Opens a file, existing or not
 *
 *	Input:  Ptr to a FileName ASCII string
 *		Ptr to File Pointer which is the offset within the file, (not Media)
 *		Modes as follows.. 
 *		R	Read only: uses the *FilePtr to load the sector at which the FilePtr points to
 *		W	Write: uses the *FilePtr to load the sector at which writing is to commence
 *		A	Append: Loads the last sector in the file, and
 *
 *	Return:	The position of the last byte in the FilePtr
 *		The ErrorCodes are a composite of errorcodes at lower levels, 
 *		so that the calling routine can, if it wants, decode the error message and 
 *		so perform user/system responses to the error.		
 *
 *	Notes:	Function which opens a file, and acts differently depending upon the mode,
 *		If the file does not exist, and the mode is write or append then it is created 
 *		and opend in that mode. If the mode is Read, then an error of File not found is returned.
 *
 */
uint16	FileOpen(uint8 *FileName, uint32 *FilePtr, uint16 Mode)	/*	FilePtr is offset within file	*/
{
uint16	ErrorCode = NoError;
uint32 SectAddr;
uint16	ClusterNum;

	if(!FileExists(FileName)){
		ErrorCode = FileNew(FileName);
	}
	/*	Make sure the file is there and get the basic info into the file structure	*/
	if(!ErrorCode){
		ErrorCode = DirectorySearch(&FileName[0]);
	}
	if(ErrorCode == FOUND_ENTRY){
		if(Mode == 'A'){
			*FilePtr = File.FileByteSize;	/*	Put this here incase the FilePtr is not initialized	*/
		}
		else{
			if(*FilePtr > File.FileByteSize){	/*If the data requested extends beyond the file end	*/
				ErrorCode = END_OF_FILE;											/*Exit with error.									*/
			}
		}
	}												
	if(ErrorCode == FOUND_ENTRY){
   		File.StartAddress = Boot.Cluster2Start + 
 	    			((Boot.SectorsPerCluster * ((uint32)File.StartCluster - 2)) * (uint32)Boot.BytesPerSector);
		ErrorCode = FindFileCluster(*FilePtr, &SectAddr, &ClusterNum);		/*	Returns the address of the sector within the cluster	*/
		if(!ErrorCode){
			ErrorCode = MMCardReadSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);
			File.StartAddressOfLoaded = SectAddr;				/*	Update the file data	*/
			File.SectorInCluster = 0;
			File.ClusterLoaded = ClusterNum;
			File.Flags = OPENED_OK;
			File.Mode = Mode;
   			ErrorCode = NoError;
			File.Ptr = *FilePtr;
		}
	}
 	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/
/*	Function which reads data from a file, starting at the FilePtr,
 *	Adding to the ReadStr for LenReadStr number of bytes,
 *	The ReadStr should be at least 1 byte longer than required, 
 *	as it will be nul terminated. 
 *
 *	Returns ErrorCode 
 *		 
 */
/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function: Reads data from a file
 *
 *	Input:	Ptr to File Pointer which is the offset within the file, (not Media)
 *		Ptr to the string where read data will be stored
 *		Length of the string, as it does not really need to be a nul terminated string
 *
 *	Return: Loads the ReadStr with data read from file at FilePtr for LenReadStr bytes. 	
 *		ErrorCodes; END of File, Empty file, file not open,
 *		The ErrorCodes are a composite of errorcodes at lower levels, 
 *		so that the calling routine can, if it wants, decode the error message and 
 *		so perform user/system responses to the error.		
 *
 *	Notes:	Function which reads data from a file, starting at the FilePtr,
 *		Adding to the ReadStr for LenReadStr number of bytes,
 *		The ReadStr should be at least 1 byte longer than required, 
 *		as it will be nul terminated. 
	
 */
uint16	FileRead(uint32 *FilePtr, uint8 *ReadStr, uint16 LenReadStr)
{
uint16	ErrorCode = NoError;
uint32	i = 0;
uint32	SectAddr;
uint16	DataIndex = 0;
uint16	ClusterNum, NextCluster;

	DataIndex = (*FilePtr % Boot.BytesPerSector);						/*	Find indexoffset of the start of data required	*/
	while((i < LenReadStr) && (!ErrorCode)){						/*	If the ByteCount is 0 exit			*/
		ErrorCode = FindFileCluster((*FilePtr + i), &SectAddr, &ClusterNum);		/*	Function follows the FAT trail..		*/
		if(!ErrorCode){									/*	And returns the Address of the sector, that contains the data required	*/
			ErrorCode = MMCardReadSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);
			File.StartAddressOfLoaded = SectAddr;					/*	Update the file data	*/
			if(!ErrorCode){
				while((i < LenReadStr) && (DataIndex < Boot.BytesPerSector)){	/*	Copy the data to the return string for ByteCount	*/
					ReadStr[i++] = File.LocalSector[DataIndex++];		/*	If at end of sector then increment the sectorcount	*/
					}
				if((i<LenReadStr) && (DataIndex >= Boot.BytesPerSector)){	/*	If not finished, reading, but come to the end of the sector	*/
					File.SectorInCluster++;					/*	Point to the next sector in the cluster		*/
					if(File.SectorInCluster >= Boot.SectorsPerCluster){	/*	If at end of cluster. Need to get a new cluster	*/
						File.SectorInCluster = 0;			/*	Point to first sector				*/
						NextCluster = GetNextClusterNumber(File.ClusterLoaded);	/*	Find the next cluster in the chain	*/
						SectAddr = Boot.Cluster2Start +	((Boot.SectorsPerCluster * ((uint32)(NextCluster) - 2)) * (uint32)Boot.BytesPerSector);
						ErrorCode = MMCardReadSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);	/*	Load it	*/
						File.StartAddressOfLoaded = SectAddr;			/*	Advise it has been loaded		*/
					}
				}
			}
		}
	}
 	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function: Writes data to a file
 *
 *	Input:	Ptr to File Pointer which is the offset within the file, (not Media)
 *		Ptr to the string of data that will be written to the open file
 *		Length of the string, as it does not really need to be a nul terminated string
 *
 *	Return: ErrorCodes; FILE_NOT_OPEN, FILE_READ_ONLY, FILE_LEN_EXCEED,
 *		The ErrorCodes are a composite of errorcodes at lower levels, 
 *		so that the calling routine can, if it wants, decode the error message and 
 *		so perform user/system responses to the error.		
 *
 *	Notes:	Function which writes data to a file, starting at the FilePtr.
 *		Overwriting the bytes already at that point,
 *		Extending the file size if required, 
 */
uint16	FileWrite(uint32 *FilePtr, uint8 *WriteStr, uint16	LenWriteStr)
{
uint16	ErrorCode = NoError;
uint16	DataIndex = 0;
uint16	IndexOfWrite = 0;
uint32	SectAddr = 0;
uint16	NextCluster;

	do{/*	Check a few things	*/
		if(!FileOpenOK()){/*	Check if a file is open OK	*/
			ErrorCode = FILE_NOT_OPEN;
			break;
		}
		if(File.Mode == 'R'){/*	Check if the mode is ReadOnly	*/
			ErrorCode = FILE_READ_ONLY;
			break;
		}
		if(*FilePtr > File.FileByteSize){/*	Check if the FilePtr is beyond the end of the file	*/
			ErrorCode = FILE_LEN_EXCEED;
			break;
		}
		if(File.Mode == 'A'){/*	Check if the mode is Append, if so make the FilePtr = FileBytesize	*/
			*FilePtr = File.FileByteSize;
		}

	}while(0);

/*	BUT WE DON'T KNOW IF THE CORRECT SECTOR IS LOADED, SO TO BE MODIFIED, AND WRITEN !!!!!	*/
/*	Assume it is for speed?	*/

   /*	If OK so far, go and write data to buffer, and other sectors as required	*/
	while(!ErrorCode && (DataIndex < LenWriteStr)){
		/*	Setup the pointer for the file sector	*/
		IndexOfWrite  = (*FilePtr % Boot.BytesPerSector);	
		/*	Copy the data into the loaded file sector	*/
		while((IndexOfWrite < Boot.BytesPerSector) && (DataIndex < LenWriteStr)){ /*	Check end of loaded string or sector	*/
			File.LocalSector[IndexOfWrite++] = WriteStr[DataIndex++];
			*FilePtr +=1;
			if(*FilePtr > File.FileByteSize){
				File.FileByteSize = *FilePtr;
			}
		} 
		if(IndexOfWrite >= Boot.BytesPerSector){	/* 	not at end of string, but at end of sector	*/					
			ErrorCode = FileSave();			/*	But first save the data we had, otherwise the next function will wipe it	*/
			if(*FilePtr >= File.FileByteSize){	/*	Then we haven't finished and we've run out of file so get new SECTOR and MAYBE NEW cluster and use that	*/
				File.SectorInCluster++;
				if(File.SectorInCluster >= Boot.SectorsPerCluster){
					ErrorCode = AllocNewCluster();	/*	Follow the cluster trail from the start, and then find a spare and allocate that	*/
					if(ErrorCode){
						ErrorCode += 0x55;
					}
					ErrorCode = LoadFileAt(FilePtr);	/*	Load the new cleared sector ?? This is not really necessary as the sector is 
								*	new and should be blank, what about just clearing the local sector and 
								*	writing to that, making the assumption that it can be simply written to 
								*	the sector pointed to.	No. this is an easy way to get all the correct file data	*/
					if(ErrorCode){
						ErrorCode += 0xaa;
					}
				}
				else{
					SectAddr = File.StartAddressOfLoaded + Boot.BytesPerSector;	/*	Calc new sector address		`*/
					ErrorCode = MMCardReadSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);	/*	Load it	*/
					File.StartAddressOfLoaded = SectAddr;				/*	Advise it has been loaded	*/
				}
			}
			else{	/*	we are not at the end of the file but need to get the next sector	*/
				File.SectorInCluster++;					/*	Point to the next sector in the cluster		*/
				if(File.SectorInCluster >= Boot.SectorsPerCluster){	/*	If at end of cluster. Need to get a new cluster	*/
					File.SectorInCluster = 0;			/*	Point to first sector				*/
					NextCluster = GetNextClusterNumber(File.ClusterLoaded);	/*	Find the next cluster in the chain	*/
					SectAddr = Boot.Cluster2Start +	((Boot.SectorsPerCluster * ((uint32)(NextCluster) - 2)) * (uint32)Boot.BytesPerSector);
					ErrorCode = MMCardReadSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);	/*	Load it	*/
					File.StartAddressOfLoaded = SectAddr;			/*	Advise it has been loaded		*/
				}
				else{	/*	if not at the end of the cluster then get next sector	*/
					SectAddr = File.StartAddressOfLoaded + Boot.BytesPerSector ;				/*	Calc next sector address	*/
					ErrorCode = MMCardReadSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);	/*	Load it				*/
					File.StartAddressOfLoaded = SectAddr;							/*	Advise it has been loaded	*/
				}
			}
			*FilePtr = File.FileByteSize;
			IndexOfWrite = 0;
   		}
	}/*	End while	*/
 	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function: Writes data at the end of a file
 *
 *	Input:	Ptr to the string of data that will be written to the open file
 *		Length of the string, as it does not really need to be a nul terminated string
 *
 *	Return: The ErrorCodes are a composite of errorcodes at lower levels, 
 *		so that the calling routine can, if it wants, decode the error message and 
 *		so perform user/system responses to the error.		
 *
 *	Notes:	Function which writes data to the end of a file, 
 *		Extending the file size if required, 
 *		this is simply a call to the write function with the FilePtr set to the end of the file.. 
 */
uint16	FileAppend(uint8 *WriteStr, uint16 LenWriteStr)
{
uint16	ErrorCode = NoError;

	ErrorCode = FileWrite(&File.FileByteSize, WriteStr, LenWriteStr);
 	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/
/*	Function which closes the open file 
 *	If the mode is Read, no data is saved.
 *	If the mode is write or append then the currently loaded buffer is saved.
 *
 *	Returns ErrorCode 
 *		Media Full, file not open, write error, 
 */ 
uint16	FileClose(void)
{
uint16	ErrorCode = NoError;
uint16	i;

	if((File.Mode == 'W') || (File.Mode == 'A')){
		ErrorCode = FileSave();
		}
	for(i=0;i<8;i++){File.FileName[i]='\0';}
	for(i=0;i<3;i++){File.FileExt[i]='\0';}
	for(i=0;i<512;i++){File.LocalSector[i]='\0';}

	File.Attribute = 0;
	File.StartCluster = 0;
	File.FileByteSize = 0;
	File.SectorInCluster = 0;
	File.StartAddress = 0;
	File.StartAddressOfLoaded = 0;	/*	What is the address of the loaded sector	*/
	File.ClusterLoaded = 0;
	File.Ptr = 0;			
	File.Flags = 0;
	File.Mode = 0;
	
 	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/
/*	Function which deletes the file named, 
 *	frees the directory entry and deallocates the clusters.
 */ 
uint16	FileDelete(uint8 *FileName)
{
uint16	ErrorCode = NoError;

/*	Find the directory entry	*/
	ErrorCode = DirectorySearch(FileName);
/*	Make first character of the filename = 0xe5	*/
	File.FileName[0] = 0xe5;
	ErrorCode = FileSave();	/*	Lazy way to do it	*/	
/*	Deallocate the clusters	*/
	ErrorCode = DeAllocClusters();
 	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/
/*	Function to determine if the file exists
 *	Take the filename as a string and searches the root directory for it
 *	Returns 0 if not found, non zero if found
 */
uint16	FileExists(uint8 *FileName)
{
uint16	Result = 0;
/*	Find the directory entry	*/
	if(DirectorySearch(FileName) == FOUND_ENTRY){
		Result = 1;	/*	It doesn't matter what the reason is */
		}
	
	return	Result;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	A function to start a new file entry
 *	This would need to 
 * 		find the first unused directory entry,
 *		Write the file name,
 *		set atributes,
 *		Find the first available cluster, 
 *		write the start cluster in the directory entry,
 *		write the file size,
 *		and other stuff???... 
 *	Do all the above in both FAT tables.	
 */
uint16	FileNew(uint8 *FileName)
{
uint16	ErrorCode = NoError;
uint16	i=0;
#define EXIT_LOOP 0xfe

	if(!FileExists(FileName)){
	/*	Clear the LocalSector, this can be done by close file	*/
		ErrorCode = FileClose();
	/*	Set the filename and fileextension in the File. structure	
	 *	As well as all the directory data
	 */	
		LoadFileName(FileName);		/*	Loads the File.FileName and File.FileExt	*/
		i=0;
		while(FileName[i]){
			FileName[i] = toupper(FileName[i]);
			i++;
		}
		File.Attribute = 0x20;
		File.ResvNT = 0x00;
		File.Time100th = 0x64;
		File.Time = 0x5c6f;
		File.Date = 0x3044;
		File.LastDate = 0x3045;
		File.ResvFat32 = 0x0000;
		File.LastWriteTime = 0x5c6f;
		File.LastWriteDate = 0x3044;
		File.StartCluster = FindFreeCluster();
	
		if(File.StartCluster == 0xffff){
			ErrorCode = MEDIA_FULL;			/*	Deal with this error later	*/
		}
		File.FileByteSize = 0x00000000u;	/*	End of the all directory information	*/
	
		/*	The next is calculated for easy access later	*/	
		File.StartAddress = Boot.Cluster2Start + 
			((Boot.SectorsPerCluster * ((uint32)File.StartCluster - 2)) * (uint32)Boot.BytesPerSector);
	
	/*	Find a spare directory entry and save this data to the sector where that spare space resides	*/
		ErrorCode =	SaveNewDirEntry();
		if(!ErrorCode){
			ErrorCode = WriteToFAT1and2(File.StartCluster, 0xffff);
		}
	
	/*	Clear the File buffer and write it to the Next Free Cluster	*/
		if(!ErrorCode){
			for(i=0;i<Boot.BytesPerSector;i++){
				File.LocalSector[i] = 0;
			}
			ErrorCode = MMCardWriteSector(File.StartAddress, Boot.BytesPerSector, &File.LocalSector[0]);
		}
	}
	else{
		ErrorCode = FILE_ALREADY_EXISTS;
	}
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	A function to save the currently open file, 
 *	both Data Sector and the directory data 
 *	This uses the file buffer so after 
 *	execution the file buffer no long contains the open sector.
 *	If needed, it will have to be reloaded, with FileOpenAt, etc
 */
uint16	FileSave(void)
{
uint16	ErrorCode = NoError;

	ErrorCode = SaveFileData();
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	A function to return the date string of the currently open file */
uint16	FileDateStr(uint8 *DateString)
{
uint16	ErrorCode = NoError;
struct FAT16Date *FileDate;
uint16	i;
uint8	c[5];

	FileDate = (struct FAT16Date*)File.Date;

	i = (uint16)FileDate->Year;
	itoa(c,i,10);
	strcpy(DateString,c);
	strcpy(DateString,"/");
	i = (uint16)FileDate->Month;
	itoa(c,i,10);
	strcat(DateString,c);
	strcpy(DateString,"/");
	i = (uint16)FileDate->Day;
	itoa(c,i,10);
	strcpy(DateString,c);

	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	A function to return the time string of the currently open file */
uint16	FileTimeStr(uint8 *TimeString)
{
uint16	ErrorCode = NoError;
struct FAT16Time *FileTime;
uint16	i;
uint8	c[5];

	FileTime = (struct FAT16Time*)File.Time;

	i = (uint16)FileTime->Hour;
	itoa(c,i,10);
	strcpy(TimeString,c);
	strcpy(TimeString,":");
	i = (uint16)FileTime->Min;
	itoa(c,i,10);
	strcat(TimeString,c);
	strcpy(TimeString,":");
	i = (uint16)FileTime->Sec;
	itoa(c,i,10);
	strcpy(TimeString,c);

	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	A function to return the time encoded for Fat16format
	accepts the Date in string dd/mm/yy format with leading zeros	*/
uint16	FileTimeBits(uint16 *DateString)
{
uint16	ErrorCode = NoError;
//struct FAT16Time *FileTime;

	
	
	return ErrorCode;
}


/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	A function to return the the value of FilePtr to the first occurance
 *	of the SearchStr in the open file.
 *	Assume file may be manipulated, and all saving has been done
 *	Assume FilePtr points to where the search should start
 */
uint16	FileSearchStr(uint8 *SearchStr, uint32 *FilePtr)
{
uint16	ErrorCode = NoError;
uint16	Sectori = 0;
uint16	Searchi = 0;
uint16	Finding = 0;
uint16	Found = 0;
uint32	StringStart = 0;

/*	CHECK FILEPTR START/COUNTER	*/
	if(*FilePtr >= File.FileByteSize){
		ErrorCode = NOT_WITHIN_FILE;
	}
/*	LOAD SECTOR	*/
	if(!ErrorCode){
		ErrorCode = LoadFileAt(FilePtr);
	}
/*	SEARCH SECTOR	*/
	while(!ErrorCode && !Found){
		if(File.LocalSector[Sectori] == SearchStr[Searchi]){ 	/* We have a first character match, continue*/
			StringStart = *FilePtr;
			Finding = 1; *FilePtr +=1; Sectori++; Searchi++;
			if(*FilePtr >= File.FileByteSize){
				ErrorCode = NOT_WITHIN_FILE;
			}
			else{
				if(Sectori >= Boot.BytesPerSector){
					Sectori = 0;
					ErrorCode = LoadFileAt(FilePtr);
				}
			}
			while(!ErrorCode && Finding){
				while(SearchStr[Searchi] && Finding){	/* Assumed null terminated string	*/
					if(File.LocalSector[Sectori] == SearchStr[Searchi]){
						Finding = 1; *FilePtr +=1; Sectori++; Searchi++;
						if(*FilePtr >= File.FileByteSize){
							ErrorCode = NOT_WITHIN_FILE;
						}
						else{
							if(Sectori >= Boot.BytesPerSector){
								Sectori = 0;
								ErrorCode = LoadFileAt(FilePtr);
							}
						}
					}
					else{
						Finding = 0;
					}
				}
				if(SearchStr[Searchi] == 0){
					Found = 1;
					Finding = 0;
					*FilePtr = StringStart;
				}
			}
		}
		else{
			Finding = 0; *FilePtr+=1; Sectori++; Searchi = 0;
			if(*FilePtr >= File.FileByteSize){
				ErrorCode = NOT_WITHIN_FILE;
			}
			else{
				if(Sectori >= Boot.BytesPerSector){
					Sectori = 0;
					ErrorCode = LoadFileAt(FilePtr);
				}
			}
		}
	}
	return ErrorCode;
}



