/*	Functions to manage the Directories		*/

#include <hardwareIF.h>
#include "..\MMCard\MMCard.h"
#include "FAT16.h"
#include "FAT16P.h"
#include "FAT16Man.h"
#include "FAT16DirMan.h"
//#include <DeviceManager.h>
#include <string.h>
 
extern struct BootDataType Boot;
extern struct FileDataType File;


/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to find a filename within a directory
 *	Starts in the root directory and keeps looking
 *	for the specified filename until end or found
 *	Assumes the media/is mounted OK
 *	First approximation will only look in Root DIR
 */	
uint16		DirectorySearch(uint8 *FileName)
{
/*	Get the root directory and start search there
 *	then if not found, look for any sub directories
 *	and look there as well.
 *	If found the File data structure attribute section is filled.
 */
	
/*	Root dir search. 
 *	This different because there is no FAT table allocation for the root
 *	it is sequensial only.	
 */
uint16	ErrorCode = NoError;
uint8	FileAtrib[32];
uint16	DirSectCount = 0;
uint32	SectAddr;

	/*	SEARCH ROOT DIRECTORY	*/
	do{
		SectAddr = Boot.RootDirStart + ((uint32)DirSectCount * (uint32)Boot.BytesPerSector);
		if(SectAddr >= Boot.RootDirStart + Boot.RootDirSize){ 
			ErrorCode = NOT_FOUND_IN_ROOT;
			}
		if(!ErrorCode){
			ErrorCode = MMCardReadSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);
			if(!ErrorCode){
				ErrorCode = SearchDirSecForFileName(FileName, &FileAtrib[0]);
				switch(ErrorCode){
					case FOUND_ENTRY:		LoadFileData(&FileAtrib[0]); break;
					case NO_FURTHER_ENTRIES : ErrorCode = NOT_FOUND_IN_ROOT; break;
					case NOT_IN_THIS_SECTOR : 	DirSectCount++; ErrorCode = NoError; break;
					default : ;break;
					}
				}
			else{
				ErrorCode = NOT_FOUND_IN_ROOT;/*It is really read error but...*/
				}
			}
		}while ((ErrorCode != NOT_FOUND_IN_ROOT) && (ErrorCode != FOUND_ENTRY));
	if(ErrorCode == NOT_FOUND_IN_ROOT){
		/*	SEARCH IN OTHER DIRECTORIES	*/
		ErrorCode = FILE_NOT_FOUND;
		//SearchDirSecForSubDir(SkipSub, &FileAtrib[0])
		}
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to find filename entry in the sector of directory
 *	This searches the File.LocalSector for the string filename
 *	Returns 
 *		FOUND_ENTRY, and fill the FileAttrib String
 *		NOT_IN_THIS_SECTOR
 *		NO_FURTHER_ENTRIES
 */
uint16	SearchDirSecForFileName(uint8 *FileName, uint8	*FileAttrib)
{
uint16	ErrorCode = NoError;
uint16	DirEntryCount = 0;
uint16	i = 0;
uint16	x = 0;

	do{
		if(FileName[i] == File.LocalSector[(DirEntryCount * 32) + x]){	/*	check the byte and compare If the characters are the same,*/
			i++;							/*	Increment the 2 pointers				*/
			x++;							/*	If the next character of the filename			*/
			if(FileName[i] == '.'){					/*	is a . then skip over, and also point			*/
				i++; x = 8;					/*	 to the first char of the File EXT			*/
			}
			if((FileName[i] == '\0') || (x>11)){			/*	If the next char of the filename is a nul, or x>11	*/
				ErrorCode = FOUND_ENTRY;				/*	Then this is finished, as the whole string has been matched*/
				for(i=0; i<=DIR_ENTRY_SIZE; i++){			/*	So fill the return string with the 			*/
					FileAttrib[i] = File.LocalSector[(DirEntryCount * 32)+i];/*	Directory entry data			*/
				}
			}
		}	
		else{															
			if(File.LocalSector[(DirEntryCount * 32)] == 0x00){	/*	Check if the first byte of the entry 		*/
				ErrorCode = NO_FURTHER_ENTRIES;			/*	is the no-more-entries-marker			*/
			}
			DirEntryCount++;					/*	If not found, then point to the next		*/
			i = 0;							/*	Directory entry, and clear the indexes		*/
			x = 0;							/*	Check if the entry count has come to		*/
			if(DirEntryCount >= (Boot.BytesPerSector/DIR_ENTRY_SIZE)){/*	the end, if so then not fiound here		*/
				ErrorCode = NOT_IN_THIS_SECTOR;
			}
		}
	}while (!ErrorCode);							/*	Do all of this while there is no result		*/
		
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to find valid file entries in the directories
 *	and return these in a string.
 *	Starting at EntryStart, and for DirEntryCount, returning the FileName and Attribute in DirConStr 
 *
 *	This cannot be called with an open file, as the system uses the same buffer space.
 *	Returns ErrorCode
 * 		FileOpen NoEntries
 *	Fills a char array with Filename.ext<space>Filetype (either file of dir.)
 *		array is assumed to allow a max for (DirEntryNum x 12)bytes
 */
uint16	GetDirContents(uint16 DirEntryStart, uint16 *EntryCount, uint8 *DirStrArray, uint16 SizeOfArray)
{
#define FILE_NAME_SIZE	12
#define ATTRIB_SIZE		1		/*	This could be increased, as more than one attribute can apply at a time	*/
#define INFO_ENTRY_SIZE 	(FILE_NAME_SIZE + ATTRIB_SIZE)
uint16  DirEntryNum;
uint16	ErrorCode = 0;
uint32	SectAddr;
uint8	Attribute;
uint16	SkipCount;
uint16	SecSkipCount;
uint16	DirSectCount = 0;
uint8	NameStr[12];
uint16	i;
uint16	StringEntry;
uint16	FoundCount;

	DirEntryNum = SizeOfArray / INFO_ENTRY_SIZE;
	SkipCount = DirEntryStart;
	FoundCount = 0;
	
	/*	SEARCH ROOT DIRECTORY	*/
	do{
		SectAddr = Boot.RootDirStart + ((uint32)DirSectCount * (uint32)Boot.BytesPerSector);		/*	Calculate the Sector Address				*/
		if(SectAddr >= Boot.RootDirStart + Boot.RootDirSize)						/*	If the Start address is over the directory size */
			{ ErrorCode = NOT_FOUND_IN_ROOT;}											/*	Then we have finished and the is not more to be found	*/
		if(!ErrorCode){																/*	Else get the sector							*/
			ErrorCode = MMCardReadSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);
			}
		SecSkipCount = 0; 													/*	when a new sector is load SecSkipCount should be reset	*/
		while(!ErrorCode){
			ErrorCode = SearchDirSecForValidEntry(SecSkipCount, NameStr, &Attribute);	/*	Go search for the nth vaild entry			*/
			if(ErrorCode == FOUND_ENTRY){											
				ErrorCode = NoError;										/*	Clear the ErrorCode so that we don't exit	*/
				FoundCount++;												/* 	if an entry has been found					*/
				if(FoundCount >= DirEntryStart){							/*	Check if the entry found is wanted		*/	
					StringEntry = ((SkipCount - DirEntryStart)* INFO_ENTRY_SIZE); 	/*	The string is loaded at the place in the array	*/
					if((StringEntry + INFO_ENTRY_SIZE) >= SizeOfArray){
						ErrorCode = LIST_FULL;
						}
					else{
    					for(i=0; i<FILE_NAME_SIZE; i++){							/*	Fill the array with the string bytes 		*/
    						DirStrArray[StringEntry + i] = NameStr[i];
    						}						
    					DirStrArray[StringEntry + FILE_NAME_SIZE] = DecodeAttrib(Attribute);	/*	and the attribute							*/
  						SkipCount++;												/*	Increment the count of the required entry	*/
						} /*End Else*/
					}/*End if(FoundCount...*/									/*	Set the ErrorCode so we exit with this code	*/
				SecSkipCount++;												/*	Increment the count of the required entry	*/
				}
			if(ErrorCode == NO_FURTHER_ENTRIES){
				ErrorCode = NOT_FOUND_IN_ROOT;									/*	if the code 0x00 was found					*/
				}
			}
		if(ErrorCode == NOT_IN_THIS_SECTOR){
			DirSectCount++;
			ErrorCode = NoError;											/*	if no valid entry found in this sector		*/
			}
		}while (!ErrorCode);
	if(ErrorCode == NOT_FOUND_IN_ROOT)
		/*	SEARCH IN OTHER DIRECTORIES	*/
		{
		//SearchDirSecForSubDir(SkipSub, &FileAtrib[0])
		}
	DirStrArray[StringEntry + INFO_ENTRY_SIZE] = 0;	/*	Null terminate the array so it can be treated as a string	*/
	if(FoundCount < DirEntryStart){
		*EntryCount = 0;
		}
	else{
		*EntryCount = FoundCount - DirEntryStart;		/*	This should return the number of entries returned	*/
		}
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to load the first part of the file structure with the directory entry
 */
void	LoadFileData(uint8	*FileAtrib)
{
uint16	i =0;

	while(FileAtrib[i] != ' ')
		{	File.FileName[i] = FileAtrib[i++];}
	while(i<8)
		{	File.FileName[i++] = ' ';}
		
	while(i<11)
		{	File.FileExt[i-8] = FileAtrib[i++];}
	
	File.Attribute = FileAtrib[11];
	File.ResvNT = FileAtrib[12];
	File.Time100th = FileAtrib[13];
	File.Time = FileAtrib[14] | (FileAtrib[15]<<8);
	File.Date = FileAtrib[16] | (FileAtrib[17]<<8);
	File.LastDate = FileAtrib[18] | (FileAtrib[19]<<8);
	File.ResvFat32 = FileAtrib[20] | (FileAtrib[21]<<8);
	File.LastWriteTime = FileAtrib[22] | (FileAtrib[23]<<8);
	File.LastWriteDate = FileAtrib[24] | (FileAtrib[25]<<8);
	File.StartCluster = FileAtrib[26] | (FileAtrib[27]<<8);
	File.FileByteSize = FileAtrib[28] + (FileAtrib[29] * 0x00000100u) + (FileAtrib[30] * 0x00010000u) + (FileAtrib[31] * 0x01000000u);			/*	End of Dir entry	*/
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to find valid entries in the directory sector currently loaded
 *	Counts the entry number found, and returns the next entry only.
 *	Returns the Name of the entry, both the FileName and the FileExt, 
 *	as well as the attribute byte.
 *	ErrorCodes
 *		FOUND_ENTRY
 *		NOT_IN_THIS_SECTOR
 *		NO_FURTHER_ENTRIES
 *
 *
 */
uint16	SearchDirSecForValidEntry(uint16 EntryCount, uint8 *NameStr, uint8 *Attribute)
{
uint16	ErrorCode = NoError;
uint16	DirEntryCount = 0;
uint16	EntryFound = 0;
uint16	i = 0;
//uint16	x = 0;
uint8	c;

	do{
		switch(File.LocalSector[(DirEntryCount * 32)])					/*	Check the first character of the entry	*/
			{															/*	Because this holds the key to the validity of the entry	*/
			case 0x05: case 0xE5:	DirEntryCount++; break;				/*	These are the Entry Free flags, next entry	*/
			case 0x00:	ErrorCode = NO_FURTHER_ENTRIES; break;			/*	This is the last of the valid entries, look no further	*/
			default :													/*	This is the valid entry case 			*/
				{														/*	so check if this is a "Skip" or not		*/
				c = File.LocalSector[(DirEntryCount * DIR_ENTRY_SIZE)+ FAT16_ATTRIBUTE];
				if(!(c & ATT_VOL_NAME))									/*	to ignore Volume lable, longfile name entries etc...*/
					{
					EntryFound++;
	 				if(EntryFound >EntryCount)							/*	If not then get the name and Atribute	*/
						{
						for(i=0; i<=12;i++){
							NameStr[i] = File.LocalSector[(DirEntryCount * DIR_ENTRY_SIZE)+ i];
							}
						*Attribute =  File.LocalSector[(DirEntryCount * DIR_ENTRY_SIZE)+ FAT16_ATTRIBUTE];
						ErrorCode = FOUND_ENTRY;						/*	Flag this valid entry so we exit the loop	*/
						}
					}
				break;
				}
			}
/*	Some redundent code here but gota move on...!	*/
			if(DirEntryCount > 0x0f){
				ErrorCode = NOT_IN_THIS_SECTOR;
				}
			if(DirEntryCount == 0x0f){
				if(EntryCount > EntryFound){
					ErrorCode = NOT_IN_THIS_SECTOR;
					}
				}

			if((DirEntryCount >= (Boot.BytesPerSector/DIR_ENTRY_SIZE)) && (EntryCount > EntryFound)){		/*	check if the end, if so then not fiound here*/
				ErrorCode = NOT_IN_THIS_SECTOR;
				}

			DirEntryCount++;

		}while (!ErrorCode);											/*	Do all of this while there is no result	*/
		
	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to Decode the attribute to a single ASCII char for human interface.
 */
uint8	DecodeAttrib(uint8 Attribute)
{
uint8	AttChar;
	if(Attribute & ATT_VOL_NAME)
		{AttChar = 'V';}
	else if(Attribute & ATT_DIR)
		{AttChar = 'D';}
	else if(Attribute & ATT_READ_ONLY)
		{AttChar = 'R';}
	else 	if(Attribute & ATT_HIDDEN)
		{AttChar = 'H';}
	else 	if(Attribute & ATT_ARCHIVE)
		{AttChar = 'A';}
	else 	if(Attribute & ATT_SYSTEM)
		{AttChar = 'S';}
	return AttChar;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to Search the directory for a free directory entry
 *	This also loads the sector containing the free entry.
 *	And saves the current File structure data to that free entry which is now not free!
 */
uint16	SaveNewDirEntry(void)
{
uint16	ErrorCode = NoError;
uint16	DirEntryCount = 0;
uint16	i = 0;
uint32	SectAddr;
uint16	DirSectCount = 0;
uint16	EntryFound = 0;

/*	Search the Root directory for a "Entry Free symbol (0x05 or 0xe5)	
 *	once found the sector is obviously loaded so, return the index of the entry
 *	And then proceed to load the Directory data	
 */

	do{
		SectAddr = Boot.RootDirStart + ((uint32)DirSectCount * (uint32)Boot.BytesPerSector);		/*	Calculate the Sector Address				*/
		if(!ErrorCode){																/*	Else get the sector							*/
			DirEntryCount = 0;
			ErrorCode = MMCardReadSector(SectAddr, Boot.BytesPerSector, &File.LocalSector[0]);
			if(!ErrorCode){
				do{
					i = File.LocalSector[(DirEntryCount * 32)];
					if(( i == 0x05)	|| (i == 0xE5) || (i == 0x00))	{
						EntryFound = FOUND_ENTRY;										/*	A new free entry has been found	*/
						}
					else{
						DirEntryCount++;
						}
					}while(((DirEntryCount * 32) < Boot.BytesPerSector) && (EntryFound != FOUND_ENTRY));
				}
			DirSectCount++;
			}
			else{
				break;
			}
		}while((DirSectCount < (Boot.RootDirSize/Boot.BytesPerSector)) && (EntryFound != FOUND_ENTRY) && !ErrorCode);

	if((!ErrorCode) && (EntryFound == FOUND_ENTRY)){
		StoreDirEntry(DirEntryCount);
		/*	now write that data to the card	*/
		ErrorCode = MMCardWriteSector(SectAddr, Boot.BytesPerSector, File.LocalSector);	
		}

	return ErrorCode;
}

/*<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>*/ 
/*	Function to Store the file dir data to the current loaded sector	
 *	At DirEntryCount
 */
void	StoreDirEntry(uint16 DirEntryCount)
{
uint16	i;
uint16	DirIndex;

	DirIndex = DirEntryCount * 32;
	for(i=0; i<8; i++){
		File.LocalSector[DirIndex + i] = File.FileName[i];
		}
	for(i=8; i<11; i++){
   		File.LocalSector[DirIndex + i] = File.FileExt[i-8];
   		}
   	File.LocalSector[DirIndex + 11] = File.Attribute;
   	File.LocalSector[DirIndex + 12] = File.ResvNT;
   	File.LocalSector[DirIndex + 13] = File.Time100th;
   	File.LocalSector[DirIndex + 14] = (uint8)File.Time;
	File.LocalSector[DirIndex + 15] = (uint8)(File.Time>>8);

   	File.LocalSector[DirIndex + 16] = (uint8)File.Date;
   	File.LocalSector[DirIndex + 17] = (uint8)(File.Date>>8); 

   	File.LocalSector[DirIndex + 18] = (uint8)File.LastDate;
   	File.LocalSector[DirIndex + 19] = (uint8)(File.LastDate>>8); 

   	File.LocalSector[DirIndex + 20] = (uint8)File.ResvFat32;
   	File.LocalSector[DirIndex + 21] = (uint8)(File.ResvFat32>>8); 

   	File.LocalSector[DirIndex + 22] = (uint8)File.LastWriteTime;
   	File.LocalSector[DirIndex + 23] = (uint8)(File.LastWriteTime>>8); 

   	File.LocalSector[DirIndex + 24] = (uint8)File.LastWriteDate;
   	File.LocalSector[DirIndex + 25] = (uint8)(File.LastWriteDate>>8); 

   	File.LocalSector[DirIndex + 26] = (uint8)File.StartCluster;
   	File.LocalSector[DirIndex + 27] = (uint8)(File.StartCluster>>8); 

   	File.LocalSector[DirIndex + 28] = (uint8)File.FileByteSize;
   	File.LocalSector[DirIndex + 29] = (uint8)(File.FileByteSize / 0x00000100u); 
   	File.LocalSector[DirIndex + 30] = (uint8)(File.FileByteSize / 0x00010000u);
   	File.LocalSector[DirIndex + 31] = (uint8)(File.FileByteSize / 0x10000000u); 

}

