/*	This is the code file for the FAST file ops routines	*/

/*	The general idea is to provide some fast file transfer operations.
 *	As the standard more verstile routines are slow due to the lack 
 *	of a FAT copy in memory. As a result the fat table must be 
 *	got/modified/written as each new cluster is needed, 
 *	then the file open again etc..... 
 *	In a situation where fast data storage and retrieval is required, 
 *	there are probably a few reasonable assumption to be made.
 *	In the case of write, the file size could be assumed, from passed data.
 *	And the clusters be allocated for the whole file size, at creation. 
 *	Then the write routine can assume that the file is a contigeous 
 *	mass of sectors on the storage device. 
 *	And so no FAT table management need be done. If the data ends 
 *	before the max file size is reached then the FAT table needs 
 *	to be cleaned up, by deallocating the previously allocated clusters, 
 *	and saveing the filesize correctly.
 *	For reading a large file out quickly, also assume that the file is 
 *	contigeous, this could be checked and the start and end clusters be set.
 *	Again the read routine can continually get successive sectors until the end 
 /*
 
 
/*	May need to create a file with FastFile structure loaded into it 
 *	in case the FastFile opened needs to be accessed over a number 
 *	of different sessions.	 could call the file, FileName.FFS	
 *	This would require a controlled shut down of the file/system 
 *	that is not FileCloseFastFile().
 *	As the file data needs to be written so that the FastFile.IntFileSize is known
 */
#include <project.h>

#include "..\MMCard\MMCard.h"

#include "FAT16FastFileOps.h"
#include "FAT16.h"
#include "FAT16P.h"
#include "FAT16Man.h"
#include "FAT16DirMan.h"
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
 
extern struct BootDataType Boot;
extern struct FileDataType File;


 struct FastFileDataStructure {
	 uint8	FileName[8];
	 uint8	FileExt[3];
	 uint16	StartCluster;
	 uint16	EndCluster;
	 uint32	IntFileSize;
	 uint16	SectorIndex;
	 uint32 CurrentSectorAddr;
 }FastFile;


/*	This routine sets up the file for fast writes.
 *	This involves, checking if the filename exists, if so error, it must be a new file
 *	Opens a new file, as per normal, but then checks if the StartCluster is the 
 *	beginning of the whole file size in contigeous clusters. (using AllocateFileClusters)
 *	If not the File.StartCluster is reallocated
 */
 uint16	FileOpenFastWrite(uint8 *FileName, uint32 *MaxFileSize, uint16 Mode)
 {
uint16	ErrorCode = NoError;
uint32	*FilePtr = 0;

	if(Mode == 'W'){
		if(FileExists(FileName)){/*	if it exists then this is an error	*/
			ErrorCode = FF_FILE_EXISTS;
		}
		else{
			ErrorCode = FileOpen(FileName, FilePtr, 'W');				/* else create the file, 	*/
			if(!ErrorCode){								/* and allocate the MaxFileSize number of clusters	*/
				ErrorCode = AllocateFileClusters(MaxFileSize, &File.StartCluster);
				if(!ErrorCode){
					ErrorCode = StoreFileDirEntry();	/* Save the file data to the directory entry as it has changed to the pre allocated file */
					/* And then reload the first sector of the file ready to write	*/
					ErrorCode = MMCardReadSector(File.StartAddress, Boot.BytesPerSector, &File.LocalSector[0]);
				}
			}
		}
	}
	else{
		ErrorCode = CANNOT_CREATE_FASTFILE;
	}
	return ErrorCode;
 }


/*	This routine, finds a contigeous bunch of clusters, 
 *	starting at the FileStartCluster if posible, to give the MaxFileSize.
 *	If not possible it continues on to find the bunch, 
 *	and reallocates the File.StartCluster of the File...
 *	This is why an append to a file will not work in FAST mode
 *	Once found, it then goes and allocates them in the FAT, 
 *	updating the File.Structure and FastFile.Structure 
 */ 
uint16	AllocateFileClusters(uint32 *MaxFileSize, uint16 *StartCluster)
{
extern struct File;
uint16	ErrorCode = NoError;
uint16	ClusterMax;
uint16	i;
uint16	FatSectorCount;
uint16	ClusterNum;
uint16	BestStartCluster;
uint16	ThisClusterNum;
uint16	ClusterCount;
uint32	Fat1EntryAddress;

	/*	Calc the number of Clusters are needed	*/
	ClusterMax = (uint16)(*MaxFileSize/(Boot.BytesPerSector * Boot.SectorsPerCluster)) +1; 		/* The +1 is for the bytes greater then the rounded number of the / */  
	/*	Seed the search with StartCluster	*/
	Fat1EntryAddress = ((Boot.HiddenSectors + 1) * Boot.BytesPerSector) + (*StartCluster * 2);	 /* finding the FAT entry sector address of the StartCluster	*/
	/*	Initialize the search pointer to the StartCluster entry	*/	
	i = Fat1EntryAddress % Boot.BytesPerSector;
	/*	Initialize the FatSectorCount with the sector of the StartCluster	*/
	FatSectorCount = (Fat1EntryAddress/Boot.BytesPerSector);
	/*	Do the first check	*/
	Fat1EntryAddress = (Fat1EntryAddress / Boot.BytesPerSector) * Boot.BytesPerSector;	/* Make it point to the start of the sector	*/
	ErrorCode = MMCardReadSector(Fat1EntryAddress, Boot.BytesPerSector, &File.LocalSector[0]);	/* Get the fat table sector				*/
	if(!ErrorCode){
		ClusterNum =  LittleCharToInt(&File.LocalSector[i]);			/* Cluster numbers are in little endian format.			*/
		if(ClusterNum != 0xffff){						/* This entry should be the last, if not this is an error	*/
			ErrorCode = FILE_ERROR;						/* as the file should be new and empty				*/
		}
		else{
			BestStartCluster = *StartCluster;				/* All being well this is the best cluster to start the bunch	*/
			i++; i++;							/* FAT table entries are 2 bytes	*/
		}									/* Now we have a start entry and sector loaded to start our search*/
	/*	Start the cluster search loop	*/
		while((ClusterCount < ClusterMax) && !ErrorCode){
			ThisClusterNum = LittleCharToInt(&File.LocalSector[i]);
			if(ThisClusterNum !=0x0000){					/* Check the rest of the sector is all 0x000	*/
				ClusterCount = 0;					/* if not, then start counting clusters again, 	*/
				BestStartCluster = 0;					/* and forget the best start cluster 	*/
				i+=2;							/* Point to the next entry		*/
			}
			else{
				if(BestStartCluster == 0){				/* Check if the Best Start cluster has been noted	*/
					BestStartCluster = ThisClusterNum;		/* If not then this is it		*/
				}
				i+=2;							/* Point to the next entry		*/
				ClusterCount++;						/* Count the free contigeous clusters	*/
				if(i>=Boot.BytesPerSector){				/* If exceeded the sector size		*/
					Fat1EntryAddress += Boot.BytesPerSector;	/* Point to the next sector and ..	*/
					if((Fat1EntryAddress / Boot.BytesPerSector) >= (Boot.SectorPerFat + Boot.HiddenSectors + 1)){
						ErrorCode = CANNOT_CREATE_FASTFILE;	/* We have come to the end of the FAT 	*/
					}						/* and not found them all 		*/
					else{						/* Else get the fat table sector	*/
						ErrorCode = MMCardReadSector(Fat1EntryAddress, Boot.BytesPerSector, &File.LocalSector[0]);	
						i = 0;					/* Point to the first entry		*/
					}
				}
			}
		}
	/*	Allocate the clusters		*/
		if(!ErrorCode){	/*	Then the full bunch of clusters was found, now start at the beginning and allocate them	*/
			if(File.StartCluster != BestStartCluster){	
				ErrorCode = WriteToFAT1and2(File.StartCluster, 0x0000);	/* Go and deallocated the original file.startcluster	*/
				File.StartCluster = BestStartCluster;			/* And reallocate the best start cluster to the file	*/
			}
			ErrorCode = FastFileAllocateClusters(ClusterMax, BestStartCluster);
			if(!ErrorCode){
				File.FileByteSize = *MaxFileSize;
				memcpy(FastFile.FileName, File.FileName, 8);
				memcpy(FastFile.FileExt, File.FileExt, 3);
				FastFile.StartCluster = File.StartCluster;
				FastFile.IntFileSize = 0;
				FastFile.CurrentSectorAddr = File.StartAddress;
				FastFile.SectorIndex = 0;
			}
		}
	}
	return ErrorCode;
}


/*	This function gets the fat table and starting at the the StartCluster 
 *	entry, writes the link list to both FAT1 and FAT2
 *	Could use the WriteToFAT1and2 function, 
 *	but since we are allocating masses of clusters, 
 *	faster to do the whole sector of FAT at a time
 */
uint16	FastFileAllocateClusters(uint16	Clusters, uint16 StartCluster)
{
uint16	ErrorCode = NoError;
uint16	ClusterNum;
uint16	ClusterCount;
uint32	EntryAddr;
uint16	EntryIndex;
uint32	SectorAddr;
uint32	Fat2SectorAddr;

	ClusterNum = StartCluster;
	ClusterCount = 0;
/*	Find the sector with the FAT entry required to be changed	*/
/*	The fat table starts at ((Boot.HiddenSectors + 1) * Boot.BytesPerSector)	*/
	EntryAddr = ((uint32)(Boot.HiddenSectors + 1) * (uint32)Boot.BytesPerSector) + (uint32)(ClusterNum * 2);
	EntryIndex = EntryAddr % Boot.BytesPerSector;
	SectorAddr = EntryAddr - EntryIndex;
/*	Load it and write the linked list to the File.LocalSector, then save the sector	*/
	ErrorCode = MMCardReadSector(SectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);	/*	Get the fat table sector*/
	while(!ErrorCode && (ClusterCount < Clusters)){
		while((ClusterCount < Clusters) && (EntryIndex < Boot.BytesPerSector)){
			ClusterNum++;							/* Referrence the next cluster to be allocated	*/
			ClusterCount++;							/* Keep track of the number of clusters allocated */
			File.LocalSector[EntryIndex] = (uint8)(ClusterNum);		/* Cluster numbers are in little endian format.*/
			File.LocalSector[EntryIndex +1] = (uint8)(ClusterNum >> 8);
			EntryIndex +=2;
		}

		if(ClusterCount == Clusters){						/* If at the end of the mass allocate	*/
			File.LocalSector[EntryIndex-2] = 0xff;				/* Mark the last cluster as 		*/
			File.LocalSector[EntryIndex-1] = 0xff;				/* the last cluster in the chain	*/
			FastFile.EndCluster = ClusterNum;
		}

		if(!ErrorCode){								/* Save the local sector to the FAT table */
			ErrorCode = MMCardWriteSector(SectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);
		}
		
		if(!ErrorCode){								/* Assume a second copy of the FAT exists. */
			Fat2SectorAddr = SectorAddr + ((uint32)Boot.SectorPerFat * (uint32)Boot.BytesPerSector);/*And save the same sector to the second copy */
			ErrorCode = MMCardWriteSector(Fat2SectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);
		}
		
		if(!ErrorCode && (ClusterCount < Clusters)){				/* Need next sector			*/
			SectorAddr += Boot.BytesPerSector;				/* Get the fat table sector		*/
			ErrorCode = MMCardReadSector(SectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);
			EntryIndex = 0;
		}
	}
	return ErrorCode;
}


/*	This routine will close the file, but first, checks the File.FileByteSize to the FastFile.INtFileSize
 *	If these are not the same, then the file is being closed before it's full
 *	So we need to do a FAT clean up of the allocated clusters.
 *	This is done, by finding the cluster that contains the last data of the FastFile.IntFileSize
 *	Get the cluster pointed to by it, and set it, it being the Last Cluster, to 0xFFFF, 
 *	then use the number of the cluster to deallocate the rest of the chain, by writing 0x0000 to each of them
 *	Until the last cluster, i.e. the one loaded with 0xFFFF is found.
 *	Also must write the correct file size.
 */
uint16	FileCloseFastWrite(void)
{
uint16	ErrorCode = NoError;

	return ErrorCode;
}


/*	This routine maintains the File.LocalSector, 
 *	and writes it to the Mass storage device when full, 
 *	using the next sector address, until, the file is closed
 *	or the FastFile.EndCluster is full
 */
uint16	FileWriteFastWrite(uint8 *DataArray, uint16 DataSize)
{
uint16	ErrorCode = NoError;
uint16	DataIndex = 0;
//uint16	i;

/*	Put DataArray into the File.LocalSector	*/
	while((DataIndex < DataSize) && (FastFile.IntFileSize < File.FileByteSize)){
		File.LocalSector[FastFile.SectorIndex++] = DataArray[DataIndex++];
		FastFile.IntFileSize +=1;
		if(FastFile.SectorIndex >= Boot.BytesPerSector){	/* Store sector	*/
			ErrorCode = MMCardWriteSector(FastFile.CurrentSectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);
			FastFile.CurrentSectorAddr += Boot.BytesPerSector;
			FastFile.SectorIndex = 0;
			/* Clear sector data to 0x00 , comment this out for extra speed? */
//			for(i=0; i<Boot.BytesPerSector; i++){
//				File.LocalSector[i] = 0x00;
//			}
		}
	}
	if(FastFile.IntFileSize >= File.FileByteSize){
		ErrorCode = MMCardWriteSector(FastFile.CurrentSectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);
		ErrorCode = FILE_FULL;
	}
	return ErrorCode;
 
}


/*	This routine, saves the FastFile data into a file 
 *	for when it is needed again.
 *	This assumes an open FastFile file.
 */
uint16	FileEndSessionFastFile(void)
{
uint16	ErrorCode = NoError;

	return ErrorCode;
}


/*	This is a simple test designed to remove most "test" timing and just test the write speed
 *	This involves writting directly to the local sector, so this in fact becomes a "write" function.
 *	This would be the basis of a dedicated fast data write routine, where the application 
 *	specific data is writen directly to the File.LocalSector 
 */
uint16	FastFileFlatOutTest(void)
{
uint16	ErrorCode = NoError;
uint16	DataIndex = 0;
/*	Just store a static character as this is fastest	*/
	while(FastFile.IntFileSize < File.FileByteSize){
		File.LocalSector[FastFile.SectorIndex++] = 'x';
		FastFile.IntFileSize +=1;
		if(FastFile.SectorIndex >= Boot.BytesPerSector){	/* Store sector	*/
			ErrorCode = MMCardWriteSector(FastFile.CurrentSectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);
			FastFile.CurrentSectorAddr += Boot.BytesPerSector;
			FastFile.SectorIndex = 0;
		}
	}
	if(FastFile.IntFileSize >= File.FileByteSize){
		ErrorCode = MMCardWriteSector(FastFile.CurrentSectorAddr, Boot.BytesPerSector, &File.LocalSector[0]);
		ErrorCode = FILE_FULL;
	}
	return ErrorCode;
 }



