// **************tester.c*******************
// This program tests the functionality of the   
// Technological Arts MC9S12C32 board
// This program tests the module/docking combination  

// operates only in RUN mode, not BOOT mode because it uses SCI
// 1) plug 9S12C32 into testing protoboard/ZIF socket
// 2) connect RS232 cable to docking module
// 3) place the 9S12C32 in boot mode
// 4) apply power to the docking module
// 5) hit reset on the docking module
// 6) run Metrowerks and download this program (Project->Debug)
// 7) quit Metrowerks and start a terminal program e.g., Hyperterm
//     set COM port to match the cable, and baud rate 115200 bits/sec
//     8 bit data, no parity, no hardware flow control
// 8) place the 9S12C32 in RUN mode
// 9) hit reset on the docking module

// Hardware:
// PT0-PT4-PE0 via 1k registers
// PT1-PT5-PE1 via 1k registers
// PT2-PT6     via 1k register
// PT3-PT7     via 1k register
// PAD0-PAD1-PAD2-PAD3-PAD4-PAD5 via 10k resistors
// PAD6-PAD7 via 1k resistor
// PM4(MOSI)-PM2(MISO)  via 1k register
// PM3(SS)-PM1			    via 1k register
// PM5(SCLK)-PM0		   	via 1k register
// PS1->PS0 connected to RS232 to the PC

// This program assumes that if you can download object code then 
// the flash EEPROM and serial ports are functional.

//  This example accompanies the books
//   "Embedded Microcomputer Systems: Real Time Interfacing", Brooks-Cole, copyright (c) 2000,
//   "Introduction to Embedded Microcomputer Systems: 
//    Motorola 6811 and 6812 Simulation", Brooks-Cole, copyright (c) 2002,
//    Jonathan W. Valvano 9/8/04

// You may use, edit, run or distribute this file 
//    as long as the above copyright notices remain 
#include <hidef.h>      /* common defines and macros */
#include <mc9s12c32.h>  /* derivative information */

#include "SCIA.H"
#include "PLL.H"
#include "ADC.H"
#include "SPI.H"
#pragma LINK_INFO DERIVATIVE "mc9s12c32"

//---------Wait------------------
// fixed time delay
// inputs: time to wait in 42ns cycles
// outputs: none
void Wait(unsigned short time){ short TimeLeft,EndT;
  EndT = TCNT+time;       // Time (42ns cycles) to wait 
  do{
    TimeLeft = (short)TCNT-EndT;
  }
  while((TimeLeft<0)||(TimeLeft>100));   // wait 
}

unsigned short bRTIinterrupt;
unsigned short bTOFinterrupt;		

//---------------------TOF_Init---------------------
// enable TOF interrupts every 2.73ms
// Input: none
// Output: none
void TOF_Init(void){
  TSCR1 = 0x80;   // Enable TCNT, 4MHz in run mode
  TSCR2 = 0x80;   // divide by 1 TCNT prescale, TOI arm
  PACTL = 0;      // timer prescale used for TCNT
/* Bottom three bits of TSCR2 (PR2,PR1,PR0) determine TCNT period
    divide TCNT    at 24MHz  
000   1     42ns  TOF  2.73ms 
001   2     84ns  TOF  5.46ms  
010   4    167ns  TOF  10.9ms      
011   8    333ns  TOF  21.8ms 	 
100  16    667ns  TOF  43.7ms 
101  32   1.33us  TOF  87.4ms		
110  64   2.67us  TOF 174.8ms  
111 128   5.33us  TOF 349.5ms  */ 
}

//---------------------RTI_Init---------------------
// initialize RTI interrupts every 4096us or 244.14Hz
// inputs: none
// outputs: none
void RTI_Init(void){
  asm sei        // Make ritual atomic 
  CRGINT = 0x80; // RTIE=1 enable rti
  RTICTL = 0x33; // period=250ns*4096*4=4096us or 244.14Hz
/*  base clock is OSCCLK=250nsec
RTR[6:4] Real Time Interrupt Prescale Rate Select Bits
  000 off
  001 divide by 1024
  010 divide by 2048
  011 divide by 4096
  100 divide by 8192
  101 divide by 16384
  110 divide by 32768
  111 divide by 65536
RTR[3:0] Real Time Interrupt Modulus Counter Select Bits
 0000 divide by 1
 0001 divide by 2
 0010 divide by 3
 0011 divide by 4
 0100 divide by 5
 0101 divide by 6
 0110 divide by 7
 0111 divide by 8
*/  
}



//---------------------OutCRLF---------------------
// Output a CR,LF to SCI to go to a new line
// Input: none
// Output: none
void OutCRLF(void){
  SCI_OutChar(CR);
  SCI_OutChar(LF);
}
short Mode;		 // 1 if verbose mode, 2 if pause mode

//---------------------UpDateMode---------------------
// check to see if new serial input, if so change Mode
// Input: none
// Output: none
void UpDateMode(void){  unsigned char letter;
  if(SCI_InStatus()){
    letter = SCI_InChar();
    if(letter=='V'){		// activate verbose
      Mode = 1;
    } 
    else{
      if(letter=='P'){		// activate pause mode
        Mode = 2;
      }
      else{
        Mode = 0;
      }
    }
  }
}

unsigned char WriteData,Direction,ReadData,ExpectedData;
unsigned short ExpectedAnalogData,AnalogData,Channel;
unsigned short ErrorCount; // number of errors 
unsigned short AnyFailure; // 0 if ok, 1 if any failure
//---------------------CheckError---------------------
// Check if OK, Output results of last test if verbose mode
// Input: none
// Output: none
void CheckError(void){
  if(ReadData != ExpectedData){
    AnyFailure = 1;
    ErrorCount++;     // error if not equal
    if(Mode){
      OutCRLF();
      SCI_OutString("Direction="); SCI_OutUHex(Direction);
      SCI_OutString(" Write="); SCI_OutUHex(WriteData);
      SCI_OutString(" Expected="); SCI_OutUHex(ExpectedData);
      SCI_OutString(" Read="); SCI_OutUHex(ReadData);
    }
    if(Mode==2){
      SCI_InChar();
    }
  }
  UpDateMode();
}

//---------------------CheckAnalogError---------------------
// Check if OK, Output results of last test if verbose mode
// Input: none
// Output: none
void CheckAnalogError(void){ short data,expected,difference;
  data = AnalogData;
  expected = ExpectedAnalogData;
  difference = data-expected;
  if(difference<0){	 // absolute value
    difference = -difference; 
  }
  if(difference>20){
    ErrorCount++;     // error if not within 20 counts
    AnyFailure = 1;
    if(Mode){
      OutCRLF();
      SCI_OutString("Channel="); SCI_OutUHex(Channel);
      SCI_OutString(" Expected="); SCI_OutUHex(ExpectedAnalogData);
      SCI_OutString(" Read="); SCI_OutUHex(AnalogData);
    }
    if(Mode==2) SCI_InChar();
  }
  UpDateMode();
}

//--------------------- PORTTE test---------------------
// Tests Port T and E
// Input: none
// Output: none
void PORTTETest(void){ unsigned char i;
  SCI_OutString("PORTT test: "); 
  ErrorCount = 0;
  DDRT = 0x0F;  // least significant nibble are outputs
  Direction = 0x0F;  
  for(i=0; i<16; i++){
    PTT = i;
    WriteData = i;
    ReadData = PTT;
    ExpectedData = (i<<4)+i;
    CheckError();
    if(ErrorCount>8) break;
  }
  DDRT = 0xF0;  // most significant nibble are outputs
  Direction = 0xF0;  
  for(i=0; i<16; i++){
    PTT = i<<4;
    WriteData = i<<4;
    ReadData = PTT;
    ExpectedData = (i<<4)+i;
    CheckError();
    if(ErrorCount>8) break;
  }
  if(ErrorCount){
    SCI_OutString(" failed"); 
  } else{
    SCI_OutString("passed"); 
  }
  OutCRLF();

//---------------------PORTE test--------------------
  SCI_OutString("PORTE test: "); 
  if(ErrorCount){
    SCI_OutString("skipped, because PORTT failed"); OutCRLF();
  } else{
    DDRT = 0x01;    // PT0=PE0
    Direction = 0;  // fixed input
    for(i=0; i<1; i++){
      PTT = i;
      WriteData = i;
      ReadData = 0x01&PORTE;
      ExpectedData = i;
      CheckError();
    }
    if(ErrorCount){
      SCI_OutString(" failed"); 
    }  else{
      SCI_OutString("passed"); 
    }
  }
  OutCRLF();
}

//---------------------  PAD test ---------------------
// Tests Port AD, digital and analog test
// Input: none
// Output: none
void PADTest(void){  unsigned char bit;
  SCI_OutString("Digital PAD test: "); 
  ATDDIEN = 0xFF; // enable AD digital I/O drivers  
  ErrorCount = 0;
  
  for(bit=1; bit<=0x20; bit=bit<<1){ // 1,2,4,8,0x10,0x20
    DDRAD = bit;      // one bit is output
    Direction = bit;  
    PTAD = 0;			    // output low
    Wait(1000);       // 42us
    WriteData = 0;			
    ReadData = 0x3F&PTAD;
    ExpectedData = 0; // five inputs should be low
    CheckError();
    PTAD = bit;	      // output high
    Wait(1000);       // 42us
    WriteData = bit;	 
    ReadData = 0x3F&PTAD;
    ExpectedData = 0x3F;		 // five inputs should be high
    CheckError();
    if(ErrorCount>8) break;
  }
  for(bit=0x40; bit; bit=bit<<1){ // 0x40,0x80
    DDRAD = bit;      // one bit is output
    Direction = bit;  
    PTAD = 0;			    // output low
    Wait(1000);       // 42us
    WriteData = 0;			
    ReadData = 0xC0&PTAD;
    ExpectedData = 0; // other input should be low
    CheckError();
    PTAD = bit;	      // output high
    Wait(1000);       // 42us
    WriteData = bit;	 
    ReadData = 0xC0&PTAD;
    ExpectedData = 0xC0;		 // other input should be high
    CheckError();
    if(ErrorCount>8) break;
  }

  if(ErrorCount){
    SCI_OutString(" failed"); 
  } else{
    SCI_OutString("passed"); 
  }
  OutCRLF();
//*********** Analog PAD test **********************
  SCI_OutString("Analog PAD test: "); 
  if(ErrorCount){
    SCI_OutString("skipped, because digital PAD failed"); OutCRLF();
  } else{
    ADC_Init();     // Activate ADC 
    ATDDIEN = 0x07; // enable bottom half of AD digital I/O drivers  
    DDRAD = 0x07;   // least significant 3 bits are outputs
    Direction = 0x07; 
    for(Channel=0x83; Channel<=0x85; Channel++){
      PTAD = 0;
      Wait(1000);       // 42us
      WriteData = 0;
      AnalogData = ADC_In(Channel); 
      ExpectedAnalogData = 0;
      CheckAnalogError();
      PTAD = 1;	 // 1 high, 2 low Vin=1.25V
      Wait(1000);       // 42us
      WriteData = 1;
      AnalogData = ADC_In(Channel);
      ExpectedAnalogData = 0x154;
      CheckAnalogError();
      PTAD = 3;	 // 2 high, 1 low Vin=2.5V
      Wait(1000);       // 42us
      WriteData = 3;	
      AnalogData = ADC_In(Channel);
      ExpectedAnalogData = 0x2AB;
      CheckAnalogError();
      PTAD = 7;	 // 3 high Vin=5V
      Wait(1000);       // 42us
      WriteData = 7;	
      AnalogData = ADC_In(Channel);
      ExpectedAnalogData = 0x3FF;
      CheckAnalogError();
    }
    ATDDIEN = 0x38; // enable top half of AD digital I/O drivers  
    DDRAD = 0x38;   // bits 5,4,3 are outputs
    Direction = 0x38; 
    for(Channel=0x80; Channel<=0x82; Channel++){
      PTAD = 0;
      Wait(1000);       // 42us
      WriteData = 0;
      AnalogData = ADC_In(Channel); 
      ExpectedAnalogData = 0;
      CheckAnalogError();
      PTAD = 0x08;	    // 1 high, 2 low Vin=1.25V
      Wait(1000);       // 42us
      WriteData = 0x08;	
      AnalogData = ADC_In(Channel);
      ExpectedAnalogData = 0x154;
      CheckAnalogError();
      PTAD = 0x18;	    // 2 high, 1 low Vin=2.5V
      Wait(1000);       // 42us
      WriteData = 0x18;	 
      AnalogData = ADC_In(Channel);
      ExpectedAnalogData = 0x2AB;
      CheckAnalogError();
      PTAD = 0x38;	    // 3 high,Vin=5V
      Wait(1000);       // 42us
      WriteData = 0x38;	
      AnalogData = ADC_In(Channel);
      ExpectedAnalogData = 0x3FF;
      CheckAnalogError();
    }
    if(ErrorCount){
      SCI_OutString(" failed"); 
    } else{
      SCI_OutString("passed"); 
    }
    ADC_Stop(); 
  }
  OutCRLF();
}

//--------------------- Digital PORTM test ---------------------
// Tests Port M, digital and SPI
// Input: none
// Output: none
void PORTMTest(void){ unsigned char i;
  SCI_OutString("Digital PORTM test: "); 
  ErrorCount = 0;
  DDRM = 0x07;  // least significant 3 bits are outputs
  Direction = 0x07;  
  for(i=0; i<7; i++){
    PTM = i;
    WriteData = i;
    ReadData = PTM;
    ExpectedData = 0;
    if(i&0x04) ExpectedData |= 0x14; // bit4-bit2
    if(i&0x02) ExpectedData |= 0x0A; // bit3-bit1
    if(i&0x01) ExpectedData |= 0x21; // bit5-bit0
    CheckError();
    if(ErrorCount>8) break;
  }
  Direction = DDRM = 0x38;  // most significant 3 bits are outputs
  for(i=0; i<7; i++){
    PTM = i<<3;
    WriteData = i<<3;
    ReadData = PTM;
    ExpectedData = 0;
    if(i&0x04) ExpectedData |= 0x21; // bit5-bit0
    if(i&0x02) ExpectedData |= 0x14; // bit4-bit2
    if(i&0x01) ExpectedData |= 0x0A; // bit3-bit1
    CheckError();
    if(ErrorCount>8) break;
  }
  if(ErrorCount){
    SCI_OutString(" failed"); 
  } else{
    SCI_OutString("passed"); 
  }
  OutCRLF();
  SCI_OutString("SPI test: "); 
  if(ErrorCount){
    SCI_OutString("skipped, because digital PORTM failed"); OutCRLF();
  } else{
    Direction = 0x38; 
    SPI_Init();     // Activate SPI
    for(WriteData=0; WriteData<255; WriteData++){
      ReadData = SPI_Data(WriteData);
      ExpectedData = WriteData;
      CheckError();
      if(ErrorCount>8) break;
    }
    SPI_Stop();
    if(ErrorCount){
      SCI_OutString(" failed"); 
    } else{
      SCI_OutString("passed"); 
    }
    OutCRLF();
  }
}

//---------- RTI interrupt test-----------------
// Tests RTI interrupts
// Input: none
// Output: none
void RTITest(void){
  SCI_OutString("RTI interrupt test: "); 
  if(bRTIinterrupt){
    SCI_OutString("passed"); 
  } else{
    SCI_OutString(" failed"); 
  }
  bRTIinterrupt = 0;  // for next time through loop
  OutCRLF();
}
    
//---------- TOF interrupt test-----------------
// Tests TOF interrupts
// Input: none
// Output: none
void TOFTest(void){
  SCI_OutString("TOF interrupt test: "); 
  if(bTOFinterrupt){
    SCI_OutString("passed"); 
  } else{
    SCI_OutString(" failed"); 
  }
  bTOFinterrupt = 0;
  OutCRLF();
}

void main(void) { unsigned char letter;	 
  INTCR = 0;        // PORTE PE1 is normal input, no external IRQ 
  PLL_Init();       // running at 24MHz
  TOF_Init();				// TOF interrupts in background
  RTI_Init();       // RTI interrupts in background
  SCI_Init(115200);	// fastest standard baud rate on run mode 9S12C32
  SCI_OutString("TechArts 9S12C32 tester  9/8/04 -JWV"); OutCRLF();
  bRTIinterrupt = 0;    
  bTOFinterrupt = 0; 
  asm cli           // enable interrupts 
  for(;;) {
    AnyFailure = 0;

    SCI_OutString("Type any character, V for verbose, P for pause"); 
    letter = SCI_InChar();
    if(letter=='V'){		// activate verbose
       Mode = 1;
    } 
    else{
      if(letter=='P'){		// activate pause mode
        Mode = 2;
      }
      else{
        Mode = 0;
      }
    }
    OutCRLF();
    PORTTETest();
		PADTest();   // digital and analog
    PORTMTest(); // digital and SPI
    RTITest();
		TOFTest();
		if(AnyFailure){
      SCI_OutString("********************"); OutCRLF();
      SCI_OutString("* BOARD HAS FAILED *"); OutCRLF();
      SCI_OutString("********************"); OutCRLF();
		} else{
      SCI_OutString("++++++++++++++++++++"); OutCRLF();
      SCI_OutString("+ BOARD IS OK      +"); OutCRLF();
      SCI_OutString("++++++++++++++++++++"); OutCRLF();
		}
  }    
}

void interrupt 7 RTIhandler(){
  CRGFLG = 0x80;      // acknowledge, clear RTIF flag
  bRTIinterrupt = 1;  // check to see if RTI interrupts are occuring
}

interrupt 16 void TOFhndlr(void){
  TFLG2 = 0x80;       // acknowledge TOF 
  bTOFinterrupt = 1;  // check to see if clock interrupts are occuring 
}