;SFR04.asm
*******************************************************************************
*REVISION HISTORY:
*
*DATE			REV. NO.	DESCRIPTION
*
*August 13, 2004	1.00		Sonar Application
*
*Author: Exequiel Rarama for the 9S12C32
*******************************************************************************
;Compiled using MiniIDE
;

#include C32Regs.INC

	ORG RAM

;SFR04 Variables
PulseTime	ds	2		;10 microsecond delay
EchoTimerMax	ds	2		;Echo drive timer
RechargeTime	ds	2		;10 millisecond Recharge delay before next firing
MaxEchoTime	ds	2		;36 millisecond for MAX no echo return
EchoTime	ds	2

IC6Var1		ds	2
IC6Var2		ds	2
IC6Var3		ds	2

cap6Flag	ds	1
cap6Valid	ds	1
Send6Flag	ds	1

SFR04State	ds	2
DBUFR		ds	5


;------------------------------------------------------------------------------
FPACC1EX:	ds	1		;FLOATING POINT ACCUMULATOR #1..
FPACC1MN:	ds	3
MANTSGN1:	ds	1		;MANTISSA SIGN FOR FPACC1 (0=+, FF=-).

FPACC2EX:	ds	1		;FLOATING POINT ACCUMULATOR #2.
FPACC2MN:	ds	3
MANTSGN2:	ds	1		;MANTISSA SIGN FOR FPACC2 (0=+, FF=-).
;------------------------------------------------------------------------------
Lattitude2	ds	8		;Fix lattitude coordinates of target
Longitude1	ds	8		;Current longitude of UAV
Longitude2	ds	8		;Fix longitude coordinates of target
;
Distance	ds	5		;Distance from UAV to target
;
Vartemp1	ds	5		;Sine save
Vartemp2	ds	5		;Sine save
;
Vartemp3	ds	5		;Cosine save
Vartemp4	ds	5		;Cosine save
;

tVar		ds	5		;Floating point value
tVarTemp1	ds	5
tVarTemp2	ds	5
;
Float2Ascii	ds	15		;Conversion from Floating point to ASCII
;
;------------------------------------------------------------------------------
;General timers
dispDly		ds	2


*  Operational Parameters

OC1mask1	equ	%00000010	;IOS1 = 1, Bit 1 as output compare
OC1mask2	equ	%00000010	;C1I = 1, Enable Interrupt
OC1flag		equ	%00000010	;C1F = 1 to clear Interrupt flag

IC6mask1	equ	%01000000	;IOS6 = 1

;SFR04 parameters
MaxTravelTime	equ	300		;Maximum echo travel time
SonarPulse	equ	%00000001	;Port M bit 0 as SFR04 sonar driver

TooClose	equ	300		;Short sonar distance measurement.
					;Any value < is considered too close
TooFar		equ	8000		;Long sonar distance measurement.  
					;Any value > is considered too far

ms100		equ	$9C6
ms1		equ	$1B
ms8		equ	$CC

us100		equ	300
us10		equ	30
us5		equ	15
us1		equ	3

;------------------------------------------------------------------------------
*  Operational Parameters
RAM		equ	$3800		;RAM
STACK		equ	$3F80		;Stack at below Ubug12
FLASH		equ	$4000		;Fixed FLASH or PPAGE = $3E
VectorTable	equ	$FF80		;Beginning of Vector Table interrupt

OscFreq		equ	8000		;Enter Osc speed
initSYNR	equ	$02		; mult by synr + 1 = 3 (24MHz)
initREFDV	equ	$00		;
PLLSEL		equ	%10000000	;PLL select bit
LOCK		equ	%00001000	;lock status bit
PLLON		equ	%01000000	;phase lock loop on bit

;------------------------------------------------------------------------------
;RTI Variables
clrmask         equ     %11000000       ;mask for clearing timer flags
;RTIRate		equ     %00110000       ;[6:4]=2^12 =>512 microsecond RTI rate
RTIRate		equ     %00110000       ;[6:4]=2^12 =>512 microsecond RTI rate

RTIF		equ	%10000000
RTIE		equ	%10000000

;SCI Variables
scimask         equ     %00101100       ;RIE - SCI Interrupt enable
                                        ;RE - Receiver Enable
RDRFflag        equ     %00100000       ;RDRF - Receive Data Register Full flag
TDREflag        equ     %10000000       ;TDRE - Transmit Data Register Empty flag

;Baud rate definitions
;OscFreq		equ	8000		;Enter Osc speed
;initSYNR	equ	$02		; mult by synr + 1 = 3 (24MHz)
;initREFDV	equ	$00		;

BusFreq		equ	((OscFreq/(initREFDV+1))*(initSYNR+1))
baud115200	equ	(BusFreq/16)*10/1152  			;sets baud rate to 115,200
baud9600	equ	(BusFreq/16)*10/96  			;sets baud rate to 009,600
initSCI0CR2	equ	$0C					;SCI0 Control Register 2


***************************** Program *******************************

	ORG FLASH

ResetFunc				;This is where the RESET vector points to
	sei				;Disable Any interrupts

	movb	#$00,INITRG		;set registers at $0000 
	movb	#$39,INITRM		;move and set ram to end at $3fff 

;Initialize Stack
	lds	#STACK			;initialize stack pointer

; Initialize clock generator and PLL
	bclr	CLKSEL,PLLSEL      	;disengage PLL to system
	bset	PLLCTL,PLLON       	;turn on PLL

	movb	#initSYNR,SYNR     	;set PLL multiplier 
	movb	#initREFDV,REFDV   	;set PLL divider

	nop
	nop
	nop

	nop
	nop
	nop
	nop

	brclr	CRGFLG,LOCK,*+0    	;while (!(crg.crgflg.bit.lock==1))
	bset	CLKSEL,PLLSEL      	;engage PLL to system
	cli

;------------------------------------------------------------------------------
OC1Init					;Initialize Output Compare 1
	bset	DDRM,SonarPulse		;Bit 0 output the rest inputs
	bclr	PORTM,SonarPulse	;Init Port M

	bset	TSCR1,%10000000		;TEN=1 - Enable timer
	bset	TIOS,OC1mask1		;Set IOS1 for Output Compare
	bclr	TCTL2,%1100		;Make sure Port is not tied with OC

	bset	TSCR2,%0010		;Bus clock/4 = 24MHz/4 = 6MHz
;	bset	TSCR2,%0011		;Bus clock/8 = 24MHz/8 = 3MHz
;	bset	TSCR2,%0100		;Bus clock/16 = 24MHz/16 = 1.5MHz

	movb	#OC1flag+IC6mask1,TFLG1	;clear flag;
	bset	TIE,IC6mask1		;enable IC6 interrupt

;Change capture edge to capture from Low to high of IC6
	ldaa	TCTL3
	anda	#%11001111		;masked lower 2 bits
	adda	#%00010000
	staa	TCTL3

	clr	cap6Flag
	clr	cap6Valid
	clr	Send6Flag

	movw	#00,RechargeTime		;Initialize recharge time
	movw	#SendPulse,SFR04State		;Initialize SFR04State

	ldd	#us10				;get constant for 10us delay
	std	PulseTime			;PulseTime = 10*10^-3 * 3 * 10^6 = 30

;------------------------------------------------------------------------------
RealTimeInit				;Initialize Real Time Interrupt
	movb	#RTIRate,RTICTL		;and initialize RTI rate
	bset	CRGFLG,RTIF		;clear flag
	bset	CRGINT,RTIE		;Enable RTI

;Initialize Analog To Digital
	movb	#$80,ATDCTL2	        ;enable ATD
	movb	#$40,ATDCTL3	        ;8 Channels ATD result
	movb	#$60,ATDCTL4		;Select Sample rate
	movb	#$B0,ATDCTL5		;Select 8 channel mode, Continuous scan

;Initialize first Serial Communication Interface
	movw	#baud9600,SCIBDH	;Set baud rate to 9600
	movb	#scimask,SCICR2		;Rx and Tx on
	ldab	SCISR1  		;read register to clear flag RDRF
	ldab	SCIDRL			;read receive buffer

	ldx	#MSG1
        jsr     OutStr0

	movw	#00,dispDly

;========START of MAIN==================================================================
main						
	ldx	SFR04State		;Process Sonar state
	jsr	0,x

	ldx	dispDly			;check the display delay
	bne	main

	jsr	CalDistance
	movw	#122,dispDly		;1 second delay refresh

	bra	main

;---------------TEST--------------
	ldx	#IC6Var3
	jsr	HTOD

	ldaa	#'-'
	jsr	SerOutput0

	ldx	#IC6Var1
	jsr	HTOD

	ldaa	#'='
	jsr	SerOutput0

	ldx	#IC6Var2
	jsr	HTOD

	ldaa	#$D
	jsr	SerOutput0

	ldaa	#$A
	jsr	SerOutput0

	bra	main

;-------------END TEST--------------


;=======END of MAIN====================================================================


;------------------------------------------------------------------------------------
SendPulse
	ldx	RechargeTime		;Check recharge time delay
	bne	SendPulseEx

	ldaa	Send6Flag		;Will not send another Impulse until flag is clear
	bne	SendPulseCheckTimer

;Change capture edge to capture from Low to high
	ldaa	TCTL3
	anda	#%11001111		;masked lower 2 bits
	adda	#%00010000
	staa	TCTL3

	clr	cap6Flag
	movb	#1,Send6Flag
	movw	#59,EchoTimerMax	;Initialize echo drive time limit

	movb	#OC1flag+IC6mask1,TFLG1		;clear flag;

	bset	PORTM,SonarPulse	;Port M bit 0 = 1 will drive Sonar
	ldd	PulseTime		;get constant for 10us delay
	addd	TCNT
	std	TC1

	bset	TIE,OC1mask2		;enable IC6, OC1 interrupt
	bra	SendPulseEx

SendPulseCheckTimer			;Check to see if waited too long in firing
	ldx	EchoTimerMax
	bne	SendPulseEx

	clr	Send6Flag
	bclr	PORTM,SonarPulse	;Port M bit 0 = 0 Sonar driver off
	movw	#20,RechargeTime	;Initialize recharge time

SendPulseEx
	rts

;------------------------------------------------------------------------------
; SFR04 OC1 Interrupt Routine
;----------------------------

SFR04OC1Int
	bclr	PORTM,SonarPulse	;Port M bit 0 = 0 Sonar driver off
	movw	#20,RechargeTime	;Initialize recharge time

	movb	#OC1flag,TFLG1		;clear flag;

	ldaa	TIE
	anda	#%11111101
	staa	TIE			;disable OC1 interrupt
	rti

;------------------------------------------------------------------------------
; SFR04 IC6 Interrupt Routine
;----------------------------

SFR04IC6Int
	ldd	TC6
	pshd				;Save momentarily

	ldaa	cap6Flag		;Check if 1st or 2nd capture
	bne	Second6Cap		;If flag is set then it is 2nd falling edge

	puld
	std	IC6Var1			;1st Capture
	movb	#1,cap6Flag		;Set flag for 2nd capture

;Change capture edge to capture from high to low
	ldaa	TCTL3
	anda	#%11001111		;masked lower 2 bits
	adda	#%00100000
	staa	TCTL3
	bra	IC6ex			;Wait until next capture

Second6Cap
	puld
	std	IC6Var3
	subd	IC6Var1			;Subtract from 1st Capture
;	lsrd				;Divide 2
	std	IC6Var2			;Result in Var2

	clr	cap6Flag
	clr	Send6Flag		;A pulse is received, flag is initialized for a new pulse

	movw	#20,RechargeTime	;Initialize recharge time to 10 millisecond
	movb	#1,cap6Valid		;set flag for echo

;Change capture edge to capture from Low to high
	ldaa	TCTL3
	anda	#%11001111		;masked lower 2 bits
	adda	#%00010000
	staa	TCTL3

IC6ex
	movb	#%01000000,TFLG1        ;clear input interrupt flag
        rti

;------------------------------------------------------------------------------
* Real-time Interrupt Routine

RealTimeInt
	bset	CRGFLG,RTIF		;clear real-time interrupt flag
	cli				; so that other interrupts can be service

RTI_6	
	ldx	dispDly
	beq	RTI_7
	dex
	stx	dispDly

RTI_7
	ldx	RechargeTime
	beq	RTI_8
	dex
	stx	RechargeTime

RTI_8
	ldaa	cap6Flag
	beq	RTI_9

	ldx	EchoTime
	inx	
	stx	EchoTime

RTI_9
	ldx	EchoTimerMax
	beq	RTI_10
	dex
	stx	EchoTimerMax

RTI_10
	rti

;==============================================================================
;-----------------------------------------------------------------------------
;HTOD-SUBROUTINE TO CONVERT A 16-BIT HEX NUMBER TO A 5 DIGIT DECIMAL
;This routine is taken from the HC11 manual.
;D=HEX VALUE TO BE CONVERTED
;X=ADDRESS WHERE THE NUMBER IS STORED TO BE CONVERTED

HTOD
        LDD     0,X             ;
	LDX     #10000
        IDIV                    ;
	XGDX
	ADDB    #$30
	STAB    DBUFR
	XGDX
	LDX     #1000
	IDIV
	XGDX
	ADDB    #$30
	STAB    DBUFR+1
	XGDX
	LDX     #100
	IDIV
	XGDX
	ADDB    #$30
	STAB    DBUFR+2
	XGDX
	LDX     #10
	IDIV
	ADDB    #$30
	STAB    DBUFR+4
	XGDX
	ADDB    #$30
	STAB    DBUFR+3
        LDX     #DBUFR+1        ;POINT AT DECIMAL

;        BRA     P1K

P5DEC
        LDX     #DBUFR          ;POINT AT DECIMAL
	LDAA    #$30            ;CHECK FOR LEADING ZEROS
	CMPA    0,X             ;CHECK FOR 10,000S DIGIT

	BNE     P10K            ;START AT 10K DIGIT
        BSR     SKP1            ;INX AND PRINT A SPACE
	CMPA    0,X             ;CHECK FOR 1,000S

	BNE     P1K             ;START AT 1K DIGIT
        BSR     SKP1
        BSR     SKP1
	DEX
	CMPA    0,X             ;CHECK FOR 100S DIGIT

	BNE     P100            ;START AT 100 DIGIT
        BSR     SKP1
	CMPA    0,X             ;CHECK 10S DIGIT

	BNE     P10
        BSR     SKP1
	BRA     P1              ;START AT 1S DIGIT

P10K    LDAA    0,X             ;10,000 DIGIT
	PSHX
        jsr     SerOutput0
        PULX
	INX

P1K     LDAA    0,X
	PSHX
        jsr     SerOutput0
        PULX
	INX

P100    LDAA    0,X
	PSHX
        jsr     SerOutput0
	PULX
	INX

P10     LDAA    0,X
	PSHX
        jsr     SerOutput0
        PULX
	INX

P1      LDAA    0,X
        jsr     SerOutput0
        RTS

SKP1    PSHA
	INX
	LDAA    #$20
        jsr     SerOutput0
	PULA
	RTS


;------------------------------------------------------------------------------
*  SCI Input Interrupt Handler

*  Gets bytes from SCI.  Sets COMMAND_PENDING flag.


OutStr0				; send a null terminated string to the display.
	ldaa	1,x+            ; get a character, advance pointer, null?
	beq	OutStrDone      ; yes. return.
	bsr	SerOutput0      ; no. send it out the SCI.
	bra	OutStr0          ; go get the next character.
;
OutStrDone
	rts

;------------------------------------------------------------------------------
SerOutput0
	brclr	SCISR1,TDREflag,SerOutput0     ;check if buffer is empty
	staa	SCIDRL
	rts

SerInputInt0
	ldaa	SCISR1                  ;read register to clear flag RDRF
	ldaa	SCIDRL			;read receive buffer
	rti

;------------------------------------------------------------------------------
* Messages

MSG1	dc.b	'9S12C32 Sonar Demo V1.00',$D,$A,0 


;--------------------------INCLUDE FILES START HERE----------------------------

#include Math.Asm

;--------------------------INCLUDE FILES END HERE------------------------------

CalDistance
	ldab	#5			;Copy Float speed of sound into FPACC1
	ldx	#SoundSpeed
	ldy	#FPACC1EX
	jsr	Datacopy

	ldab	#5			;Copy Float 6MHz into FPACC1
	ldx	#Freq6M
	ldy	#FPACC2EX
	jsr	Datacopy

	jsr	FLTDIV			;Speed of sound / 6MHz

	ldab	#5			;Save results into tVar
	ldx	#FPACC1EX
	ldy	#tVar
	jsr	Datacopy

	ldd	IC6Var2
	lsrd				;Divide by 2
	std	MANTSGN1-2		;16 bit integer to be converted
	jsr	UINT2FLT
	
	ldab	#5			;Copy tVar into FPACC2
	ldx	#tVar
	ldy	#FPACC2EX
	jsr	Datacopy

	jsr	FLTMUL			;Result is the distance

	ldx	#Float2Ascii		;Convert from Fload to ASCII
	jsr	FLTASC			;

	ldx	#Float2Ascii		;Display result
	jsr	OutStr0

	ldx	#mpers
	jsr	OutStr0

	rts


mpers	dc.b	' meters    ', $D, $0
mDist	dc.b	'Distance = ', $D, $0

;------------------------------------------------------------------------------------------
Datacopy
	movb	1,x+,1,y+		;temporary save
	dbne	b,Datacopy
	rts

;---------------------------------------------------------------------------	
SoundSpeed	dc.b	$89, $A5, $B3, $33, $00	;Speed of Sound = 331.40000
Freq6M		dc.b	$97, $B7, $1B, $00, $00	;6MHz
Freq3M		dc.b	$96, $B7, $1B, $00, $00	;3MHz
Freq1M5		dc.b	$95, $B7, $1B, $00, $00	;1.5MHz

AD1024		dc.b	$8B, $80, $00, $00, $00	;10 bit A/D
Vref5V		dc.b	$83, $A0, $00, $00, $00	;Vref = 5.00000 volts
VrefDiv1024	dc.b	$79, $A0, $00, $00, $00	;Vref/1024 = 5/1024


;==============================================================================
	ORG	VectorTable		;Definition of Vector tables
	dc.w	ResetFunc		;Reserve
	dc.w	ResetFunc		;Reserve
	dc.w	ResetFunc		;Reserve
	dc.w	ResetFunc		;Reserve

	dc.w	ResetFunc		;PWM Emergency Shutdown
	dc.w	ResetFunc		;VREG LVI
	dc.w	ResetFunc		;Port P
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved

	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;CAN transmit
	dc.w	ResetFunc		;CAN receive
	dc.w	ResetFunc		;CAN errors
	dc.w	ResetFunc		;CAN wake-up
	dc.w	ResetFunc		;FLASH

	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserve
	dc.w	ResetFunc		;Reserve

	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved

	dc.w	ResetFunc		;CRG Self Clock Mode
	dc.w	ResetFunc		;CRG PLL lock
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;Reserved

	dc.w	ResetFunc		;Port J (PIEP)
	dc.w	ResetFunc		;Reserved
	dc.w	ResetFunc		;ATD (ATDCTL2 - ASCIE)
	dc.w	ResetFunc		;Reserved
	dc.w	SerInputInt0		;SCI
	dc.w	ResetFunc		;SPI
	dc.w	ResetFunc		;Pulse Accumulator 0 input edge
	dc.w	ResetFunc		;Pulse Accumulator 0 overflow
	dc.w	ResetFunc		;Standard Timer 0 Overflow
	dc.w	ResetFunc		;Timer 0 Channel 7
	dc.w	SFR04IC6Int		;Timer 0 Channel 6
	dc.w	ResetFunc		;Timer 0 Channel 5
	dc.w	ResetFunc		;Timer 0 Channel 4

	dc.w	ResetFunc		;Timer 0 Channel 3
	dc.w	ResetFunc		;Timer 0 Channel 2
	dc.w	SFR04OC1Int		;Timer 0 Channel 1
	dc.w	ResetFunc		;Timer 0 Channel 0

	dc.w	RealTimeInt		;Real Time Interrupt
	dc.w	ResetFunc		;IRQ
	dc.w	ResetFunc		;XIRQ
	dc.w	ResetFunc		;SWI
	dc.w	ResetFunc		;Instruction Trap
	dc.w	ResetFunc		;COP failure
	dc.w	ResetFunc		;Clock Monitor
	dc.w	ResetFunc		;Power On Reset

